<?php
/**
 * أداة تنظيف النظام
 * 
 * هذه الأداة تقوم بتنظيف النظام من الملفات المكررة والغير مستخدمة
 * وتحسين هيكل المجلدات وتوحيد قاعدة البيانات
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// قائمة الملفات المكررة التي يجب حذفها
$duplicate_files = [
    // ملفات الاختبار
    'test.php',
    'test_connection.php',
    'test_db.php',
    'test_db_connection.php',
    'test_db_connection_updated.php',
    'test_login.php',
    'test_order_number.php',
    'test-login.php',
    'test-register.php',
    'debug_register.php',
    'debug_session.php',
    
    // ملفات تسجيل الدخول المكررة
    'login-test.php',
    'login-test-new.php',
    'register-simple.php',
    'simple-register.php',
    
    // ملفات التكوين المكررة
    'subdomain_config.php',
    'shipping/config/subdomain_config.php',
    'shipping/config/deployment.php',
    'config/deployment.php',
    'deploy_config.php',
    
    // ملفات قاعدة البيانات المكررة
    'shipping/database/create_orders_table.sql',
    'shipping/database/create_users_table.sql',
    'shipping/database/create_sqlite_tables.sql',
    'database/create_branches_table.sql',
    'database/create_countries_table.sql',
    'database/create_orders_table.sql',
    'database/fix_users_table.sql',
    
    // ملفات تحويل SVG المكررة
    'convert_svg_simple.py',
    'convert_svg_to_png.py',
    
    // ملفات HTML المكررة
    'shipping/index.html',
    'templates/index.html',
    'shipping/templates/index.html',
    
    // ملفات PHP المكررة
    'shipping/index.php',
    'public_html/index.php',
    'database/test_connection.php',
    
    // لوحات التحكم القديمة
    'dashboard.php',
    'admin/dashboard.php',
    'shipping/dashboard.php',
    'control-panel.php',
    'admin-panel.php',
    'user-panel.php'
];

// المجلدات المكررة التي يجب حذفها
$duplicate_directories = [
    'DWREF',
    'shipping/DWREF',
    'old_templates',
    'shipping/old_templates',
    'test_files',
    'shipping/test_files'
];

// دالة لحذف ملف
function deleteFile($file_path) {
    global $base_path;
    
    $full_path = $base_path . '/' . $file_path;
    
    if (file_exists($full_path)) {
        if (unlink($full_path)) {
            echo "تم حذف الملف: $file_path<br>";
            return true;
        } else {
            echo "فشل حذف الملف: $file_path<br>";
            return false;
        }
    } else {
        echo "الملف غير موجود: $file_path<br>";
        return false;
    }
}

// دالة لحذف مجلد
function deleteDirectory($dir_path) {
    global $base_path;
    
    $full_path = $base_path . '/' . $dir_path;
    
    if (!file_exists($full_path)) {
        echo "المجلد غير موجود: $dir_path<br>";
        return false;
    }
    
    // حذف جميع الملفات والمجلدات الفرعية
    $files = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($full_path, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::CHILD_FIRST
    );
    
    foreach ($files as $file) {
        if ($file->isDir()) {
            rmdir($file->getRealPath());
        } else {
            unlink($file->getRealPath());
        }
    }
    
    // حذف المجلد الرئيسي
    if (rmdir($full_path)) {
        echo "تم حذف المجلد: $dir_path<br>";
        return true;
    } else {
        echo "فشل حذف المجلد: $dir_path<br>";
        return false;
    }
}

// دالة للتحقق من وجود ملف
function fileExists($file_path) {
    global $base_path;
    
    $full_path = $base_path . '/' . $file_path;
    
    return file_exists($full_path);
}

// دالة للتحقق من وجود مجلد
function directoryExists($dir_path) {
    global $base_path;
    
    $full_path = $base_path . '/' . $dir_path;
    
    return is_dir($full_path);
}

// دالة لإنشاء مجلد إذا لم يكن موجودًا
function createDirectoryIfNotExists($dir_path) {
    global $base_path;
    
    $full_path = $base_path . '/' . $dir_path;
    
    if (!is_dir($full_path)) {
        if (mkdir($full_path, 0755, true)) {
            echo "تم إنشاء المجلد: $dir_path<br>";
            return true;
        } else {
            echo "فشل إنشاء المجلد: $dir_path<br>";
            return false;
        }
    }
    
    return true;
}

// دالة لنقل ملف
function moveFile($source_path, $dest_path) {
    global $base_path;
    
    $full_source_path = $base_path . '/' . $source_path;
    $full_dest_path = $base_path . '/' . $dest_path;
    
    // التأكد من وجود المجلد الوجهة
    $dest_dir = dirname($full_dest_path);
    if (!is_dir($dest_dir)) {
        mkdir($dest_dir, 0755, true);
    }
    
    if (file_exists($full_source_path)) {
        if (rename($full_source_path, $full_dest_path)) {
            echo "تم نقل الملف من: $source_path إلى: $dest_path<br>";
            return true;
        } else {
            echo "فشل نقل الملف من: $source_path إلى: $dest_path<br>";
            return false;
        }
    } else {
        echo "الملف المصدر غير موجود: $source_path<br>";
        return false;
    }
}

// دالة لإنشاء ملف سجل التنظيف
function createCleanupLog($deleted_files, $deleted_directories, $moved_files) {
    global $base_path;
    
    $log_content = "# تقرير تنظيف النظام\n\n";
    $log_content .= "تاريخ التنظيف: " . date('Y-m-d H:i:s') . "\n\n";
    
    $log_content .= "## الملفات التي تم حذفها\n\n";
    foreach ($deleted_files as $file) {
        $log_content .= "- $file\n";
    }
    
    $log_content .= "\n## المجلدات التي تم حذفها\n\n";
    foreach ($deleted_directories as $dir) {
        $log_content .= "- $dir\n";
    }
    
    $log_content .= "\n## الملفات التي تم نقلها\n\n";
    foreach ($moved_files as $move) {
        $log_content .= "- من: {$move['source']} إلى: {$move['dest']}\n";
    }
    
    $log_file = $base_path . '/logs/cleanup_' . date('Y-m-d_H-i-s') . '.log';
    
    // التأكد من وجود مجلد السجلات
    if (!is_dir($base_path . '/logs')) {
        mkdir($base_path . '/logs', 0755, true);
    }
    
    file_put_contents($log_file, $log_content);
    
    echo "تم إنشاء ملف سجل التنظيف: $log_file<br>";
}

// إنشاء المجلدات الضرورية
$required_directories = [
    'logs',
    'config',
    'database',
    'includes',
    'assets/css',
    'assets/js',
    'assets/images',
    'assets/fonts',
    'api',
    'utils'
];

foreach ($required_directories as $dir) {
    createDirectoryIfNotExists($dir);
}

// حذف الملفات المكررة
$deleted_files = [];
foreach ($duplicate_files as $file) {
    if (fileExists($file)) {
        if (deleteFile($file)) {
            $deleted_files[] = $file;
        }
    }
}

// حذف المجلدات المكررة
$deleted_directories = [];
foreach ($duplicate_directories as $dir) {
    if (directoryExists($dir)) {
        if (deleteDirectory($dir)) {
            $deleted_directories[] = $dir;
        }
    }
}

// نقل الملفات إلى المجلدات المناسبة
$moved_files = [];

// إنشاء ملف سجل التنظيف
createCleanupLog($deleted_files, $deleted_directories, $moved_files);

echo "<h2>تم الانتهاء من تنظيف النظام</h2>";
echo "<p>تم حذف " . count($deleted_files) . " ملف و " . count($deleted_directories) . " مجلد.</p>";
echo "<p>تم نقل " . count($moved_files) . " ملف.</p>";
echo "<p>يمكنك الاطلاع على سجل التنظيف في مجلد logs.</p>";
