<?php
/**
 * أداة التحقق من صحة النظام
 * 
 * هذه الأداة تقوم بالتحقق من صحة النظام وإصلاح المشاكل المحتملة
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// تضمين ملف الاتصال بقاعدة البيانات
require_once $base_path . '/config/db_connection.php';

// دالة للتحقق من إصدار PHP
function checkPhpVersion() {
    $required_version = '7.4.0';
    $current_version = PHP_VERSION;
    
    if (version_compare($current_version, $required_version, '>=')) {
        return [
            'status' => 'success',
            'message' => "إصدار PHP الحالي ($current_version) متوافق مع النظام"
        ];
    } else {
        return [
            'status' => 'error',
            'message' => "إصدار PHP الحالي ($current_version) غير متوافق مع النظام. الإصدار المطلوب هو $required_version أو أعلى"
        ];
    }
}

// دالة للتحقق من الإضافات المطلوبة
function checkRequiredExtensions() {
    $required_extensions = [
        'pdo',
        'pdo_mysql',
        'mbstring',
        'json',
        'gd',
        'xml',
        'curl',
        'zip'
    ];
    
    $missing_extensions = [];
    
    foreach ($required_extensions as $extension) {
        if (!extension_loaded($extension)) {
            $missing_extensions[] = $extension;
        }
    }
    
    if (empty($missing_extensions)) {
        return [
            'status' => 'success',
            'message' => 'جميع الإضافات المطلوبة متوفرة'
        ];
    } else {
        return [
            'status' => 'error',
            'message' => 'الإضافات التالية غير متوفرة: ' . implode(', ', $missing_extensions)
        ];
    }
}

// دالة للتحقق من صلاحيات المجلدات
function checkDirectoryPermissions() {
    global $base_path;
    
    $directories = [
        'logs',
        'uploads',
        'backups',
        'cache',
        'assets/images',
        'assets/css',
        'assets/js'
    ];
    
    $results = [];
    
    foreach ($directories as $directory) {
        $dir_path = $base_path . '/' . $directory;
        
        // التحقق من وجود المجلد
        if (!file_exists($dir_path)) {
            // إنشاء المجلد إذا لم يكن موجودًا
            if (mkdir($dir_path, 0755, true)) {
                $results[] = [
                    'directory' => $directory,
                    'status' => 'success',
                    'message' => "تم إنشاء المجلد $directory بنجاح"
                ];
            } else {
                $results[] = [
                    'directory' => $directory,
                    'status' => 'error',
                    'message' => "فشل إنشاء المجلد $directory"
                ];
                continue;
            }
        }
        
        // التحقق من صلاحيات المجلد
        if (is_writable($dir_path)) {
            $results[] = [
                'directory' => $directory,
                'status' => 'success',
                'message' => "المجلد $directory قابل للكتابة"
            ];
        } else {
            $results[] = [
                'directory' => $directory,
                'status' => 'error',
                'message' => "المجلد $directory غير قابل للكتابة"
            ];
        }
    }
    
    return $results;
}

// دالة للتحقق من وجود الملفات الأساسية
function checkRequiredFiles() {
    global $base_path;
    
    $required_files = [
        'config/config.php',
        'config/db_config.php',
        'config/db_connection.php',
        'includes/common.php',
        'includes/helpers.php',
        'includes/auth.php',
        'ASDRF43/unified-dashboard.php',
        'auth/login.php',
        'auth/logout.php',
        'orders/orders.php',
        'reports/weekly_reports.php',
        'reports/monthly_reports.php',
        'api/reports.php',
        'api/orders.php'
    ];
    
    $missing_files = [];
    
    foreach ($required_files as $file) {
        if (!file_exists($base_path . '/' . $file)) {
            $missing_files[] = $file;
        }
    }
    
    if (empty($missing_files)) {
        return [
            'status' => 'success',
            'message' => 'جميع الملفات الأساسية موجودة'
        ];
    } else {
        return [
            'status' => 'error',
            'message' => 'الملفات التالية غير موجودة: ' . implode(', ', $missing_files)
        ];
    }
}

// دالة للتحقق من الاتصال بقاعدة البيانات
function checkDatabaseConnection() {
    global $conn;
    
    if (!$conn) {
        return [
            'status' => 'error',
            'message' => 'فشل الاتصال بقاعدة البيانات'
        ];
    }
    
    try {
        // محاولة تنفيذ استعلام بسيط
        $stmt = $conn->query("SELECT 1");
        
        return [
            'status' => 'success',
            'message' => 'تم الاتصال بقاعدة البيانات بنجاح'
        ];
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل الاتصال بقاعدة البيانات: ' . $e->getMessage()
        ];
    }
}

// دالة للتحقق من وجود الجداول المطلوبة
function checkRequiredTables() {
    global $conn;
    
    if (!$conn) {
        return [
            'status' => 'error',
            'message' => 'فشل الاتصال بقاعدة البيانات'
        ];
    }
    
    $required_tables = [
        'users',
        'orders',
        'branches',
        'activity_log'
    ];
    
    $missing_tables = [];
    
    try {
        // الحصول على قائمة الجداول
        $stmt = $conn->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        foreach ($required_tables as $table) {
            if (!in_array($table, $tables)) {
                $missing_tables[] = $table;
            }
        }
        
        if (empty($missing_tables)) {
            return [
                'status' => 'success',
                'message' => 'جميع الجداول المطلوبة موجودة'
            ];
        } else {
            return [
                'status' => 'error',
                'message' => 'الجداول التالية غير موجودة: ' . implode(', ', $missing_tables)
            ];
        }
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل التحقق من وجود الجداول: ' . $e->getMessage()
        ];
    }
}

// دالة للتحقق من وجود مستخدم مدير
function checkAdminUser() {
    global $conn;
    
    if (!$conn) {
        return [
            'status' => 'error',
            'message' => 'فشل الاتصال بقاعدة البيانات'
        ];
    }
    
    try {
        // البحث عن مستخدم مدير
        $stmt = $conn->query("SELECT * FROM users WHERE role = 'admin'");
        $admin_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (count($admin_users) > 0) {
            return [
                'status' => 'success',
                'message' => 'يوجد مستخدم مدير في النظام'
            ];
        } else {
            return [
                'status' => 'warning',
                'message' => 'لا يوجد مستخدم مدير في النظام'
            ];
        }
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل التحقق من وجود مستخدم مدير: ' . $e->getMessage()
        ];
    }
}

// دالة للتحقق من وجود فروع
function checkBranches() {
    global $conn;
    
    if (!$conn) {
        return [
            'status' => 'error',
            'message' => 'فشل الاتصال بقاعدة البيانات'
        ];
    }
    
    try {
        // البحث عن فروع
        $stmt = $conn->query("SELECT COUNT(*) FROM branches");
        $branches_count = $stmt->fetchColumn();
        
        if ($branches_count > 0) {
            return [
                'status' => 'success',
                'message' => "يوجد $branches_count فرع في النظام"
            ];
        } else {
            return [
                'status' => 'warning',
                'message' => 'لا يوجد فروع في النظام'
            ];
        }
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل التحقق من وجود فروع: ' . $e->getMessage()
        ];
    }
}

// دالة للتحقق من وجود طلبات
function checkOrders() {
    global $conn;
    
    if (!$conn) {
        return [
            'status' => 'error',
            'message' => 'فشل الاتصال بقاعدة البيانات'
        ];
    }
    
    try {
        // البحث عن طلبات
        $stmt = $conn->query("SELECT COUNT(*) FROM orders");
        $orders_count = $stmt->fetchColumn();
        
        return [
            'status' => 'success',
            'message' => "يوجد $orders_count طلب في النظام"
        ];
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل التحقق من وجود طلبات: ' . $e->getMessage()
        ];
    }
}

// دالة لإنشاء تقرير بنتائج الفحص
function generateReport($results) {
    $report = "# تقرير فحص النظام\n\n";
    $report .= "تاريخ الفحص: " . date('Y-m-d H:i:s') . "\n\n";
    
    foreach ($results as $key => $result) {
        $report .= "## $key\n\n";
        
        if (is_array($result) && isset($result['status'])) {
            $status_icon = $result['status'] === 'success' ? '✅' : ($result['status'] === 'warning' ? '⚠️' : '❌');
            $report .= "$status_icon {$result['message']}\n\n";
        } else if (is_array($result) && is_array($result[0])) {
            foreach ($result as $item) {
                $status_icon = $item['status'] === 'success' ? '✅' : ($item['status'] === 'warning' ? '⚠️' : '❌');
                $report .= "$status_icon {$item['message']}\n";
            }
            $report .= "\n";
        }
    }
    
    return $report;
}

// تنفيذ الفحوصات
$results = [
    'إصدار PHP' => checkPhpVersion(),
    'الإضافات المطلوبة' => checkRequiredExtensions(),
    'صلاحيات المجلدات' => checkDirectoryPermissions(),
    'الملفات الأساسية' => checkRequiredFiles(),
    'الاتصال بقاعدة البيانات' => checkDatabaseConnection(),
    'الجداول المطلوبة' => checkRequiredTables(),
    'مستخدم مدير' => checkAdminUser(),
    'الفروع' => checkBranches(),
    'الطلبات' => checkOrders()
];

// إنشاء تقرير بنتائج الفحص
$report = generateReport($results);

// حفظ التقرير في ملف
$report_file = $base_path . '/logs/system_check_' . date('Y-m-d_H-i-s') . '.log';
file_put_contents($report_file, $report);

// عرض النتائج
echo "<h1>نتائج فحص النظام</h1>";

foreach ($results as $key => $result) {
    echo "<h2>$key</h2>";
    
    if (is_array($result) && isset($result['status'])) {
        $status_class = $result['status'] === 'success' ? 'success' : ($result['status'] === 'warning' ? 'warning' : 'error');
        echo "<p class='$status_class'>{$result['message']}</p>";
    } else if (is_array($result) && is_array($result[0])) {
        echo "<ul>";
        foreach ($result as $item) {
            $status_class = $item['status'] === 'success' ? 'success' : ($item['status'] === 'warning' ? 'warning' : 'error');
            echo "<li class='$status_class'>{$item['message']}</li>";
        }
        echo "</ul>";
    }
}

echo "<p>تم حفظ التقرير في: $report_file</p>";

// تنسيق الصفحة
?>
<style>
    body {
        font-family: Arial, sans-serif;
        line-height: 1.6;
        margin: 0;
        padding: 20px;
        direction: rtl;
    }
    h1 {
        color: #333;
        border-bottom: 2px solid #eee;
        padding-bottom: 10px;
    }
    h2 {
        color: #444;
        margin-top: 20px;
    }
    ul {
        list-style-type: none;
        padding: 0;
    }
    li {
        margin-bottom: 5px;
        padding: 5px 10px;
        border-radius: 3px;
    }
    .success {
        color: #155724;
        background-color: #d4edda;
        border-left: 3px solid #28a745;
        padding: 10px;
        margin-bottom: 10px;
    }
    .warning {
        color: #856404;
        background-color: #fff3cd;
        border-left: 3px solid #ffc107;
        padding: 10px;
        margin-bottom: 10px;
    }
    .error {
        color: #721c24;
        background-color: #f8d7da;
        border-left: 3px solid #dc3545;
        padding: 10px;
        margin-bottom: 10px;
    }
</style>
