<?php
/**
 * أداة تحسين أمان النظام
 * 
 * هذه الأداة تقوم بتحسين أمان النظام وحمايته من الهجمات المحتملة
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// دالة لإنشاء ملف .htaccess
function createHtaccessFile($base_path) {
    $htaccess_content = <<<EOT
# تحسين أمان النظام
# تم إنشاؤه بواسطة أداة تحسين أمان النظام

# منع عرض محتويات المجلدات
Options -Indexes

# منع الوصول إلى ملفات معينة
<FilesMatch "^(\.htaccess|\.htpasswd|\.git|\.env|\.gitignore|composer\.json|composer\.lock|package\.json|package-lock\.json|README\.md|LICENSE)$">
    Order Allow,Deny
    Deny from all
</FilesMatch>

# منع الوصول إلى مجلدات معينة
<DirectoryMatch "^/.git/">
    Order Allow,Deny
    Deny from all
</DirectoryMatch>

# حماية من هجمات XSS
<IfModule mod_headers.c>
    Header set X-XSS-Protection "1; mode=block"
    Header set X-Content-Type-Options "nosniff"
    Header set X-Frame-Options "SAMEORIGIN"
    Header set Referrer-Policy "strict-origin-when-cross-origin"
    Header set Content-Security-Policy "default-src 'self'; script-src 'self' 'unsafe-inline' 'unsafe-eval' https://cdn.jsdelivr.net https://cdnjs.cloudflare.com; style-src 'self' 'unsafe-inline' https://cdn.jsdelivr.net https://cdnjs.cloudflare.com https://fonts.googleapis.com; font-src 'self' https://fonts.gstatic.com https://cdnjs.cloudflare.com; img-src 'self' data:; connect-src 'self'"
</IfModule>

# تعيين نوع MIME الافتراضي
DefaultType text/html

# تعيين ترميز افتراضي
AddDefaultCharset UTF-8

# تفعيل إعادة الكتابة
RewriteEngine On

# إعادة توجيه HTTP إلى HTTPS (إذا كان متاحًا)
# RewriteCond %{HTTPS} off
# RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]

# منع الوصول إلى ملفات PHP في مجلد uploads
<Directory "$base_path/uploads">
    <Files "*.php">
        Order Allow,Deny
        Deny from all
    </Files>
</Directory>

# منع الوصول إلى ملفات PHP في مجلد logs
<Directory "$base_path/logs">
    Order Allow,Deny
    Deny from all
</Directory>

# منع الوصول إلى ملفات PHP في مجلد database
<Directory "$base_path/database">
    Order Allow,Deny
    Deny from all
</Directory>

# منع الوصول إلى ملفات PHP في مجلد config
<Directory "$base_path/config">
    Order Allow,Deny
    Deny from all
</Directory>

# منع الوصول إلى ملفات PHP في مجلد utils
<Directory "$base_path/utils">
    Order Allow,Deny
    Deny from all
</Directory>

# تعيين صفحة الخطأ 404
ErrorDocument 404 /404.php

# تعيين صفحة الخطأ 403
ErrorDocument 403 /403.php

# تعيين صفحة الخطأ 500
ErrorDocument 500 /500.php

# منع تنفيذ الأوامر عن بعد
php_flag allow_url_fopen off
php_flag allow_url_include off

# تعطيل عرض الأخطاء
php_flag display_errors off
php_flag display_startup_errors off

# تعطيل تنفيذ الأكواد الخطيرة
php_flag enable_dl off

# تعطيل تسجيل الأخطاء في ملف
php_flag log_errors on
php_value error_log logs/error.log

# تعيين الحد الأقصى لحجم الملفات المرفوعة
php_value upload_max_filesize 10M
php_value post_max_size 10M

# تعيين الحد الأقصى لوقت التنفيذ
php_value max_execution_time 30
php_value max_input_time 60

# تعيين الحد الأقصى للذاكرة
php_value memory_limit 128M

# تعيين منطقة زمنية افتراضية
php_value date.timezone Asia/Riyadh
EOT;

    $htaccess_file = $base_path . '/.htaccess';
    
    if (file_put_contents($htaccess_file, $htaccess_content)) {
        echo "تم إنشاء ملف .htaccess بنجاح<br>";
        return true;
    } else {
        echo "فشل إنشاء ملف .htaccess<br>";
        return false;
    }
}

// دالة لإنشاء ملف robots.txt
function createRobotsFile($base_path) {
    $robots_content = <<<EOT
# robots.txt
# تم إنشاؤه بواسطة أداة تحسين أمان النظام

User-agent: *
Disallow: /admin/
Disallow: /config/
Disallow: /database/
Disallow: /logs/
Disallow: /utils/
Disallow: /uploads/
Disallow: /includes/
Disallow: /api/
Disallow: /auth/
Disallow: /assets/
Disallow: /*.php$
Disallow: /*.js$
Disallow: /*.css$
Disallow: /*.sql$
Disallow: /*.md$
Disallow: /*.json$
Disallow: /*.lock$
Disallow: /*.gitignore$
Disallow: /*.htaccess$
Disallow: /*.env$
Disallow: /*.log$
Disallow: /*.txt$
Disallow: /*.xml$
Disallow: /*.yml$
Disallow: /*.yaml$
Disallow: /*.sh$
Disallow: /*.bat$
Disallow: /*.cmd$
Disallow: /*.ini$
Disallow: /*.conf$
Disallow: /*.config$
Disallow: /*.bak$
Disallow: /*.old$
Disallow: /*.tmp$
Disallow: /*.temp$
Disallow: /*.swp$
Disallow: /*.swo$
Disallow: /*.swn$
Disallow: /*.swm$
Disallow: /*.swl$
Disallow: /*.swk$
Disallow: /*.swj$
Disallow: /*.swi$
Disallow: /*.swh$
Disallow: /*.swg$
Disallow: /*.swf$
Disallow: /*.swe$
Disallow: /*.swd$
Disallow: /*.swc$
Disallow: /*.swb$
Disallow: /*.swa$
Disallow: /*.sw9$
Disallow: /*.sw8$
Disallow: /*.sw7$
Disallow: /*.sw6$
Disallow: /*.sw5$
Disallow: /*.sw4$
Disallow: /*.sw3$
Disallow: /*.sw2$
Disallow: /*.sw1$
Disallow: /*.sw0$
EOT;

    $robots_file = $base_path . '/robots.txt';
    
    if (file_put_contents($robots_file, $robots_content)) {
        echo "تم إنشاء ملف robots.txt بنجاح<br>";
        return true;
    } else {
        echo "فشل إنشاء ملف robots.txt<br>";
        return false;
    }
}

// دالة لإنشاء صفحات الخطأ
function createErrorPages($base_path) {
    // صفحة الخطأ 404
    $error_404_content = <<<EOT
<?php
// تعيين رمز الحالة HTTP
http_response_code(404);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>404 - الصفحة غير موجودة</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css">
    <style>
        body {
            background-color: #f8f9fa;
            font-family: 'Tajawal', sans-serif;
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .error-container {
            text-align: center;
            padding: 2rem;
            background-color: white;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
            max-width: 500px;
        }
        .error-code {
            font-size: 6rem;
            font-weight: bold;
            color: #dc3545;
            margin-bottom: 1rem;
        }
        .error-message {
            font-size: 1.5rem;
            margin-bottom: 2rem;
            color: #343a40;
        }
        .home-button {
            background-color: #0d6efd;
            color: white;
            border: none;
            padding: 0.5rem 1.5rem;
            border-radius: 5px;
            font-size: 1rem;
            cursor: pointer;
            transition: background-color 0.3s;
        }
        .home-button:hover {
            background-color: #0b5ed7;
        }
    </style>
</head>
<body>
    <div class="error-container">
        <div class="error-code">404</div>
        <div class="error-message">الصفحة التي تبحث عنها غير موجودة</div>
        <a href="/" class="home-button">العودة للصفحة الرئيسية</a>
    </div>
</body>
</html>
EOT;

    // صفحة الخطأ 403
    $error_403_content = <<<EOT
<?php
// تعيين رمز الحالة HTTP
http_response_code(403);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>403 - الوصول محظور</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css">
    <style>
        body {
            background-color: #f8f9fa;
            font-family: 'Tajawal', sans-serif;
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .error-container {
            text-align: center;
            padding: 2rem;
            background-color: white;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
            max-width: 500px;
        }
        .error-code {
            font-size: 6rem;
            font-weight: bold;
            color: #dc3545;
            margin-bottom: 1rem;
        }
        .error-message {
            font-size: 1.5rem;
            margin-bottom: 2rem;
            color: #343a40;
        }
        .home-button {
            background-color: #0d6efd;
            color: white;
            border: none;
            padding: 0.5rem 1.5rem;
            border-radius: 5px;
            font-size: 1rem;
            cursor: pointer;
            transition: background-color 0.3s;
        }
        .home-button:hover {
            background-color: #0b5ed7;
        }
    </style>
</head>
<body>
    <div class="error-container">
        <div class="error-code">403</div>
        <div class="error-message">ليس لديك صلاحية للوصول إلى هذه الصفحة</div>
        <a href="/" class="home-button">العودة للصفحة الرئيسية</a>
    </div>
</body>
</html>
EOT;

    // صفحة الخطأ 500
    $error_500_content = <<<EOT
<?php
// تعيين رمز الحالة HTTP
http_response_code(500);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>500 - خطأ في الخادم</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.rtl.min.css">
    <style>
        body {
            background-color: #f8f9fa;
            font-family: 'Tajawal', sans-serif;
            height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .error-container {
            text-align: center;
            padding: 2rem;
            background-color: white;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
            max-width: 500px;
        }
        .error-code {
            font-size: 6rem;
            font-weight: bold;
            color: #dc3545;
            margin-bottom: 1rem;
        }
        .error-message {
            font-size: 1.5rem;
            margin-bottom: 2rem;
            color: #343a40;
        }
        .home-button {
            background-color: #0d6efd;
            color: white;
            border: none;
            padding: 0.5rem 1.5rem;
            border-radius: 5px;
            font-size: 1rem;
            cursor: pointer;
            transition: background-color 0.3s;
        }
        .home-button:hover {
            background-color: #0b5ed7;
        }
    </style>
</head>
<body>
    <div class="error-container">
        <div class="error-code">500</div>
        <div class="error-message">حدث خطأ في الخادم</div>
        <a href="/" class="home-button">العودة للصفحة الرئيسية</a>
    </div>
</body>
</html>
EOT;

    // إنشاء صفحات الخطأ
    $error_pages = [
        '404.php' => $error_404_content,
        '403.php' => $error_403_content,
        '500.php' => $error_500_content
    ];
    
    $success = true;
    
    foreach ($error_pages as $filename => $content) {
        $file_path = $base_path . '/' . $filename;
        
        if (file_put_contents($file_path, $content)) {
            echo "تم إنشاء صفحة الخطأ $filename بنجاح<br>";
        } else {
            echo "فشل إنشاء صفحة الخطأ $filename<br>";
            $success = false;
        }
    }
    
    return $success;
}

// تنفيذ العمليات
$operations = [
    'htaccess' => createHtaccessFile($base_path),
    'robots' => createRobotsFile($base_path),
    'error_pages' => createErrorPages($base_path)
];

// عرض النتائج
echo "<h1>نتائج تحسين أمان النظام</h1>";

foreach ($operations as $operation => $result) {
    echo "<h2>$operation</h2>";
    
    if ($result) {
        echo "<p style='color: green;'>تمت العملية بنجاح</p>";
    } else {
        echo "<p style='color: red;'>فشلت العملية</p>";
    }
}

echo "<p>تم الانتهاء من تحسين أمان النظام بنجاح.</p>";
