<?php
/**
 * أداة تحسين أداء النظام
 * 
 * هذه الأداة تقوم بتحسين أداء النظام وتقليل استهلاك الموارد
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// دالة لضغط ملفات CSS
function minifyCssFiles($base_path) {
    // البحث عن ملفات CSS
    $css_files = glob($base_path . '/assets/css/*.css');
    
    $results = [];
    
    foreach ($css_files as $file) {
        // تجاهل الملفات المضغوطة بالفعل
        if (strpos(basename($file), '.min.') !== false) {
            continue;
        }
        
        // قراءة محتوى الملف
        $css = file_get_contents($file);
        
        // ضغط CSS
        $minified = minifyCss($css);
        
        // إنشاء اسم الملف المضغوط
        $min_file = str_replace('.css', '.min.css', $file);
        
        // حفظ الملف المضغوط
        if (file_put_contents($min_file, $minified)) {
            $original_size = filesize($file);
            $minified_size = filesize($min_file);
            $savings = round(($original_size - $minified_size) / $original_size * 100, 2);
            
            $results[] = [
                'file' => basename($file),
                'original_size' => $original_size,
                'minified_size' => $minified_size,
                'savings' => $savings
            ];
        }
    }
    
    return $results;
}

// دالة لضغط ملفات JavaScript
function minifyJsFiles($base_path) {
    // البحث عن ملفات JavaScript
    $js_files = glob($base_path . '/assets/js/*.js');
    
    $results = [];
    
    foreach ($js_files as $file) {
        // تجاهل الملفات المضغوطة بالفعل
        if (strpos(basename($file), '.min.') !== false) {
            continue;
        }
        
        // قراءة محتوى الملف
        $js = file_get_contents($file);
        
        // ضغط JavaScript
        $minified = minifyJs($js);
        
        // إنشاء اسم الملف المضغوط
        $min_file = str_replace('.js', '.min.js', $file);
        
        // حفظ الملف المضغوط
        if (file_put_contents($min_file, $minified)) {
            $original_size = filesize($file);
            $minified_size = filesize($min_file);
            $savings = round(($original_size - $minified_size) / $original_size * 100, 2);
            
            $results[] = [
                'file' => basename($file),
                'original_size' => $original_size,
                'minified_size' => $minified_size,
                'savings' => $savings
            ];
        }
    }
    
    return $results;
}

// دالة لضغط CSS
function minifyCss($css) {
    // إزالة التعليقات
    $css = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css);
    
    // إزالة المسافات الزائدة
    $css = preg_replace('/\s+/', ' ', $css);
    
    // إزالة المسافات قبل وبعد الأقواس
    $css = preg_replace('/\s*{\s*/', '{', $css);
    $css = preg_replace('/\s*}\s*/', '}', $css);
    
    // إزالة المسافات قبل وبعد الفواصل
    $css = preg_replace('/\s*;\s*/', ';', $css);
    
    // إزالة المسافات قبل وبعد النقطتين
    $css = preg_replace('/\s*:\s*/', ':', $css);
    
    // إزالة الفاصلة المنقوطة الأخيرة في كل كتلة
    $css = preg_replace('/;}/', '}', $css);
    
    // إزالة المسافات في بداية ونهاية الملف
    $css = trim($css);
    
    return $css;
}

// دالة لضغط JavaScript
function minifyJs($js) {
    // إزالة التعليقات متعددة الأسطر
    $js = preg_replace('/\/\*.*?\*\//s', '', $js);
    
    // إزالة التعليقات أحادية السطر
    $js = preg_replace('/\/\/.*$/m', '', $js);
    
    // إزالة المسافات الزائدة
    $js = preg_replace('/\s+/', ' ', $js);
    
    // إزالة المسافات قبل وبعد الأقواس
    $js = preg_replace('/\s*{\s*/', '{', $js);
    $js = preg_replace('/\s*}\s*/', '}', $js);
    
    // إزالة المسافات قبل وبعد الفواصل
    $js = preg_replace('/\s*;\s*/', ';', $js);
    
    // إزالة المسافات قبل وبعد النقطتين
    $js = preg_replace('/\s*:\s*/', ':', $js);
    
    // إزالة المسافات قبل وبعد الأقواس المربعة
    $js = preg_replace('/\s*\[\s*/', '[', $js);
    $js = preg_replace('/\s*\]\s*/', ']', $js);
    
    // إزالة المسافات قبل وبعد الأقواس المستديرة
    $js = preg_replace('/\s*\(\s*/', '(', $js);
    $js = preg_replace('/\s*\)\s*/', ')', $js);
    
    // إزالة المسافات قبل وبعد علامات الجمع والطرح
    $js = preg_replace('/\s*\+\s*/', '+', $js);
    $js = preg_replace('/\s*\-\s*/', '-', $js);
    
    // إزالة المسافات قبل وبعد علامات الضرب والقسمة
    $js = preg_replace('/\s*\*\s*/', '*', $js);
    $js = preg_replace('/\s*\/\s*/', '/', $js);
    
    // إزالة المسافات قبل وبعد علامات المقارنة
    $js = preg_replace('/\s*==\s*/', '==', $js);
    $js = preg_replace('/\s*===\s*/', '===', $js);
    $js = preg_replace('/\s*!=\s*/', '!=', $js);
    $js = preg_replace('/\s*!==\s*/', '!==', $js);
    $js = preg_replace('/\s*>\s*/', '>', $js);
    $js = preg_replace('/\s*<\s*/', '<', $js);
    $js = preg_replace('/\s*>=\s*/', '>=', $js);
    $js = preg_replace('/\s*<=\s*/', '<=', $js);
    
    // إزالة المسافات في بداية ونهاية الملف
    $js = trim($js);
    
    return $js;
}

// دالة لضغط الصور
function optimizeImages($base_path) {
    // البحث عن الصور
    $image_files = array_merge(
        glob($base_path . '/assets/images/*.{jpg,jpeg,png,gif}', GLOB_BRACE),
        glob($base_path . '/uploads/*.{jpg,jpeg,png,gif}', GLOB_BRACE)
    );
    
    $results = [];
    
    foreach ($image_files as $file) {
        $original_size = filesize($file);
        
        // تحديد نوع الصورة
        $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
        
        // ضغط الصورة حسب النوع
        switch ($extension) {
            case 'jpg':
            case 'jpeg':
                $image = imagecreatefromjpeg($file);
                imagejpeg($image, $file, 85); // جودة 85%
                break;
                
            case 'png':
                $image = imagecreatefrompng($file);
                imagepng($image, $file, 8); // مستوى ضغط 8 (من 0 إلى 9)
                break;
                
            case 'gif':
                $image = imagecreatefromgif($file);
                imagegif($image, $file);
                break;
        }
        
        // تحرير الذاكرة
        if (isset($image)) {
            imagedestroy($image);
        }
        
        $optimized_size = filesize($file);
        $savings = round(($original_size - $optimized_size) / $original_size * 100, 2);
        
        $results[] = [
            'file' => basename($file),
            'original_size' => $original_size,
            'optimized_size' => $optimized_size,
            'savings' => $savings
        ];
    }
    
    return $results;
}

// دالة لإنشاء ملف .htaccess لتحسين الأداء
function createPerformanceHtaccess($base_path) {
    $htaccess_content = <<<EOT
# تحسين أداء النظام
# تم إنشاؤه بواسطة أداة تحسين أداء النظام

# تفعيل ضغط GZIP
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript application/x-javascript application/json application/xml
</IfModule>

# تعيين منطقة زمنية افتراضية
SetEnv TZ Asia/Riyadh

# تعيين ترميز افتراضي
AddDefaultCharset UTF-8

# تعيين مدة صلاحية الملفات الثابتة
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/svg+xml "access plus 1 year"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType text/javascript "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType application/x-javascript "access plus 1 month"
    ExpiresByType application/json "access plus 1 day"
    ExpiresByType application/xml "access plus 1 day"
    ExpiresByType text/xml "access plus 1 day"
    ExpiresByType text/html "access plus 1 hour"
    ExpiresByType application/x-font-woff "access plus 1 year"
    ExpiresByType application/x-font-ttf "access plus 1 year"
    ExpiresByType application/x-font-otf "access plus 1 year"
    ExpiresByType application/vnd.ms-fontobject "access plus 1 year"
    ExpiresByType font/ttf "access plus 1 year"
    ExpiresByType font/otf "access plus 1 year"
    ExpiresByType font/woff "access plus 1 year"
    ExpiresByType font/woff2 "access plus 1 year"
</IfModule>

# تعيين رؤوس ETag
<IfModule mod_headers.c>
    <FilesMatch "\.(jpg|jpeg|png|gif|svg|css|js|woff|woff2|ttf|otf)$">
        Header set Cache-Control "max-age=31536000, public"
    </FilesMatch>
    <FilesMatch "\.(html|php)$">
        Header set Cache-Control "max-age=0, no-cache, no-store, must-revalidate"
        Header set Pragma "no-cache"
        Header set Expires "Wed, 11 Jan 1984 05:00:00 GMT"
    </FilesMatch>
</IfModule>

# تعطيل ETag
FileETag None
EOT;

    $htaccess_file = $base_path . '/assets/.htaccess';
    
    if (file_put_contents($htaccess_file, $htaccess_content)) {
        echo "تم إنشاء ملف .htaccess لتحسين الأداء بنجاح<br>";
        return true;
    } else {
        echo "فشل إنشاء ملف .htaccess لتحسين الأداء<br>";
        return false;
    }
}

// تنفيذ العمليات
$operations = [
    'css' => minifyCssFiles($base_path),
    'js' => minifyJsFiles($base_path),
    'images' => optimizeImages($base_path),
    'htaccess' => createPerformanceHtaccess($base_path)
];

// عرض النتائج
echo "<h1>نتائج تحسين أداء النظام</h1>";

// عرض نتائج ضغط CSS
echo "<h2>ضغط ملفات CSS</h2>";
if (!empty($operations['css'])) {
    echo "<table border='1' cellpadding='5' cellspacing='0'>";
    echo "<tr><th>الملف</th><th>الحجم الأصلي</th><th>الحجم المضغوط</th><th>نسبة التوفير</th></tr>";
    
    foreach ($operations['css'] as $result) {
        echo "<tr>";
        echo "<td>{$result['file']}</td>";
        echo "<td>" . round($result['original_size'] / 1024, 2) . " كيلوبايت</td>";
        echo "<td>" . round($result['minified_size'] / 1024, 2) . " كيلوبايت</td>";
        echo "<td>{$result['savings']}%</td>";
        echo "</tr>";
    }
    
    echo "</table>";
} else {
    echo "<p>لم يتم العثور على ملفات CSS للضغط</p>";
}

// عرض نتائج ضغط JavaScript
echo "<h2>ضغط ملفات JavaScript</h2>";
if (!empty($operations['js'])) {
    echo "<table border='1' cellpadding='5' cellspacing='0'>";
    echo "<tr><th>الملف</th><th>الحجم الأصلي</th><th>الحجم المضغوط</th><th>نسبة التوفير</th></tr>";
    
    foreach ($operations['js'] as $result) {
        echo "<tr>";
        echo "<td>{$result['file']}</td>";
        echo "<td>" . round($result['original_size'] / 1024, 2) . " كيلوبايت</td>";
        echo "<td>" . round($result['minified_size'] / 1024, 2) . " كيلوبايت</td>";
        echo "<td>{$result['savings']}%</td>";
        echo "</tr>";
    }
    
    echo "</table>";
} else {
    echo "<p>لم يتم العثور على ملفات JavaScript للضغط</p>";
}

// عرض نتائج ضغط الصور
echo "<h2>ضغط الصور</h2>";
if (!empty($operations['images'])) {
    echo "<table border='1' cellpadding='5' cellspacing='0'>";
    echo "<tr><th>الملف</th><th>الحجم الأصلي</th><th>الحجم المضغوط</th><th>نسبة التوفير</th></tr>";
    
    foreach ($operations['images'] as $result) {
        echo "<tr>";
        echo "<td>{$result['file']}</td>";
        echo "<td>" . round($result['original_size'] / 1024, 2) . " كيلوبايت</td>";
        echo "<td>" . round($result['optimized_size'] / 1024, 2) . " كيلوبايت</td>";
        echo "<td>{$result['savings']}%</td>";
        echo "</tr>";
    }
    
    echo "</table>";
} else {
    echo "<p>لم يتم العثور على صور للضغط</p>";
}

echo "<p>تم الانتهاء من تحسين أداء النظام بنجاح.</p>";
