<?php
/**
 * أداة تنظيم المشروع
 * تستخدم هذه الأداة لتنظيم هيكل المجلدات وإزالة الملفات المكررة
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// تحديد هيكل المجلدات المطلوب
$required_folders = [
    'api',
    'assets',
    'assets/css',
    'assets/js',
    'assets/images',
    'assets/images/logo',
    'assets/images/flags',
    'config',
    'database',
    'includes',
    'public',
    'templates',
    'utils'
];

// تحديد الملفات المكررة التي يجب حذفها
$duplicate_files = [
    // ملفات التكوين المكررة
    'config/subdomain_config.php',
    'shipping/config/subdomain_config.php',
    'subdomain_config.php',
    'shipping/config/deployment.php',
    'config/deployment.php',
    'deploy_config.php',
    
    // ملفات قاعدة البيانات المكررة
    'shipping/database/create_orders_table.sql',
    'shipping/database/create_users_table.sql',
    'shipping/database/create_sqlite_tables.sql',
    
    // ملفات الاختبار
    'test.php',
    'test_connection.php',
    'test_db.php',
    'test_db_connection.php',
    'test_db_connection_updated.php',
    'test_login.php',
    'test_order_number.php',
    'test-login.php',
    'test-register.php',
    'simple-register.php',
    'login-test.php',
    'login-test-new.php',
    'register-simple.php'
];

// دالة لإنشاء المجلدات
function createFolders($folders, $base_path) {
    $created = [];
    $errors = [];
    
    foreach ($folders as $folder) {
        $folder_path = $base_path . '/' . $folder;
        
        if (!file_exists($folder_path)) {
            if (mkdir($folder_path, 0755, true)) {
                $created[] = $folder;
            } else {
                $errors[] = $folder;
            }
        }
    }
    
    return [
        'created' => $created,
        'errors' => $errors
    ];
}

// دالة لحذف الملفات المكررة
function removeDuplicateFiles($files, $base_path) {
    $removed = [];
    $errors = [];
    $not_found = [];
    
    foreach ($files as $file) {
        $file_path = $base_path . '/' . $file;
        
        if (file_exists($file_path)) {
            if (unlink($file_path)) {
                $removed[] = $file;
            } else {
                $errors[] = $file;
            }
        } else {
            $not_found[] = $file;
        }
    }
    
    return [
        'removed' => $removed,
        'errors' => $errors,
        'not_found' => $not_found
    ];
}

// دالة لنقل الملفات من مجلد إلى آخر
function moveFiles($source_dir, $dest_dir, $file_pattern = '*') {
    $moved = [];
    $errors = [];
    
    // التأكد من وجود المجلد المصدر
    if (!file_exists($source_dir)) {
        return [
            'moved' => $moved,
            'errors' => ['المجلد المصدر غير موجود: ' . $source_dir]
        ];
    }
    
    // التأكد من وجود المجلد الهدف
    if (!file_exists($dest_dir)) {
        if (!mkdir($dest_dir, 0755, true)) {
            return [
                'moved' => $moved,
                'errors' => ['فشل في إنشاء المجلد الهدف: ' . $dest_dir]
            ];
        }
    }
    
    // الحصول على قائمة الملفات
    $files = glob($source_dir . '/' . $file_pattern);
    
    foreach ($files as $file) {
        if (is_file($file)) {
            $file_name = basename($file);
            $dest_file = $dest_dir . '/' . $file_name;
            
            // نقل الملف
            if (copy($file, $dest_file)) {
                unlink($file);
                $moved[] = $file_name;
            } else {
                $errors[] = $file_name;
            }
        }
    }
    
    return [
        'moved' => $moved,
        'errors' => $errors
    ];
}

// دالة لإنشاء ملف .htaccess
function createHtaccess($base_path) {
    $htaccess_content = <<<EOT
# تكوين .htaccess الرئيسي
# يحتوي على قواعد إعادة الكتابة وإعدادات الأمان

# تفعيل محرك إعادة الكتابة
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteBase /
    
    # إعادة توجيه HTTP إلى HTTPS (يمكن تفعيله في الإنتاج)
    # RewriteCond %{HTTPS} off
    # RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
    
    # إعادة توجيه API
    RewriteRule ^api/(.*)$ api/$1 [L]
    
    # إعادة توجيه الصفحات
    RewriteRule ^dashboard$ dashboard.php [L]
    RewriteRule ^orders$ orders.php [L]
    RewriteRule ^create-order$ create-order.php [L]
    RewriteRule ^settings$ settings.php [L]
    RewriteRule ^archive$ archive.php [L]
    RewriteRule ^users$ users.php [L]
</IfModule>

# تعيين حجم الملف الأقصى للتحميل
<IfModule mod_php.c>
    php_value upload_max_filesize 10M
    php_value post_max_size 10M
    php_value memory_limit 128M
    php_value max_execution_time 300
    php_value max_input_time 300
</IfModule>

# منع فهرسة المجلدات
Options -Indexes

# منع الوصول إلى مجلدات معينة
<IfModule mod_rewrite.c>
    RewriteRule ^(\.git|\.github|node_modules|vendor)/ - [F,L]
</IfModule>

# منع الوصول إلى ملفات معينة
<FilesMatch "^(\.htaccess|\.htpasswd|config\.php|.*\.sql)$">
    Order Allow,Deny
    Deny from all
</FilesMatch>

# السماح بالوصول إلى ملفات معينة من مجلد config
<FilesMatch "^(config\.js|settings\.json)$">
    Order Allow,Deny
    Allow from all
</FilesMatch>

# ضبط نوع MIME للملفات
<IfModule mod_mime.c>
    AddType application/javascript .js
    AddType text/css .css
    AddType image/svg+xml .svg
    AddType application/font-woff .woff
    AddType application/font-woff2 .woff2
</IfModule>

# ضبط ترميز الملفات
<IfModule mod_charset.c>
    AddDefaultCharset UTF-8
</IfModule>

# تمكين ضغط الملفات
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css application/javascript application/json
</IfModule>
EOT;

    $htaccess_path = $base_path . '/.htaccess';
    
    if (file_put_contents($htaccess_path, $htaccess_content)) {
        return true;
    } else {
        return false;
    }
}

// دالة لإنشاء ملف README.md
function createReadme($base_path) {
    $readme_content = <<<EOT
# نظام طلبات الشحن

نظام متكامل لإدارة طلبات الشحن وتتبعها، مع دعم كامل للغة العربية.

## المميزات

- إدارة الطلبات (إنشاء، تعديل، حذف، عرض)
- إدارة الفروع والمستخدمين
- نظام تسجيل دخول وصلاحيات
- دعم كامل للغة العربية
- واجهة مستخدم سهلة الاستخدام
- تقارير وإحصائيات
- أرشفة الطلبات

## متطلبات النظام

- PHP 7.4 أو أحدث
- MySQL 5.7 أو أحدث (أو SQLite كبديل)
- خادم ويب (Apache أو Nginx)

## التثبيت

1. قم بتنزيل الملفات إلى مجلد الخادم
2. قم بإنشاء قاعدة بيانات جديدة
3. قم بتعديل ملف `config/config.php` بمعلومات قاعدة البيانات
4. قم بزيارة `http://your-domain.com/database/setup_database.php` لإعداد قاعدة البيانات
5. قم بتسجيل الدخول باستخدام اسم المستخدم `admin` وكلمة المرور `admin123`

## هيكل المشروع

- `/api`: ملفات واجهة برمجة التطبيقات
- `/assets`: الملفات الثابتة (CSS, JS, الصور)
- `/config`: ملفات التكوين
- `/database`: ملفات قاعدة البيانات
- `/includes`: الملفات المشتركة والدوال
- `/public`: الملفات العامة
- `/templates`: قوالب النظام
- `/utils`: الأدوات المساعدة

## المساهمة

نرحب بمساهماتكم في تطوير النظام. يرجى اتباع الخطوات التالية:

1. قم بعمل fork للمشروع
2. قم بإنشاء فرع جديد للميزة التي تريد إضافتها
3. قم بإرسال طلب سحب (pull request)

## الترخيص

هذا المشروع مرخص تحت رخصة MIT.
EOT;

    $readme_path = $base_path . '/README.md';
    
    if (file_put_contents($readme_path, $readme_content)) {
        return true;
    } else {
        return false;
    }
}

// عرض نموذج التنظيم
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    ?>
    <!DOCTYPE html>
    <html lang="ar" dir="rtl">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>تنظيم المشروع</title>
        <style>
            body {
                font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                line-height: 1.6;
                color: #333;
                max-width: 800px;
                margin: 0 auto;
                padding: 20px;
                background-color: #f5f5f5;
            }
            h1, h2 {
                color: #2c3e50;
            }
            .container {
                background-color: #fff;
                padding: 20px;
                border-radius: 5px;
                box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            }
            .warning {
                background-color: #fff3cd;
                color: #856404;
                padding: 10px;
                border-radius: 5px;
                margin-bottom: 20px;
            }
            .form-group {
                margin-bottom: 15px;
            }
            label {
                display: block;
                margin-bottom: 5px;
                font-weight: bold;
            }
            button {
                background-color: #3498db;
                color: white;
                border: none;
                padding: 10px 15px;
                border-radius: 4px;
                cursor: pointer;
                font-size: 16px;
            }
            button:hover {
                background-color: #2980b9;
            }
        </style>
    </head>
    <body>
        <div class="container">
            <h1>تنظيم المشروع</h1>
            
            <div class="warning">
                <strong>تحذير:</strong> ستقوم هذه الأداة بتنظيم هيكل المجلدات وإزالة الملفات المكررة. يرجى عمل نسخة احتياطية من المشروع قبل المتابعة.
            </div>
            
            <form method="post" action="">
                <div class="form-group">
                    <label for="create_folders">إنشاء المجلدات المطلوبة:</label>
                    <input type="checkbox" id="create_folders" name="create_folders" value="1" checked>
                </div>
                
                <div class="form-group">
                    <label for="remove_duplicates">إزالة الملفات المكررة:</label>
                    <input type="checkbox" id="remove_duplicates" name="remove_duplicates" value="1" checked>
                </div>
                
                <div class="form-group">
                    <label for="create_htaccess">إنشاء ملف .htaccess:</label>
                    <input type="checkbox" id="create_htaccess" name="create_htaccess" value="1" checked>
                </div>
                
                <div class="form-group">
                    <label for="create_readme">إنشاء ملف README.md:</label>
                    <input type="checkbox" id="create_readme" name="create_readme" value="1" checked>
                </div>
                
                <div class="form-group">
                    <button type="submit">تنظيم المشروع</button>
                </div>
            </form>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// معالجة النموذج
$create_folders = isset($_POST['create_folders']) && $_POST['create_folders'] === '1';
$remove_duplicates = isset($_POST['remove_duplicates']) && $_POST['remove_duplicates'] === '1';
$create_htaccess = isset($_POST['create_htaccess']) && $_POST['create_htaccess'] === '1';
$create_readme = isset($_POST['create_readme']) && $_POST['create_readme'] === '1';

// عرض النتائج
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>نتائج تنظيم المشروع</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
        }
        h1, h2 {
            color: #2c3e50;
        }
        .container {
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .success {
            background-color: #d4edda;
            color: #155724;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .error {
            background-color: #f8d7da;
            color: #721c24;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        ul {
            padding-right: 20px;
        }
        a {
            display: inline-block;
            background-color: #3498db;
            color: white;
            text-decoration: none;
            padding: 10px 15px;
            border-radius: 4px;
            margin-top: 20px;
        }
        a:hover {
            background-color: #2980b9;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>نتائج تنظيم المشروع</h1>
        
        <?php if ($create_folders): ?>
            <h2>إنشاء المجلدات</h2>
            <?php
            $folders_result = createFolders($required_folders, $base_path);
            
            if (!empty($folders_result['created'])):
            ?>
                <div class="success">
                    <p>تم إنشاء المجلدات التالية بنجاح:</p>
                    <ul>
                        <?php foreach ($folders_result['created'] as $folder): ?>
                            <li><?php echo htmlspecialchars($folder); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($folders_result['errors'])): ?>
                <div class="error">
                    <p>فشل في إنشاء المجلدات التالية:</p>
                    <ul>
                        <?php foreach ($folders_result['errors'] as $folder): ?>
                            <li><?php echo htmlspecialchars($folder); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <?php if ($remove_duplicates): ?>
            <h2>إزالة الملفات المكررة</h2>
            <?php
            $duplicates_result = removeDuplicateFiles($duplicate_files, $base_path);
            
            if (!empty($duplicates_result['removed'])):
            ?>
                <div class="success">
                    <p>تم إزالة الملفات التالية بنجاح:</p>
                    <ul>
                        <?php foreach ($duplicates_result['removed'] as $file): ?>
                            <li><?php echo htmlspecialchars($file); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($duplicates_result['errors'])): ?>
                <div class="error">
                    <p>فشل في إزالة الملفات التالية:</p>
                    <ul>
                        <?php foreach ($duplicates_result['errors'] as $file): ?>
                            <li><?php echo htmlspecialchars($file); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($duplicates_result['not_found'])): ?>
                <div class="success">
                    <p>الملفات التالية غير موجودة أصلاً:</p>
                    <ul>
                        <?php foreach ($duplicates_result['not_found'] as $file): ?>
                            <li><?php echo htmlspecialchars($file); ?></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <?php if ($create_htaccess): ?>
            <h2>إنشاء ملف .htaccess</h2>
            <?php if (createHtaccess($base_path)): ?>
                <div class="success">
                    <p>تم إنشاء ملف .htaccess بنجاح.</p>
                </div>
            <?php else: ?>
                <div class="error">
                    <p>فشل في إنشاء ملف .htaccess.</p>
                </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <?php if ($create_readme): ?>
            <h2>إنشاء ملف README.md</h2>
            <?php if (createReadme($base_path)): ?>
                <div class="success">
                    <p>تم إنشاء ملف README.md بنجاح.</p>
                </div>
            <?php else: ?>
                <div class="error">
                    <p>فشل في إنشاء ملف README.md.</p>
                </div>
            <?php endif; ?>
        <?php endif; ?>
        
        <a href="../index.php">العودة إلى الصفحة الرئيسية</a>
    </div>
</body>
</html>
