<?php
/**
 * أداة تحسين قاعدة البيانات
 * 
 * هذه الأداة تقوم بتحسين أداء قاعدة البيانات وتوحيدها
 * وإصلاح المشاكل المحتملة
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تضمين ملف الاتصال بقاعدة البيانات
require_once __DIR__ . '/../config/db_connection.php';

// دالة لفحص حالة قاعدة البيانات
function checkDatabaseStatus($conn) {
    try {
        // التحقق من الاتصال بقاعدة البيانات
        $stmt = $conn->query("SELECT 1");
        if ($stmt === false) {
            return [
                'status' => 'error',
                'message' => 'فشل الاتصال بقاعدة البيانات'
            ];
        }
        
        // الحصول على معلومات قاعدة البيانات
        $stmt = $conn->query("SELECT VERSION() as version");
        $version = $stmt->fetchColumn();
        
        // الحصول على قائمة الجداول
        $stmt = $conn->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        // الحصول على حجم قاعدة البيانات
        $stmt = $conn->query("
            SELECT 
                table_schema as database_name,
                ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) as size_mb
            FROM information_schema.tables
            WHERE table_schema = DATABASE()
            GROUP BY table_schema
        ");
        $size = $stmt->fetchColumn(1);
        
        return [
            'status' => 'success',
            'version' => $version,
            'tables' => $tables,
            'size' => $size,
            'table_count' => count($tables)
        ];
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل فحص حالة قاعدة البيانات: ' . $e->getMessage()
        ];
    }
}

// دالة لفحص هيكل الجداول
function checkTablesStructure($conn) {
    try {
        // الحصول على قائمة الجداول
        $stmt = $conn->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $tableStructures = [];
        
        foreach ($tables as $table) {
            // الحصول على هيكل الجدول
            $stmt = $conn->query("DESCRIBE `$table`");
            $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // الحصول على مفاتيح الجدول
            $stmt = $conn->query("SHOW KEYS FROM `$table`");
            $keys = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // الحصول على عدد السجلات في الجدول
            $stmt = $conn->query("SELECT COUNT(*) FROM `$table`");
            $count = $stmt->fetchColumn();
            
            $tableStructures[$table] = [
                'columns' => $columns,
                'keys' => $keys,
                'record_count' => $count
            ];
        }
        
        return [
            'status' => 'success',
            'structures' => $tableStructures
        ];
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل فحص هيكل الجداول: ' . $e->getMessage()
        ];
    }
}

// دالة لتحسين الجداول
function optimizeTables($conn) {
    try {
        // الحصول على قائمة الجداول
        $stmt = $conn->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $results = [];
        
        foreach ($tables as $table) {
            // تحسين الجدول
            $stmt = $conn->query("OPTIMIZE TABLE `$table`");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $results[$table] = $result['Msg_text'];
        }
        
        return [
            'status' => 'success',
            'results' => $results
        ];
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل تحسين الجداول: ' . $e->getMessage()
        ];
    }
}

// دالة لإصلاح الجداول
function repairTables($conn) {
    try {
        // الحصول على قائمة الجداول
        $stmt = $conn->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $results = [];
        
        foreach ($tables as $table) {
            // إصلاح الجدول
            $stmt = $conn->query("REPAIR TABLE `$table`");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $results[$table] = $result['Msg_text'];
        }
        
        return [
            'status' => 'success',
            'results' => $results
        ];
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل إصلاح الجداول: ' . $e->getMessage()
        ];
    }
}

// دالة لتحليل الجداول
function analyzeTables($conn) {
    try {
        // الحصول على قائمة الجداول
        $stmt = $conn->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $results = [];
        
        foreach ($tables as $table) {
            // تحليل الجدول
            $stmt = $conn->query("ANALYZE TABLE `$table`");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $results[$table] = $result['Msg_text'];
        }
        
        return [
            'status' => 'success',
            'results' => $results
        ];
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل تحليل الجداول: ' . $e->getMessage()
        ];
    }
}

// دالة لإنشاء نسخة احتياطية من قاعدة البيانات
function backupDatabase($conn, $filename = null) {
    try {
        // تحديد اسم الملف
        if ($filename === null) {
            $filename = __DIR__ . '/../database/backup_' . date('Y-m-d_H-i-s') . '.sql';
        }
        
        // التأكد من وجود مجلد النسخ الاحتياطية
        $backup_dir = dirname($filename);
        if (!is_dir($backup_dir)) {
            mkdir($backup_dir, 0755, true);
        }
        
        // الحصول على قائمة الجداول
        $stmt = $conn->query("SHOW TABLES");
        $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        $output = "-- نسخة احتياطية من قاعدة البيانات\n";
        $output .= "-- تاريخ النسخ: " . date('Y-m-d H:i:s') . "\n\n";
        $output .= "SET FOREIGN_KEY_CHECKS=0;\n\n";
        
        foreach ($tables as $table) {
            // الحصول على هيكل الجدول
            $stmt = $conn->query("SHOW CREATE TABLE `$table`");
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            $create_table = $row['Create Table'];
            
            $output .= "-- هيكل جدول `$table`\n";
            $output .= "DROP TABLE IF EXISTS `$table`;\n";
            $output .= "$create_table;\n\n";
            
            // الحصول على بيانات الجدول
            $stmt = $conn->query("SELECT * FROM `$table`");
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (count($rows) > 0) {
                $output .= "-- بيانات جدول `$table`\n";
                $output .= "INSERT INTO `$table` VALUES\n";
                
                $values = [];
                foreach ($rows as $row) {
                    $row_values = [];
                    foreach ($row as $value) {
                        if ($value === null) {
                            $row_values[] = 'NULL';
                        } else {
                            $row_values[] = $conn->quote($value);
                        }
                    }
                    
                    $values[] = '(' . implode(', ', $row_values) . ')';
                }
                
                $output .= implode(",\n", $values) . ";\n\n";
            }
        }
        
        $output .= "SET FOREIGN_KEY_CHECKS=1;\n";
        
        // حفظ النسخة الاحتياطية
        file_put_contents($filename, $output);
        
        return [
            'status' => 'success',
            'filename' => $filename,
            'size' => filesize($filename)
        ];
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => 'فشل إنشاء نسخة احتياطية: ' . $e->getMessage()
        ];
    }
}

// تنفيذ العمليات
$operations = [
    'status' => checkDatabaseStatus($conn),
    'structure' => checkTablesStructure($conn),
    'optimize' => optimizeTables($conn),
    'repair' => repairTables($conn),
    'analyze' => analyzeTables($conn),
    'backup' => backupDatabase($conn)
];

// عرض النتائج
echo "<h1>نتائج تحسين قاعدة البيانات</h1>";

foreach ($operations as $operation => $result) {
    echo "<h2>$operation</h2>";
    
    if ($result['status'] === 'success') {
        echo "<p style='color: green;'>تمت العملية بنجاح</p>";
        
        if ($operation === 'status') {
            echo "<p>إصدار قاعدة البيانات: {$result['version']}</p>";
            echo "<p>عدد الجداول: {$result['table_count']}</p>";
            echo "<p>حجم قاعدة البيانات: {$result['size']} ميجابايت</p>";
        } elseif ($operation === 'backup') {
            echo "<p>تم إنشاء نسخة احتياطية: {$result['filename']}</p>";
            echo "<p>حجم النسخة الاحتياطية: " . round($result['size'] / 1024 / 1024, 2) . " ميجابايت</p>";
        }
    } else {
        echo "<p style='color: red;'>{$result['message']}</p>";
    }
}

echo "<p>تم الانتهاء من تحسين قاعدة البيانات بنجاح.</p>";
