<?php
/**
 * أداة فحص جاهزية النظام للاستضافة على cPanel
 * تستخدم هذه الأداة للتحقق من جاهزية النظام للنشر على استضافة cPanel
 * 
 * @version 1.0
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// دالة للتحقق من متطلبات PHP
function checkPhpRequirements() {
    $requirements = [
        'php_version' => [
            'name' => 'إصدار PHP',
            'required' => '7.0.0',
            'current' => PHP_VERSION,
            'status' => version_compare(PHP_VERSION, '7.0.0', '>=')
        ],
        'pdo_mysql' => [
            'name' => 'PDO MySQL',
            'required' => 'مفعل',
            'current' => extension_loaded('pdo_mysql') ? 'مفعل' : 'غير مفعل',
            'status' => extension_loaded('pdo_mysql')
        ],
        'mbstring' => [
            'name' => 'mbstring',
            'required' => 'مفعل',
            'current' => extension_loaded('mbstring') ? 'مفعل' : 'غير مفعل',
            'status' => extension_loaded('mbstring')
        ],
        'json' => [
            'name' => 'JSON',
            'required' => 'مفعل',
            'current' => extension_loaded('json') ? 'مفعل' : 'غير مفعل',
            'status' => extension_loaded('json')
        ],
        'openssl' => [
            'name' => 'OpenSSL',
            'required' => 'مفعل',
            'current' => extension_loaded('openssl') ? 'مفعل' : 'غير مفعل',
            'status' => extension_loaded('openssl')
        ],
        'file_uploads' => [
            'name' => 'تحميل الملفات',
            'required' => 'مفعل',
            'current' => ini_get('file_uploads') ? 'مفعل' : 'غير مفعل',
            'status' => ini_get('file_uploads')
        ],
        'max_execution_time' => [
            'name' => 'الحد الأقصى لوقت التنفيذ',
            'required' => '30 ثانية',
            'current' => ini_get('max_execution_time') . ' ثانية',
            'status' => ini_get('max_execution_time') >= 30
        ],
        'memory_limit' => [
            'name' => 'حد الذاكرة',
            'required' => '128M',
            'current' => ini_get('memory_limit'),
            'status' => intval(ini_get('memory_limit')) >= 128
        ]
    ];
    
    return $requirements;
}

// دالة للتحقق من صلاحيات الملفات والمجلدات
function checkFilePermissions($base_path) {
    $directories_to_check = [
        'uploads' => '755',
        'logs' => '755',
        'cache' => '755',
        'backups' => '755'
    ];
    
    $files_to_check = [
        'config/db_credentials.php' => '600',
        '.htaccess' => '644'
    ];
    
    $results = [
        'directories' => [],
        'files' => []
    ];
    
    // التحقق من صلاحيات المجلدات
    foreach ($directories_to_check as $directory => $required_permission) {
        $directory_path = $base_path . DIRECTORY_SEPARATOR . $directory;
        
        if (!is_dir($directory_path)) {
            $results['directories'][$directory] = [
                'path' => $directory,
                'required' => $required_permission,
                'current' => 'غير موجود',
                'status' => false
            ];
            continue;
        }
        
        $current_permission = substr(sprintf('%o', fileperms($directory_path)), -3);
        
        $results['directories'][$directory] = [
            'path' => $directory,
            'required' => $required_permission,
            'current' => $current_permission,
            'status' => $this->isPermissionSufficient($current_permission, $required_permission)
        ];
    }
    
    // التحقق من صلاحيات الملفات
    foreach ($files_to_check as $file => $required_permission) {
        $file_path = $base_path . DIRECTORY_SEPARATOR . $file;
        
        if (!file_exists($file_path)) {
            $results['files'][$file] = [
                'path' => $file,
                'required' => $required_permission,
                'current' => 'غير موجود',
                'status' => false
            ];
            continue;
        }
        
        $current_permission = substr(sprintf('%o', fileperms($file_path)), -3);
        
        $results['files'][$file] = [
            'path' => $file,
            'required' => $required_permission,
            'current' => $current_permission,
            'status' => $this->isPermissionSufficient($current_permission, $required_permission)
        ];
    }
    
    return $results;
}

// دالة للتحقق من كفاية الصلاحيات
function isPermissionSufficient($current, $required) {
    // تحويل الصلاحيات إلى أرقام
    $current_digits = str_split($current);
    $required_digits = str_split($required);
    
    // التحقق من كل رقم
    for ($i = 0; $i < 3; $i++) {
        if (intval($current_digits[$i]) < intval($required_digits[$i])) {
            return false;
        }
    }
    
    return true;
}

// دالة للتحقق من وجود الملفات الضرورية
function checkRequiredFiles($base_path) {
    $required_files = [
        'index.php',
        'unified-dashboard.php',
        'config/config.php',
        'config/db_credentials.php',
        'includes/security_functions.php',
        'database/unified_database_setup.sql',
        '.htaccess'
    ];
    
    $results = [];
    
    foreach ($required_files as $file) {
        $file_path = $base_path . DIRECTORY_SEPARATOR . $file;
        
        $results[$file] = [
            'path' => $file,
            'status' => file_exists($file_path)
        ];
    }
    
    return $results;
}

// دالة للتحقق من تكوين قاعدة البيانات
function checkDatabaseConfiguration($base_path) {
    $db_credentials_file = $base_path . DIRECTORY_SEPARATOR . 'config/db_credentials.php';
    
    if (!file_exists($db_credentials_file)) {
        return [
            'status' => false,
            'message' => 'ملف بيانات الاتصال بقاعدة البيانات غير موجود'
        ];
    }
    
    // تعريف ثابت للتحقق من الوصول الآمن
    define('SECURE_ACCESS', true);
    
    // استيراد بيانات الاتصال
    $db_credentials = require $db_credentials_file;
    
    // التحقق من وجود جميع البيانات المطلوبة
    $required_keys = ['db_host', 'db_name', 'db_user', 'db_pass', 'use_mysql'];
    $missing_keys = [];
    
    foreach ($required_keys as $key) {
        if (!isset($db_credentials[$key])) {
            $missing_keys[] = $key;
        }
    }
    
    if (!empty($missing_keys)) {
        return [
            'status' => false,
            'message' => 'بيانات الاتصال بقاعدة البيانات غير مكتملة. البيانات المفقودة: ' . implode(', ', $missing_keys)
        ];
    }
    
    // التحقق من صحة البيانات
    if (empty($db_credentials['db_name']) || $db_credentials['db_name'] === 'اسم_قاعدة_البيانات') {
        return [
            'status' => false,
            'message' => 'يجب تعيين اسم قاعدة البيانات الفعلي'
        ];
    }
    
    if (empty($db_credentials['db_user']) || $db_credentials['db_user'] === 'اسم_المستخدم') {
        return [
            'status' => false,
            'message' => 'يجب تعيين اسم مستخدم قاعدة البيانات الفعلي'
        ];
    }
    
    if (empty($db_credentials['db_pass']) || $db_credentials['db_pass'] === 'كلمة_المرور') {
        return [
            'status' => false,
            'message' => 'يجب تعيين كلمة مرور قاعدة البيانات الفعلية'
        ];
    }
    
    return [
        'status' => true,
        'message' => 'تكوين قاعدة البيانات صحيح',
        'config' => [
            'host' => $db_credentials['db_host'],
            'name' => $db_credentials['db_name'],
            'user' => $db_credentials['db_user'],
            'pass' => '********' // إخفاء كلمة المرور
        ]
    ];
}

// تنفيذ الفحوصات
$php_requirements = checkPhpRequirements();
$file_permissions = checkFilePermissions($base_path);
$required_files = checkRequiredFiles($base_path);
$database_config = checkDatabaseConfiguration($base_path);

// عرض النتائج
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>فحص جاهزية النظام للاستضافة على cPanel</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            color: #333;
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        h1, h2, h3 {
            color: #2c3e50;
        }
        .success {
            color: #27ae60;
        }
        .error {
            color: #e74c3c;
        }
        .warning {
            color: #f39c12;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        th, td {
            padding: 10px;
            text-align: right;
            border-bottom: 1px solid #ddd;
        }
        th {
            background-color: #f8f9fa;
        }
        .back-link {
            display: inline-block;
            margin-top: 20px;
            padding: 10px 15px;
            background-color: #3498db;
            color: white;
            text-decoration: none;
            border-radius: 4px;
        }
        .back-link:hover {
            background-color: #2980b9;
        }
        .summary {
            background-color: #f0f7fb;
            border-left: 5px solid #3498db;
            padding: 10px 15px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>فحص جاهزية النظام للاستضافة على cPanel</h1>
        
        <div class="summary">
            <p>هذه الأداة تقوم بفحص جاهزية النظام للنشر على استضافة cPanel.</p>
        </div>
        
        <h2>متطلبات PHP</h2>
        <table>
            <tr>
                <th>المتطلب</th>
                <th>القيمة المطلوبة</th>
                <th>القيمة الحالية</th>
                <th>الحالة</th>
            </tr>
            <?php foreach ($php_requirements as $requirement): ?>
                <tr>
                    <td><?php echo htmlspecialchars($requirement['name']); ?></td>
                    <td><?php echo htmlspecialchars($requirement['required']); ?></td>
                    <td><?php echo htmlspecialchars($requirement['current']); ?></td>
                    <td class="<?php echo $requirement['status'] ? 'success' : 'error'; ?>">
                        <?php echo $requirement['status'] ? 'متوافق' : 'غير متوافق'; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </table>
        
        <h2>الملفات الضرورية</h2>
        <table>
            <tr>
                <th>الملف</th>
                <th>الحالة</th>
            </tr>
            <?php foreach ($required_files as $file): ?>
                <tr>
                    <td><?php echo htmlspecialchars($file['path']); ?></td>
                    <td class="<?php echo $file['status'] ? 'success' : 'error'; ?>">
                        <?php echo $file['status'] ? 'موجود' : 'غير موجود'; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </table>
        
        <h2>تكوين قاعدة البيانات</h2>
        <?php if ($database_config['status']): ?>
            <p class="success"><?php echo htmlspecialchars($database_config['message']); ?></p>
            <table>
                <tr>
                    <th>الإعداد</th>
                    <th>القيمة</th>
                </tr>
                <?php foreach ($database_config['config'] as $key => $value): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($key); ?></td>
                        <td><?php echo htmlspecialchars($value); ?></td>
                    </tr>
                <?php endforeach; ?>
            </table>
        <?php else: ?>
            <p class="error"><?php echo htmlspecialchars($database_config['message']); ?></p>
        <?php endif; ?>
        
        <h2>صلاحيات الملفات والمجلدات</h2>
        <h3>المجلدات</h3>
        <table>
            <tr>
                <th>المجلد</th>
                <th>الصلاحية المطلوبة</th>
                <th>الصلاحية الحالية</th>
                <th>الحالة</th>
            </tr>
            <?php foreach ($file_permissions['directories'] as $directory): ?>
                <tr>
                    <td><?php echo htmlspecialchars($directory['path']); ?></td>
                    <td><?php echo htmlspecialchars($directory['required']); ?></td>
                    <td><?php echo htmlspecialchars($directory['current']); ?></td>
                    <td class="<?php echo $directory['status'] ? 'success' : 'error'; ?>">
                        <?php echo $directory['status'] ? 'متوافق' : 'غير متوافق'; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </table>
        
        <h3>الملفات</h3>
        <table>
            <tr>
                <th>الملف</th>
                <th>الصلاحية المطلوبة</th>
                <th>الصلاحية الحالية</th>
                <th>الحالة</th>
            </tr>
            <?php foreach ($file_permissions['files'] as $file): ?>
                <tr>
                    <td><?php echo htmlspecialchars($file['path']); ?></td>
                    <td><?php echo htmlspecialchars($file['required']); ?></td>
                    <td><?php echo htmlspecialchars($file['current']); ?></td>
                    <td class="<?php echo $file['status'] ? 'success' : 'error'; ?>">
                        <?php echo $file['status'] ? 'متوافق' : 'غير متوافق'; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </table>
        
        <a href="../index.php" class="back-link">العودة إلى الصفحة الرئيسية</a>
    </div>
</body>
</html>
