# Script to convert SVG files to PNG using .NET Framework
# This script requires Internet access to download the Svg.NET library

# Function to convert SVG to PNG
function Convert-SvgToPng {
    param (
        [string]$svgPath,
        [string]$pngPath,
        [int]$width = 0,
        [int]$height = 0
    )

    try {
        # Load the Svg.NET library
        Add-Type -Path "$PSScriptRoot\Svg.dll"
        
        # Create SVG document
        $svgDocument = [Svg.SvgDocument]::Open($svgPath)
        
        # If width and height are not specified, use the original dimensions
        if ($width -eq 0 -or $height -eq 0) {
            $width = [int]$svgDocument.Width.Value
            $height = [int]$svgDocument.Height.Value
            
            # If dimensions are still 0, use default values
            if ($width -eq 0) { $width = 800 }
            if ($height -eq 0) { $height = 600 }
        }
        
        # Create bitmap
        $bitmap = $svgDocument.Draw($width, $height)
        
        # Save as PNG
        $bitmap.Save($pngPath, [System.Drawing.Imaging.ImageFormat]::Png)
        
        Write-Host "Converted $svgPath to $pngPath"
        return $true
    }
    catch {
        Write-Host "Error converting $svgPath to PNG: $_" -ForegroundColor Red
        return $false
    }
}

# Download Svg.NET library if not present
function Download-SvgLibrary {
    $svgDllPath = "$PSScriptRoot\Svg.dll"
    
    if (-not (Test-Path $svgDllPath)) {
        Write-Host "Downloading Svg.NET library..."
        
        # URL for the Svg.dll
        $url = "https://github.com/svg-net/SVG/releases/download/v3.4.4/Svg.dll"
        
        try {
            Invoke-WebRequest -Uri $url -OutFile $svgDllPath
            Write-Host "Svg.NET library downloaded successfully." -ForegroundColor Green
            return $true
        }
        catch {
            Write-Host "Failed to download Svg.NET library: $_" -ForegroundColor Red
            return $false
        }
    }
    
    return $true
}

# Alternative conversion using System.Windows.Media
function Convert-SvgToPngAlternative {
    param (
        [string]$svgPath,
        [string]$pngPath,
        [int]$width = 800,
        [int]$height = 600
    )
    
    try {
        # Load necessary assemblies
        Add-Type -AssemblyName PresentationCore
        Add-Type -AssemblyName PresentationFramework
        Add-Type -AssemblyName WindowsBase
        Add-Type -AssemblyName System.Xaml
        
        # Read SVG content
        $svgContent = Get-Content -Path $svgPath -Raw
        
        # Create a memory stream from the SVG content
        $stream = New-Object System.IO.MemoryStream
        $writer = New-Object System.IO.StreamWriter($stream)
        $writer.Write($svgContent)
        $writer.Flush()
        $stream.Position = 0
        
        # Create an SVG reader
        $xamlReader = New-Object System.Windows.Markup.XamlReader
        
        # Parse the SVG
        $svgElement = $xamlReader.Load($stream)
        
        # Create a render target bitmap
        $renderBitmap = New-Object System.Windows.Media.Imaging.RenderTargetBitmap($width, $height, 96, 96, [System.Windows.Media.PixelFormats]::Pbgra32)
        
        # Render the SVG to the bitmap
        $renderBitmap.Render($svgElement)
        
        # Create a PNG encoder
        $pngEncoder = New-Object System.Windows.Media.Imaging.PngBitmapEncoder
        
        # Create a frame from the bitmap
        $pngEncoder.Frames.Add([System.Windows.Media.Imaging.BitmapFrame]::Create($renderBitmap))
        
        # Save to file
        $fileStream = New-Object System.IO.FileStream($pngPath, [System.IO.FileMode]::Create)
        $pngEncoder.Save($fileStream)
        $fileStream.Close()
        
        Write-Host "Converted $svgPath to $pngPath using alternative method"
        return $true
    }
    catch {
        Write-Host "Error converting $svgPath to PNG using alternative method: $_" -ForegroundColor Red
        return $false
    }
}

# Main function to process all SVG files
function Process-SvgFiles {
    # Get all SVG files
    $svgFiles = @(
        "images\backgrounds\login-bg.svg",
        "images\countries\sa.svg",
        "images\countries\ye.svg",
        "images\logo\company-logo.svg",
        "pages\images\flags\china.svg",
        "pages\images\flags\oman.svg",
        "pages\images\flags\saudi.svg",
        "pages\images\flags\yemen.svg"
    )
    
    # Try to download the Svg.NET library
    $libraryDownloaded = Download-SvgLibrary
    
    foreach ($svgFile in $svgFiles) {
        # Create PNG path by replacing extension
        $pngFile = $svgFile -replace "\.svg$", ".png"
        
        Write-Host "Processing $svgFile to $pngFile..."
        
        # Ensure directory exists
        $pngDir = Split-Path -Path $pngFile -Parent
        if (-not (Test-Path $pngDir)) {
            New-Item -Path $pngDir -ItemType Directory -Force | Out-Null
        }
        
        $success = $false
        
        # Try the first method if library was downloaded
        if ($libraryDownloaded) {
            $success = Convert-SvgToPng -svgPath $svgFile -pngPath $pngFile
        }
        
        # If first method failed, try the alternative
        if (-not $success) {
            $success = Convert-SvgToPngAlternative -svgPath $svgFile -pngPath $pngFile
        }
        
        if ($success) {
            Write-Host "Successfully converted $svgFile to $pngFile" -ForegroundColor Green
        }
        else {
            Write-Host "Failed to convert $svgFile to $pngFile" -ForegroundColor Red
        }
    }
}

# Run the main function
Process-SvgFiles
