<?php
/**
 * أداة تنظيف الملفات المكررة
 * تستخدم هذه الأداة لتحديد وإزالة الملفات المكررة والغير مستخدمة في النظام
 * 
 * @version 1.0
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تحديد المسار الأساسي للمشروع
$base_path = realpath(__DIR__ . '/..');

// قائمة الملفات المكررة التي يجب إزالتها
$duplicate_files = [
    // ملفات التكوين المكررة
    'config/subdomain_config.php',
    'shipping/config/subdomain_config.php',
    'subdomain_config.php',
    'shipping/config/deployment.php',
    'config/deployment.php',
    'deploy_config.php',
    
    // ملفات قاعدة البيانات المكررة
    'shipping/database/create_orders_table.sql',
    'shipping/database/create_users_table.sql',
    'shipping/database/create_sqlite_tables.sql',
    'database/create_branches_table.sql',
    'database/create_countries_table.sql',
    'database/create_orders_table.sql',
    'database/fix_users_table.sql',
    
    // ملفات تحويل SVG المكررة
    'convert_svg_simple.py',
    'convert_svg_to_png.py',
    
    // ملفات HTML المكررة
    'shipping/index.html',
    'templates/index.html',
    'shipping/templates/index.html',
    
    // ملفات PHP المكررة
    'shipping/index.php',
    'public_html/index.php',
    'database/test_connection.php',
    
    // لوحات التحكم القديمة
    'dashboard.php',
    'admin/dashboard.php',
    'shipping/dashboard.php',
    'control-panel.php',
    'admin-panel.php',
    'user-panel.php',
    
    // ملفات الاختبار
    'test.php',
    'test_connection.php',
    'test_db.php',
    'test_db_connection.php',
    'test_db_connection_updated.php',
    'test_login.php',
    'test_order_number.php',
    'test-login.php',
    'test-register.php',
    'simple-register.php',
    'login-test.php',
    'login-test-new.php',
    'register-simple.php',
    'home.php',
    
    // ملفات الأمان المكررة
    'core/security.php.new',
    
    // ملفات أخرى غير مستخدمة
    'setup/check_db.php',
    'setup/recreate_database.php',
    'setup/import_database.php',
    'setup/setup_database.php'
];

// قائمة المجلدات التي يجب الاحتفاظ بها
$required_folders = [
    'api',
    'assets',
    'assets/css',
    'assets/js',
    'assets/images',
    'assets/images/logo',
    'assets/fonts',
    'auth',
    'config',
    'database',
    'includes',
    'orders',
    'reports',
    'templates',
    'utils',
    'ASDRF43'
];

// دالة لإزالة الملفات المكررة
function removeDuplicateFiles($base_path, $duplicate_files) {
    $removed_files = [];
    $failed_files = [];
    
    foreach ($duplicate_files as $file) {
        $file_path = $base_path . '/' . $file;
        
        if (file_exists($file_path)) {
            // إنشاء نسخة احتياطية قبل الحذف
            $backup_dir = $base_path . '/backups/duplicates';
            if (!is_dir($backup_dir)) {
                mkdir($backup_dir, 0755, true);
            }
            
            $backup_path = $backup_dir . '/' . basename($file) . '.bak';
            copy($file_path, $backup_path);
            
            // حذف الملف
            if (unlink($file_path)) {
                $removed_files[] = $file;
            } else {
                $failed_files[] = $file;
            }
        }
    }
    
    return [
        'removed' => $removed_files,
        'failed' => $failed_files
    ];
}

// دالة للتأكد من وجود المجلدات المطلوبة
function ensureRequiredFolders($base_path, $required_folders) {
    $created_folders = [];
    
    foreach ($required_folders as $folder) {
        $folder_path = $base_path . '/' . $folder;
        
        if (!is_dir($folder_path)) {
            if (mkdir($folder_path, 0755, true)) {
                $created_folders[] = $folder;
            }
        }
    }
    
    return $created_folders;
}

// تنفيذ العمليات
$results = [
    'duplicate_files' => removeDuplicateFiles($base_path, $duplicate_files),
    'required_folders' => ensureRequiredFolders($base_path, $required_folders)
];

// عرض النتائج
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تنظيف الملفات المكررة</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            color: #333;
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        h1, h2, h3 {
            color: #2c3e50;
        }
        .success {
            color: #27ae60;
        }
        .error {
            color: #e74c3c;
        }
        ul {
            padding-right: 20px;
        }
        .back-link {
            display: inline-block;
            margin-top: 20px;
            padding: 10px 15px;
            background-color: #3498db;
            color: white;
            text-decoration: none;
            border-radius: 4px;
        }
        .back-link:hover {
            background-color: #2980b9;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>تنظيف الملفات المكررة</h1>
        
        <h2>الملفات المكررة</h2>
        <?php if (!empty($results['duplicate_files']['removed'])): ?>
            <h3 class="success">تم حذف الملفات التالية:</h3>
            <ul>
                <?php foreach ($results['duplicate_files']['removed'] as $file): ?>
                    <li><?php echo htmlspecialchars($file); ?></li>
                <?php endforeach; ?>
            </ul>
        <?php else: ?>
            <p>لم يتم العثور على ملفات مكررة للحذف.</p>
        <?php endif; ?>
        
        <?php if (!empty($results['duplicate_files']['failed'])): ?>
            <h3 class="error">فشل حذف الملفات التالية:</h3>
            <ul>
                <?php foreach ($results['duplicate_files']['failed'] as $file): ?>
                    <li><?php echo htmlspecialchars($file); ?></li>
                <?php endforeach; ?>
            </ul>
        <?php endif; ?>
        
        <h2>المجلدات المطلوبة</h2>
        <?php if (!empty($results['required_folders'])): ?>
            <h3 class="success">تم إنشاء المجلدات التالية:</h3>
            <ul>
                <?php foreach ($results['required_folders'] as $folder): ?>
                    <li><?php echo htmlspecialchars($folder); ?></li>
                <?php endforeach; ?>
            </ul>
        <?php else: ?>
            <p>جميع المجلدات المطلوبة موجودة بالفعل.</p>
        <?php endif; ?>
        
        <a href="../index.php" class="back-link">العودة إلى الصفحة الرئيسية</a>
    </div>
</body>
</html>
