<?php
/**
 * صفحة تحميل أعلام الدول
 * تقوم هذه الصفحة بعرض نموذج لتحميل صور أعلام الدول
 */

// تضمين ملف التكوين
require_once 'config/config.php';

// تضمين ملف الأمان
require_once 'config/security.php';

// تضمين ملف الدوال المساعدة
require_once 'includes/functions.php';

// التحقق من حالة تسجيل الدخول
if (!isLoggedIn()) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: login.php');
    exit;
}

// متغيرات الخطأ والنجاح
$error = '';
$success = '';

// التحقق من إرسال النموذج
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // التحقق من رمز CSRF
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $error = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
    } else {
        // التحقق من تحميل الملف
        if (isset($_FILES['flag_image']) && $_FILES['flag_image']['error'] === UPLOAD_ERR_OK) {
            // الحصول على معلومات الملف
            $fileName = $_FILES['flag_image']['name'];
            $fileTmpName = $_FILES['flag_image']['tmp_name'];
            $fileSize = $_FILES['flag_image']['size'];
            $fileType = $_FILES['flag_image']['type'];
            
            // التحقق من نوع الملف
            $allowedTypes = ['image/jpeg', 'image/png', 'image/gif'];
            if (!in_array($fileType, $allowedTypes)) {
                $error = 'نوع الملف غير مسموح به. يرجى تحميل صورة بتنسيق JPEG أو PNG أو GIF.';
            } else {
                // التحقق من حجم الملف (الحد الأقصى: 2 ميجابايت)
                $maxFileSize = 2 * 1024 * 1024; // 2 ميجابايت
                if ($fileSize > $maxFileSize) {
                    $error = 'حجم الملف كبير جدًا. يرجى تحميل صورة بحجم أقل من 2 ميجابايت.';
                } else {
                    // الحصول على اسم الدولة
                    $countryName = sanitizeInput($_POST['country_name'] ?? '');
                    if (empty($countryName)) {
                        $error = 'يرجى إدخال اسم الدولة.';
                    } else {
                        // إنشاء اسم ملف آمن
                        $safeFileName = slugify($countryName) . '.' . pathinfo($fileName, PATHINFO_EXTENSION);
                        
                        // مسار الملف الوجهة
                        $uploadDir = 'assets/images/flags/';
                        $uploadPath = $uploadDir . $safeFileName;
                        
                        // التأكد من وجود المجلد
                        if (!is_dir($uploadDir)) {
                            mkdir($uploadDir, 0755, true);
                        }
                        
                        // نقل الملف إلى المجلد الوجهة
                        if (move_uploaded_file($fileTmpName, $uploadPath)) {
                            $success = 'تم تحميل صورة علم ' . $countryName . ' بنجاح.';
                            
                            // إضافة الدولة إلى قاعدة البيانات إذا كانت غير موجودة
                            try {
                                $stmt = $pdo->prepare("INSERT IGNORE INTO countries (name, flag_path) VALUES (?, ?)");
                                $stmt->execute([$countryName, $uploadPath]);
                            } catch (PDOException $e) {
                                // تجاهل الخطأ إذا كان جدول الدول غير موجود
                            }
                        } else {
                            $error = 'حدث خطأ أثناء تحميل الملف. يرجى المحاولة مرة أخرى.';
                        }
                    }
                }
            }
        } else {
            $error = 'يرجى اختيار ملف لتحميله.';
        }
    }
}

// الحصول على قائمة الدول المتاحة
$countries = [];
try {
    // التحقق من وجود جدول الدول
    $stmt = $pdo->query("SHOW TABLES LIKE 'countries'");
    if ($stmt->rowCount() > 0) {
        $stmt = $pdo->query("SELECT * FROM countries ORDER BY name");
        $countries = $stmt->fetchAll();
    }
} catch (PDOException $e) {
    // تجاهل الخطأ
}

// الحصول على قائمة صور الأعلام المتاحة
$flags = [];
$flagsDir = 'assets/images/flags/';
if (is_dir($flagsDir)) {
    $files = scandir($flagsDir);
    foreach ($files as $file) {
        if ($file !== '.' && $file !== '..' && is_file($flagsDir . $file)) {
            $flags[] = [
                'name' => pathinfo($file, PATHINFO_FILENAME),
                'path' => $flagsDir . $file
            ];
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>تحميل أعلام الدول | نظام طلبات الشحن</title>
    <link rel="stylesheet" href="assets/css/styles.css">
    <style>
        .upload-form {
            max-width: 600px;
            margin: 0 auto;
            padding: 20px;
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        
        input[type="text"],
        input[type="file"] {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        
        button {
            background-color: #0a2056;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 4px;
            cursor: pointer;
        }
        
        button:hover {
            background-color: #0f3385;
        }
        
        .error-message {
            background-color: #fee2e2;
            color: #b91c1c;
            padding: 10px;
            border-radius: 4px;
            margin-bottom: 20px;
        }
        
        .success-message {
            background-color: #d1fae5;
            color: #047857;
            padding: 10px;
            border-radius: 4px;
            margin-bottom: 20px;
        }
        
        .flags-list {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            gap: 20px;
            margin-top: 30px;
        }
        
        .flag-item {
            text-align: center;
            padding: 10px;
            background-color: #f9f9f9;
            border-radius: 4px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }
        
        .flag-item img {
            max-width: 100%;
            height: auto;
            max-height: 100px;
            margin-bottom: 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>تحميل أعلام الدول</h1>
        
        <?php if (!empty($error)): ?>
        <div class="error-message"><?php echo $error; ?></div>
        <?php endif; ?>
        
        <?php if (!empty($success)): ?>
        <div class="success-message"><?php echo $success; ?></div>
        <?php endif; ?>
        
        <div class="upload-form">
            <form method="post" enctype="multipart/form-data">
                <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                
                <div class="form-group">
                    <label for="country_name">اسم الدولة</label>
                    <input type="text" id="country_name" name="country_name" required>
                </div>
                
                <div class="form-group">
                    <label for="flag_image">صورة العلم</label>
                    <input type="file" id="flag_image" name="flag_image" accept="image/jpeg, image/png, image/gif" required>
                    <small>الحد الأقصى لحجم الملف: 2 ميجابايت. الأنواع المسموح بها: JPEG، PNG، GIF.</small>
                </div>
                
                <button type="submit">تحميل العلم</button>
            </form>
        </div>
        
        <?php if (!empty($flags)): ?>
        <h2>الأعلام المتاحة</h2>
        <div class="flags-list">
            <?php foreach ($flags as $flag): ?>
            <div class="flag-item">
                <img src="<?php echo $flag['path']; ?>" alt="<?php echo $flag['name']; ?>">
                <p><?php echo $flag['name']; ?></p>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
        
        <p><a href="dashboard.php">العودة إلى لوحة التحكم</a></p>
    </div>
</body>
</html>
