<?php
/**
 * أداة لإصلاح تسلسل أرقام الطلبات للفروع
 * تستخدم هذه الأداة للتأكد من أن جميع الفروع لديها سجلات في جدول order_sequences
 * وأن رموز الفروع صحيحة ومتوافقة مع النظام الجديد
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/config.php';

// تعريف رموز الفروع الصحيحة
$correctBranchCodes = [
    'MKL25' => 'فرع المكلا',
    'SNA25' => 'فرع صنعاء',
    'ADN25' => 'فرع عدن',
    'SH25' => 'فرع شحن',
    'CN25' => 'فرع الصين'
];

// رموز الفروع القديمة وما يقابلها من رموز جديدة
$oldToNewBranchCodes = [
    'MKL' => 'MKL25',
    'SNA' => 'SNA25',
    'ADN' => 'ADN25',
    'SH' => 'SH25',
    'CN' => 'CN25'
];

// عرض رأس الصفحة
echo '<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إصلاح تسلسل أرقام الطلبات للفروع</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            line-height: 1.6;
        }
        h1, h2 {
            color: #333;
        }
        table {
            border-collapse: collapse;
            width: 100%;
            margin-bottom: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: right;
        }
        th {
            background-color: #f2f2f2;
        }
        tr:nth-child(even) {
            background-color: #f9f9f9;
        }
        .success {
            color: green;
        }
        .error {
            color: red;
        }
        .warning {
            color: orange;
        }
        .button {
            display: inline-block;
            padding: 10px 15px;
            background-color: #4f46e5;
            color: white;
            text-decoration: none;
            border-radius: 5px;
            margin-top: 20px;
        }
        .button.red {
            background-color: #ef4444;
        }
    </style>
</head>
<body>
    <h1>أداة إصلاح تسلسل أرقام الطلبات للفروع</h1>
    <p>تستخدم هذه الأداة للتأكد من أن جميع الفروع لديها سجلات في جدول order_sequences وأن رموز الفروع صحيحة ومتوافقة مع النظام الجديد.</p>
';

// التحقق من وجود جدول order_sequences
try {
    $tableExists = $pdo->query("SHOW TABLES LIKE 'order_sequences'")->rowCount() > 0;

    if (!$tableExists) {
        // إنشاء جدول order_sequences إذا لم يكن موجوداً
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS `order_sequences` (
              `id` INT AUTO_INCREMENT PRIMARY KEY,
              `branch_code` VARCHAR(10) NOT NULL UNIQUE COMMENT 'رمز الفرع',
              `last_sequence_number` INT NOT NULL DEFAULT 0 COMMENT 'آخر رقم تسلسلي',
              `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
              `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث'
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        echo "<p class='success'>تم إنشاء جدول order_sequences بنجاح.</p>";
    }

    // عرض حالة جدول order_sequences قبل التحديث
    $stmt = $pdo->query("SELECT * FROM order_sequences");
    $sequences = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo "<h2>حالة جدول order_sequences قبل التحديث:</h2>";
    echo "<table>";
    echo "<tr><th>المعرف</th><th>رمز الفرع</th><th>آخر رقم تسلسلي</th><th>تاريخ الإنشاء</th><th>تاريخ التحديث</th></tr>";
    
    if (empty($sequences)) {
        echo "<tr><td colspan='5' style='text-align:center;'>لا توجد سجلات</td></tr>";
    } else {
        foreach ($sequences as $sequence) {
            echo "<tr>";
            echo "<td>{$sequence['id']}</td>";
            echo "<td>{$sequence['branch_code']}</td>";
            echo "<td>{$sequence['last_sequence_number']}</td>";
            echo "<td>{$sequence['created_at']}</td>";
            echo "<td>{$sequence['updated_at']}</td>";
            echo "</tr>";
        }
    }
    
    echo "</table>";

    // إجراء التحديثات إذا تم طلب ذلك
    if (isset($_GET['fix']) && $_GET['fix'] === 'true') {
        // تحديث رموز الفروع القديمة إلى الرموز الجديدة
        foreach ($oldToNewBranchCodes as $oldCode => $newCode) {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM order_sequences WHERE branch_code = ?");
            $stmt->execute([$oldCode]);
            $oldCodeExists = $stmt->fetchColumn() > 0;

            $stmt = $pdo->prepare("SELECT COUNT(*) FROM order_sequences WHERE branch_code = ?");
            $stmt->execute([$newCode]);
            $newCodeExists = $stmt->fetchColumn() > 0;

            if ($oldCodeExists && !$newCodeExists) {
                // تحديث رمز الفرع القديم إلى الرمز الجديد
                $stmt = $pdo->prepare("UPDATE order_sequences SET branch_code = ? WHERE branch_code = ?");
                $stmt->execute([$newCode, $oldCode]);
                echo "<p class='success'>تم تحديث رمز الفرع من {$oldCode} إلى {$newCode}.</p>";
            } elseif ($oldCodeExists && $newCodeExists) {
                // نقل آخر رقم تسلسلي من الرمز القديم إلى الرمز الجديد إذا كان أكبر
                $stmt = $pdo->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
                $stmt->execute([$oldCode]);
                $oldSequence = $stmt->fetchColumn();

                $stmt = $pdo->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
                $stmt->execute([$newCode]);
                $newSequence = $stmt->fetchColumn();

                if ($oldSequence > $newSequence) {
                    $stmt = $pdo->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
                    $stmt->execute([$oldSequence, $newCode]);
                    echo "<p class='success'>تم تحديث آخر رقم تسلسلي للفرع {$newCode} من {$newSequence} إلى {$oldSequence}.</p>";
                }

                // حذف السجل القديم
                $stmt = $pdo->prepare("DELETE FROM order_sequences WHERE branch_code = ?");
                $stmt->execute([$oldCode]);
                echo "<p class='success'>تم حذف سجل الفرع القديم {$oldCode}.</p>";
            }
        }

        // التأكد من وجود جميع رموز الفروع الصحيحة
        foreach ($correctBranchCodes as $branchCode => $branchName) {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM order_sequences WHERE branch_code = ?");
            $stmt->execute([$branchCode]);
            $exists = $stmt->fetchColumn() > 0;

            if (!$exists) {
                // إضافة رمز الفرع إلى جدول order_sequences
                $stmt = $pdo->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, 0)");
                $stmt->execute([$branchCode]);
                echo "<p class='success'>تم إضافة رمز الفرع {$branchCode} ({$branchName}) إلى جدول order_sequences.</p>";
            }
        }

        // إعادة تعيين أرقام التسلسل إذا تم طلب ذلك
        if (isset($_GET['reset']) && $_GET['reset'] === 'true') {
            $stmt = $pdo->prepare("UPDATE order_sequences SET last_sequence_number = 0");
            $stmt->execute();
            echo "<p class='success'>تم إعادة تعيين جميع أرقام التسلسل إلى 0 بنجاح.</p>";
        }
    }

    // عرض حالة جدول order_sequences بعد التحديث
    $stmt = $pdo->query("SELECT * FROM order_sequences");
    $sequences = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo "<h2>حالة جدول order_sequences بعد التحديث:</h2>";
    echo "<table>";
    echo "<tr><th>المعرف</th><th>رمز الفرع</th><th>آخر رقم تسلسلي</th><th>تاريخ الإنشاء</th><th>تاريخ التحديث</th></tr>";
    
    if (empty($sequences)) {
        echo "<tr><td colspan='5' style='text-align:center;'>لا توجد سجلات</td></tr>";
    } else {
        foreach ($sequences as $sequence) {
            echo "<tr>";
            echo "<td>{$sequence['id']}</td>";
            echo "<td>{$sequence['branch_code']}</td>";
            echo "<td>{$sequence['last_sequence_number']}</td>";
            echo "<td>{$sequence['created_at']}</td>";
            echo "<td>{$sequence['updated_at']}</td>";
            echo "</tr>";
        }
    }
    
    echo "</table>";

    // عرض روابط الإجراءات
    echo "<div style='margin-top: 20px;'>";
    echo "<a href='?fix=true' class='button'>إصلاح رموز الفروع</a> ";
    echo "<a href='?fix=true&reset=true' class='button red'>إصلاح رموز الفروع وإعادة تعيين أرقام التسلسل</a> ";
    echo "<a href='../tools/test_sequence_api.php' class='button' style='background-color: #10b981;'>العودة إلى اختبار API</a>";
    echo "</div>";

} catch (PDOException $e) {
    echo "<p class='error'>حدث خطأ: " . $e->getMessage() . "</p>";
}

// عرض تذييل الصفحة
echo '
</body>
</html>
';
?>
