<?php
/**
 * ملف إعادة إنشاء قاعدة البيانات
 * هذا الملف يقوم بإنشاء قاعدة البيانات من جديد وإنشاء جميع الجداول اللازمة
 * يستخدم عند تحميل الملفات إلى استضافة cPanel
 */

// عرض جميع الأخطاء للمساعدة في التشخيص
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// التحقق من الأمان - يمكن تعديل هذا الجزء حسب احتياجاتك
$security_token = 'ASDRF43_SECURE_TOKEN'; // قم بتغيير هذا إلى رمز أمان قوي خاص بك
$is_secure = false;

// التحقق من وجود رمز الأمان في الطلب
if (isset($_GET['token']) && $_GET['token'] === $security_token) {
    $is_secure = true;
}

// معلومات الاتصال بقاعدة البيانات - تم تعديلها للعمل مع Laragon
$db_config = [
    'host' => 'localhost',     // الخادم المحلي في Laragon
    'dbname' => 'shipping_db', // اسم قاعدة البيانات
    'username' => 'root',      // اسم المستخدم الافتراضي في Laragon
    'password' => ''           // كلمة المرور فارغة في الإعدادات الافتراضية لـ Laragon
];

// تحديث إعدادات قاعدة البيانات من النموذج إذا تم تقديمه
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_config') {
    $db_config['host'] = $_POST['db_host'] ?? $db_config['host'];
    $db_config['dbname'] = $_POST['db_name'] ?? $db_config['dbname'];
    $db_config['username'] = $_POST['db_user'] ?? $db_config['username'];
    $db_config['password'] = $_POST['db_pass'] ?? $db_config['password'];
}

// دالة للتحقق من وجود جدول
function tableExists($conn, $tableName) {
    try {
        $result = $conn->query("SHOW TABLES LIKE '$tableName'");
        return $result->rowCount() > 0;
    } catch (PDOException $e) {
        return false;
    }
}

// دالة لتنفيذ ملف SQL
function executeSqlStatements($conn, $sqlStatements) {
    $results = [];

    // تقسيم الاستعلامات
    $queries = explode(';', $sqlStatements);

    // تنفيذ كل استعلام
    foreach ($queries as $query) {
        $query = trim($query);
        if (!empty($query)) {
            // تجاهل الأسطر التي تبدأ بـ -- (تعليقات SQL) أو DELIMITER
            if (strpos($query, '--') === 0 || strpos($query, 'DELIMITER') === 0) {
                continue;
            }

            try {
                $conn->exec($query);
                $results[] = [
                    'status' => 'success',
                    'message' => 'تم تنفيذ الاستعلام بنجاح: ' . substr($query, 0, 50) . '...'
                ];
            } catch (PDOException $e) {
                $results[] = [
                    'status' => 'error',
                    'message' => 'خطأ في تنفيذ الاستعلام: ' . $e->getMessage(),
                    'query' => $query
                ];
            }
        }
    }

    return $results;
}

// دالة لإنشاء قاعدة البيانات والجداول
function createDatabaseAndTables($config, $recreateDb = false) {
    $results = [];

    try {
        // الاتصال بالخادم بدون تحديد قاعدة البيانات
        $conn = new PDO("mysql:host={$config['host']}", $config['username'], $config['password']);
        $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        // ضبط الترميز لدعم اللغة العربية بشكل صحيح
        $conn->exec("SET NAMES utf8mb4");
        $conn->exec("SET CHARACTER SET utf8mb4");
        $conn->exec("SET collation_connection = utf8mb4_unicode_ci");

        $results[] = [
            'status' => 'success',
            'message' => 'تم الاتصال بخادم قاعدة البيانات بنجاح'
        ];

        // إنشاء قاعدة البيانات إذا لم تكن موجودة
        if ($recreateDb) {
            $conn->exec("DROP DATABASE IF EXISTS `{$config['dbname']}`");
            $results[] = [
                'status' => 'success',
                'message' => "تم حذف قاعدة البيانات {$config['dbname']} بنجاح"
            ];
        }

        $conn->exec("CREATE DATABASE IF NOT EXISTS `{$config['dbname']}` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        $results[] = [
            'status' => 'success',
            'message' => "تم إنشاء قاعدة البيانات {$config['dbname']} بنجاح"
        ];

        // استخدام قاعدة البيانات
        $conn->exec("USE `{$config['dbname']}`");

        // إنشاء جدول الفروع
        $branchesTableSQL = "
        CREATE TABLE IF NOT EXISTS `branches` (
          `id` INT AUTO_INCREMENT PRIMARY KEY,
          `name` VARCHAR(100) NOT NULL COMMENT 'اسم الفرع',
          `code` VARCHAR(10) NOT NULL UNIQUE COMMENT 'رمز الفرع',
          `address` TEXT COMMENT 'عنوان الفرع',
          `phone` VARCHAR(20) COMMENT 'رقم هاتف الفرع',
          `manager_id` INT COMMENT 'معرف مدير الفرع',
          `is_active` TINYINT(1) DEFAULT 1 COMMENT 'هل الفرع نشط',
          `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
          `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث'
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

        -- إدراج البيانات الأولية لجدول الفروع
        INSERT IGNORE INTO `branches` (`name`, `code`, `address`, `is_active`) VALUES
        ('فرع عدن', 'ADN', 'عدن - المنصورة', 1),
        ('فرع صنعاء', 'SNA', 'صنعاء - شارع الستين', 1),
        ('فرع المكلا', 'MKL', 'المكلا - الشرج', 1),
        ('فرع القاهرة', 'CN', 'القاهرة - مصر', 1),
        ('فرع الشارقة', 'SH', 'الشارقة - الإمارات', 1);
        ";

        $results = array_merge($results, executeSqlStatements($conn, $branchesTableSQL));

        // إنشاء جدول تسلسل أرقام الطلبات
        $orderSequencesTableSQL = "
        CREATE TABLE IF NOT EXISTS `order_sequences` (
          `id` INT AUTO_INCREMENT PRIMARY KEY,
          `branch_code` VARCHAR(10) NOT NULL UNIQUE COMMENT 'رمز الفرع',
          `last_sequence_number` INT NOT NULL DEFAULT 0 COMMENT 'آخر رقم تسلسلي',
          `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
          `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث'
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

        -- إدراج البيانات الأولية لجدول تسلسل أرقام الطلبات
        INSERT IGNORE INTO `order_sequences` (`branch_code`, `last_sequence_number`) VALUES
        ('ADN', 0),
        ('SNA', 0),
        ('MKL', 0),
        ('CN', 0),
        ('SH', 0);
        ";

        $results = array_merge($results, executeSqlStatements($conn, $orderSequencesTableSQL));

        // إنشاء جدول المستخدمين
        $usersTableSQL = "
        CREATE TABLE IF NOT EXISTS `users` (
          `id` INT AUTO_INCREMENT PRIMARY KEY,
          `username` VARCHAR(50) NOT NULL UNIQUE COMMENT 'اسم المستخدم',
          `password` VARCHAR(255) NOT NULL COMMENT 'كلمة المرور',
          `full_name` VARCHAR(100) NOT NULL COMMENT 'الاسم الكامل',
          `email` VARCHAR(100) UNIQUE COMMENT 'البريد الإلكتروني',
          `phone` VARCHAR(20) COMMENT 'رقم الهاتف',
          `role` ENUM('admin', 'manager', 'employee') NOT NULL DEFAULT 'employee' COMMENT 'دور المستخدم',
          `branch_id` INT COMMENT 'معرف الفرع',
          `is_active` TINYINT(1) DEFAULT 1 COMMENT 'هل المستخدم نشط',
          `is_approved` TINYINT(1) DEFAULT 0 COMMENT 'هل تمت الموافقة على المستخدم',
          `last_login` DATETIME COMMENT 'آخر تسجيل دخول',
          `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
          `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث',
          FOREIGN KEY (`branch_id`) REFERENCES `branches`(`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

        -- إنشاء مستخدم افتراضي للمسؤول (كلمة المرور: admin123)
        INSERT IGNORE INTO `users` (`username`, `password`, `full_name`, `email`, `role`, `is_active`, `is_approved`)
        VALUES ('admin', '$2y$10$5VgXuQKKW.tNOhcLBX2TkuH2U5f2CwGQ/f.MJPVqOvnZcI0qQE9Ni', 'مدير النظام', 'admin@example.com', 'admin', 1, 1);
        ";

        $results = array_merge($results, executeSqlStatements($conn, $usersTableSQL));

        // إنشاء جدول الطلبات
        $ordersTableSQL = "
        CREATE TABLE IF NOT EXISTS `orders` (
          `id` INT AUTO_INCREMENT PRIMARY KEY,
          `order_number` VARCHAR(20) NOT NULL UNIQUE COMMENT 'رقم الطلب',
          `branch_id` INT NOT NULL COMMENT 'معرف الفرع',
          `sender_name` VARCHAR(100) NOT NULL COMMENT 'اسم المرسل',
          `sender_phone` VARCHAR(20) NOT NULL COMMENT 'رقم هاتف المرسل',
          `sender_address` TEXT COMMENT 'عنوان المرسل',
          `receiver_name` VARCHAR(100) NOT NULL COMMENT 'اسم المستلم',
          `receiver_phone` VARCHAR(20) NOT NULL COMMENT 'رقم هاتف المستلم',
          `receiver_address` TEXT COMMENT 'عنوان المستلم',
          `service_type` VARCHAR(50) COMMENT 'نوع الخدمة',
          `package_type` VARCHAR(50) COMMENT 'نوع الطرد',
          `weight` DECIMAL(10,2) COMMENT 'الوزن بالكيلوغرام',
          `quantity` INT DEFAULT 1 COMMENT 'الكمية',
          `amount` DECIMAL(10,2) COMMENT 'المبلغ',
          `shipping_cost` DECIMAL(10,2) COMMENT 'تكلفة الشحن',
          `total_cost` DECIMAL(10,2) COMMENT 'التكلفة الإجمالية',
          `payment_status` ENUM('paid', 'unpaid', 'partial') DEFAULT 'unpaid' COMMENT 'حالة الدفع',
          `payment_method` VARCHAR(50) COMMENT 'طريقة الدفع',
          `status` VARCHAR(50) DEFAULT 'pending' COMMENT 'حالة الطلب',
          `notes` TEXT COMMENT 'ملاحظات',
          `created_by` INT COMMENT 'معرف المستخدم الذي أنشأ الطلب',
          `updated_by` INT COMMENT 'معرف المستخدم الذي قام بتحديث الطلب',
          `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
          `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث',
          FOREIGN KEY (`branch_id`) REFERENCES `branches`(`id`) ON DELETE CASCADE,
          FOREIGN KEY (`created_by`) REFERENCES `users`(`id`) ON DELETE SET NULL,
          FOREIGN KEY (`updated_by`) REFERENCES `users`(`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        ";

        $results = array_merge($results, executeSqlStatements($conn, $ordersTableSQL));

        // إنشاء جدول الطلبات المؤرشفة
        $archivedOrdersTableSQL = "
        CREATE TABLE IF NOT EXISTS `archived_orders` (
          `id` INT AUTO_INCREMENT PRIMARY KEY,
          `order_id` INT COMMENT 'معرف الطلب الأصلي',
          `order_number` VARCHAR(20) NOT NULL COMMENT 'رقم الطلب',
          `branch_id` INT NOT NULL COMMENT 'معرف الفرع',
          `sender_name` VARCHAR(100) NOT NULL COMMENT 'اسم المرسل',
          `sender_phone` VARCHAR(20) NOT NULL COMMENT 'رقم هاتف المرسل',
          `sender_address` TEXT COMMENT 'عنوان المرسل',
          `receiver_name` VARCHAR(100) NOT NULL COMMENT 'اسم المستلم',
          `receiver_phone` VARCHAR(20) NOT NULL COMMENT 'رقم هاتف المستلم',
          `receiver_address` TEXT COMMENT 'عنوان المستلم',
          `service_type` VARCHAR(50) COMMENT 'نوع الخدمة',
          `package_type` VARCHAR(50) COMMENT 'نوع الطرد',
          `weight` DECIMAL(10,2) COMMENT 'الوزن بالكيلوغرام',
          `quantity` INT DEFAULT 1 COMMENT 'الكمية',
          `amount` DECIMAL(10,2) COMMENT 'المبلغ',
          `shipping_cost` DECIMAL(10,2) COMMENT 'تكلفة الشحن',
          `total_cost` DECIMAL(10,2) COMMENT 'التكلفة الإجمالية',
          `payment_status` ENUM('paid', 'unpaid', 'partial') DEFAULT 'unpaid' COMMENT 'حالة الدفع',
          `payment_method` VARCHAR(50) COMMENT 'طريقة الدفع',
          `status` VARCHAR(50) DEFAULT 'archived' COMMENT 'حالة الطلب',
          `notes` TEXT COMMENT 'ملاحظات',
          `created_by` INT COMMENT 'معرف المستخدم الذي أنشأ الطلب',
          `updated_by` INT COMMENT 'معرف المستخدم الذي قام بتحديث الطلب',
          `archived_by` INT COMMENT 'معرف المستخدم الذي قام بأرشفة الطلب',
          `original_created_at` TIMESTAMP COMMENT 'تاريخ إنشاء الطلب الأصلي',
          `archived_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الأرشفة'
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        ";

        $results = array_merge($results, executeSqlStatements($conn, $archivedOrdersTableSQL));

        // التحقق من وجود الجداول بعد التنفيذ
        $tables = ['branches', 'order_sequences', 'users', 'orders', 'archived_orders'];
        foreach ($tables as $table) {
            $exists = tableExists($conn, $table);
            $results[] = [
                'status' => $exists ? 'success' : 'warning',
                'message' => 'جدول ' . $table . ': ' . ($exists ? 'تم إنشاؤه بنجاح' : 'لم يتم إنشاؤه')
            ];
        }

        return $results;
    } catch (PDOException $e) {
        return [
            [
                'status' => 'error',
                'message' => 'خطأ في الاتصال بقاعدة البيانات: ' . $e->getMessage()
            ]
        ];
    }
}

// معالجة الطلب
$results = [];
$action = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $is_secure) {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];

        if ($action === 'create_db') {
            $recreateDb = isset($_POST['recreate_db']) && $_POST['recreate_db'] === '1';
            $results = createDatabaseAndTables($db_config, $recreateDb);
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إعادة إنشاء قاعدة البيانات</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
            color: #333;
            direction: rtl;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        h1, h2 {
            color: #2c3e50;
        }
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 4px;
        }
        .alert-danger {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        .alert-warning {
            background-color: #fff3cd;
            color: #856404;
            border: 1px solid #ffeeba;
        }
        .alert-success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        .alert-info {
            background-color: #d1ecf1;
            color: #0c5460;
            border: 1px solid #bee5eb;
        }
        .form-group {
            margin-bottom: 15px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        input[type="text"], input[type="password"] {
            width: 100%;
            padding: 8px;
            border: 1px solid #ddd;
            border-radius: 4px;
            box-sizing: border-box;
        }
        button {
            background-color: #3498db;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 4px;
            cursor: pointer;
        }
        button.danger {
            background-color: #e74c3c;
        }
        button:hover {
            opacity: 0.9;
        }
        .checkbox-group {
            margin: 15px 0;
        }
        hr {
            border: 0;
            border-top: 1px solid #eee;
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>أداة إعادة إنشاء قاعدة البيانات</h1>

        <div class="alert alert-warning">
            <strong>تعليمات الاستخدام في بيئة Laragon:</strong>
            <ol>
                <li>تأكد من تشغيل خدمات Apache و MySQL في Laragon</li>
                <li>تم ضبط إعدادات قاعدة البيانات تلقائيًا للعمل مع Laragon</li>
                <li>انقر على زر "إنشاء قاعدة البيانات والجداول" لإعداد قاعدة البيانات</li>
                <li>بعد الانتهاء، انتقل إلى صفحة تسجيل الدخول للوصول إلى النظام</li>
            </ol>
        </div>

        <?php if (!$is_secure): ?>
        <div class="alert alert-danger">
            <strong>تنبيه!</strong> يجب توفير رمز الأمان الصحيح للوصول إلى هذه الأداة.
            <p>استخدم الرابط التالي: <code><?php echo $_SERVER['PHP_SELF'] . '?token=ASDRF43_SECURE_TOKEN'; ?></code></p>
            <p>يمكنك تغيير رمز الأمان في ملف PHP إذا كنت ترغب في ذلك.</p>
        </div>
        <?php else: ?>

        <div class="form-container">
            <h2>إعدادات قاعدة البيانات</h2>
            <form method="post" action="<?php echo $_SERVER['PHP_SELF'] . '?token=' . htmlspecialchars($_GET['token']); ?>">
                <input type="hidden" name="action" value="update_config">

                <div class="form-group">
                    <label for="db_host">اسم المضيف:</label>
                    <input type="text" id="db_host" name="db_host" value="<?php echo htmlspecialchars($db_config['host']); ?>" required>
                </div>

                <div class="form-group">
                    <label for="db_name">اسم قاعدة البيانات:</label>
                    <input type="text" id="db_name" name="db_name" value="<?php echo htmlspecialchars($db_config['dbname']); ?>" required>
                </div>

                <div class="form-group">
                    <label for="db_user">اسم المستخدم:</label>
                    <input type="text" id="db_user" name="db_user" value="<?php echo htmlspecialchars($db_config['username']); ?>" required>
                </div>

                <div class="form-group">
                    <label for="db_pass">كلمة المرور:</label>
                    <input type="password" id="db_pass" name="db_pass" value="<?php echo htmlspecialchars($db_config['password']); ?>">
                </div>

                <button type="submit">تحديث الإعدادات</button>
            </form>

            <hr>

            <h2>إنشاء قاعدة البيانات والجداول</h2>
            <div class="alert alert-info">
                <strong>ملاحظة هامة لمستخدمي Laragon:</strong>
                <p>سيقوم هذا السكريبت بإنشاء قاعدة البيانات تلقائيًا في Laragon:</p>
                <ol>
                    <li>تأكد من تشغيل خدمة MySQL في Laragon (الضوء الأخضر)</li>
                    <li>سيتم إنشاء قاعدة بيانات باسم "shipping_db" تلقائيًا</li>
                    <li>سيتم استخدام حساب "root" بدون كلمة مرور (الإعدادات الافتراضية في Laragon)</li>
                    <li>يمكنك تعديل الإعدادات أعلاه إذا كنت قد غيرت إعدادات MySQL الافتراضية</li>
                </ol>
            </div>
            <form method="post" action="<?php echo $_SERVER['PHP_SELF'] . '?token=' . htmlspecialchars($_GET['token']); ?>">
                <input type="hidden" name="action" value="create_db">

                <div class="checkbox-group">
                    <label>
                        <input type="checkbox" name="recreate_db" value="1"> إعادة إنشاء قاعدة البيانات (سيتم حذف قاعدة البيانات الحالية إذا كانت موجودة)
                    </label>
                </div>

                <button type="submit">إنشاء قاعدة البيانات والجداول</button>
                <button type="submit" name="recreate_db" value="1" class="danger">حذف وإعادة إنشاء قاعدة البيانات</button>
            </form>

            <?php if (!empty($results)): ?>
            <hr>

            <h2>نتائج العملية</h2>
            <?php foreach ($results as $result): ?>
                <div class="alert alert-<?php echo $result['status'] === 'success' ? 'success' : ($result['status'] === 'warning' ? 'warning' : 'danger'); ?>">
                    <?php echo htmlspecialchars($result['message']); ?>
                </div>
            <?php endforeach; ?>

            <?php if (array_filter($results, function($r) { return $r['status'] === 'success'; })): ?>
            <div class="alert alert-success">
                <strong>تم إعداد قاعدة البيانات بنجاح!</strong>
                <p>الخطوات التالية:</p>
                <ol>
                    <li>تم تحديث ملف الإعدادات الرئيسي للتطبيق (includes/config.php) تلقائيًا</li>
                    <li>انتقل إلى صفحة تسجيل الدخول: <a href="public_html/login.php">صفحة تسجيل الدخول</a></li>
                    <li>استخدم بيانات الاعتماد الافتراضية: اسم المستخدم: <strong>admin</strong>، كلمة المرور: <strong>admin123</strong></li>
                </ol>
            </div>
            <?php endif; ?>

            <?php endif; ?>

        </div>
        <?php endif; ?>
    </div>
</body>
</html>
