<?php
/**
 * إعدادات النظام
 * تعرض هذه الصفحة إعدادات النظام وتتيح التحكم فيها
 *
 * @version 1.0
 * @author فريق التطوير
 */

// بدء الجلسة
session_start();

// تضمين ملف التكوين
require_once 'config/config.php';

// تضمين ملفات إدارة الإعدادات والصلاحيات
require_once 'includes/settings_manager.php';
require_once 'includes/permissions_manager.php';

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// التحقق من حالة تسجيل الدخول
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: auth/login.php');
    exit;
}

// الحصول على معلومات المستخدم
$user = [
    'id' => $_SESSION['user_id'] ?? 0,
    'username' => $_SESSION['username'] ?? '',
    'full_name' => $_SESSION['full_name'] ?? 'المستخدم',
    'email' => $_SESSION['email'] ?? '',
    'role' => $_SESSION['role'] ?? 'user',
    'branch_id' => $_SESSION['branch_id'] ?? null
];

// تحديد ما إذا كان المستخدم مديرًا
$isAdmin = (isset($_SESSION['role']) && $_SESSION['role'] === 'admin');

// التحقق من صلاحيات المستخدم للوصول إلى صفحة الإعدادات
if (!$isAdmin && !checkUserPermission('manage_settings')) {
    // إعادة توجيه المستخدم إلى لوحة التحكم
    header('Location: unified-dashboard.php');
    exit;
}

// معالجة الإجراءات (حفظ الإعدادات)
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'save_general':
                // حفظ الإعدادات العامة
                $generalSettings = [
                    'system_name' => $_POST['system_name'] ?? '',
                    'company_name' => $_POST['company_name'] ?? '',
                    'admin_email' => $_POST['admin_email'] ?? '',
                    'support_email' => $_POST['support_email'] ?? '',
                    'phone' => $_POST['phone'] ?? '',
                    'address' => $_POST['address'] ?? '',
                    'currency' => $_POST['currency'] ?? '',
                    'timezone' => $_POST['timezone'] ?? '',
                    'date_format' => $_POST['date_format'] ?? '',
                    'time_format' => $_POST['time_format'] ?? ''
                ];

                // تحديث الإعدادات في قاعدة البيانات
                if (updateSettings($generalSettings)) {
                    $message = 'تم حفظ الإعدادات العامة بنجاح';
                    $messageType = 'success';
                } else {
                    $message = 'حدث خطأ أثناء حفظ الإعدادات العامة';
                    $messageType = 'error';
                }
                break;

            case 'save_appearance':
                // حفظ إعدادات المظهر
                $appearanceSettings = [
                    'theme' => $_POST['theme'] ?? 'default',
                    'primary_color' => $_POST['primary_color'] ?? '#4F46E5',
                    'secondary_color' => $_POST['secondary_color'] ?? '#6B7280',
                    'success_color' => $_POST['success_color'] ?? '#10B981',
                    'danger_color' => $_POST['danger_color'] ?? '#EF4444',
                    'warning_color' => $_POST['warning_color'] ?? '#F59E0B',
                    'info_color' => $_POST['info_color'] ?? '#3B82F6',
                    'logo' => $_POST['logo'] ?? 'assets/images/backgrounds/background.png',
                    'favicon' => $_POST['favicon'] ?? 'assets/images/favicon.ico',
                    'rtl_support' => isset($_POST['rtl_support']) ? '1' : '0',
                    'show_footer' => isset($_POST['show_footer']) ? '1' : '0'
                ];

                // تحديث الإعدادات في قاعدة البيانات
                if (updateSettings($appearanceSettings)) {
                    $message = 'تم حفظ إعدادات المظهر بنجاح';
                    $messageType = 'success';
                } else {
                    $message = 'حدث خطأ أثناء حفظ إعدادات المظهر';
                    $messageType = 'error';
                }
                break;

            case 'save_permissions':
                // حفظ إعدادات الصلاحيات
                if (isset($_POST['matrix_permission'])) {
                    $success = true;

                    // معالجة مصفوفة الصلاحيات
                    foreach ($_POST['matrix_permission'] as $roleId => $permissions) {
                        $permissionIds = array_keys($permissions);

                        // تحديث صلاحيات الدور
                        if (!updateRolePermissions($roleId, $permissionIds)) {
                            $success = false;
                        }
                    }

                    if ($success) {
                        $message = 'تم حفظ إعدادات الصلاحيات بنجاح';
                        $messageType = 'success';
                    } else {
                        $message = 'حدث خطأ أثناء حفظ إعدادات الصلاحيات';
                        $messageType = 'error';
                    }
                } elseif (isset($_POST['role_id']) && isset($_POST['role_display_name']) && isset($_POST['role_description'])) {
                    // تحديث معلومات الدور
                    $roleId = $_POST['role_id'];
                    $displayName = $_POST['role_display_name'];
                    $description = $_POST['role_description'];

                    if (updateRole($roleId, $displayName, $description)) {
                        $message = 'تم تحديث الدور بنجاح';
                        $messageType = 'success';
                    } else {
                        $message = 'حدث خطأ أثناء تحديث الدور';
                        $messageType = 'error';
                    }
                } elseif (isset($_POST['new_role_name']) && isset($_POST['new_role_display_name'])) {
                    // إضافة دور جديد
                    $name = $_POST['new_role_name'];
                    $displayName = $_POST['new_role_display_name'];
                    $description = $_POST['new_role_description'] ?? '';

                    if (addRole($name, $displayName, $description)) {
                        $message = 'تم إضافة الدور الجديد بنجاح';
                        $messageType = 'success';
                    } else {
                        $message = 'حدث خطأ أثناء إضافة الدور الجديد';
                        $messageType = 'error';
                    }
                } elseif (isset($_POST['delete_role_id'])) {
                    // حذف دور
                    $roleId = $_POST['delete_role_id'];

                    if (deleteRole($roleId)) {
                        $message = 'تم حذف الدور بنجاح';
                        $messageType = 'success';
                    } else {
                        $message = 'حدث خطأ أثناء حذف الدور';
                        $messageType = 'error';
                    }
                } elseif (isset($_POST['new_permission_name']) && isset($_POST['new_permission_display_name'])) {
                    // إضافة صلاحية جديدة
                    $name = $_POST['new_permission_name'];
                    $displayName = $_POST['new_permission_display_name'];
                    $description = $_POST['new_permission_description'] ?? '';
                    $category = $_POST['new_permission_category'] ?? 'general';

                    if (addPermission($name, $displayName, $description, $category)) {
                        $message = 'تم إضافة الصلاحية الجديدة بنجاح';
                        $messageType = 'success';
                    } else {
                        $message = 'حدث خطأ أثناء إضافة الصلاحية الجديدة';
                        $messageType = 'error';
                    }
                } elseif (isset($_POST['permission_id']) && isset($_POST['permission_display_name'])) {
                    // تحديث معلومات الصلاحية
                    $permissionId = $_POST['permission_id'];
                    $displayName = $_POST['permission_display_name'];
                    $description = $_POST['permission_description'] ?? '';
                    $category = $_POST['permission_category'] ?? 'general';

                    if (updatePermission($permissionId, $displayName, $description, $category)) {
                        $message = 'تم تحديث الصلاحية بنجاح';
                        $messageType = 'success';
                    } else {
                        $message = 'حدث خطأ أثناء تحديث الصلاحية';
                        $messageType = 'error';
                    }
                } elseif (isset($_POST['delete_permission_id'])) {
                    // حذف صلاحية
                    $permissionId = $_POST['delete_permission_id'];

                    if (deletePermission($permissionId)) {
                        $message = 'تم حذف الصلاحية بنجاح';
                        $messageType = 'success';
                    } else {
                        $message = 'حدث خطأ أثناء حذف الصلاحية';
                        $messageType = 'error';
                    }
                }
                break;

            case 'save_modules':
                // حفظ إعدادات الوحدات
                if (isset($_POST['module_status']) && isset($_POST['module_order'])) {
                    $success = true;

                    foreach ($_POST['module_status'] as $moduleId => $status) {
                        // تحديث حالة الوحدة
                        if (!updateModuleStatus($moduleId, $status)) {
                            $success = false;
                        }

                        // تحديث ترتيب الوحدة
                        if (isset($_POST['module_order'][$moduleId])) {
                            $order = (int)$_POST['module_order'][$moduleId];
                            if (!updateModuleOrder($moduleId, $order)) {
                                $success = false;
                            }
                        }
                    }

                    if ($success) {
                        $message = 'تم حفظ إعدادات الوحدات بنجاح';
                        $messageType = 'success';
                    } else {
                        $message = 'حدث خطأ أثناء حفظ إعدادات الوحدات';
                        $messageType = 'error';
                    }
                }
                break;

            case 'save_notifications':
                // حفظ إعدادات الإشعارات
                $notificationSettings = [
                    'email_notifications' => isset($_POST['email_notifications']) ? '1' : '0',
                    'browser_notifications' => isset($_POST['browser_notifications']) ? '1' : '0',
                    'order_created' => isset($_POST['order_created']) ? '1' : '0',
                    'order_updated' => isset($_POST['order_updated']) ? '1' : '0',
                    'order_completed' => isset($_POST['order_completed']) ? '1' : '0',
                    'order_cancelled' => isset($_POST['order_cancelled']) ? '1' : '0',
                    'user_registered' => isset($_POST['user_registered']) ? '1' : '0',
                    'password_reset' => isset($_POST['password_reset']) ? '1' : '0',
                    'system_updates' => isset($_POST['system_updates']) ? '1' : '0',
                    'backup_completed' => isset($_POST['backup_completed']) ? '1' : '0'
                ];

                // تحديث الإعدادات في قاعدة البيانات
                if (updateSettings($notificationSettings)) {
                    $message = 'تم حفظ إعدادات الإشعارات بنجاح';
                    $messageType = 'success';
                } else {
                    $message = 'حدث خطأ أثناء حفظ إعدادات الإشعارات';
                    $messageType = 'error';
                }
                break;

            case 'save_backup':
                // إنشاء نسخة احتياطية
                // هنا يمكن إضافة كود لإنشاء نسخة احتياطية من قاعدة البيانات
                $message = 'تم إنشاء نسخة احتياطية بنجاح';
                $messageType = 'success';
                break;

            default:
                $message = 'إجراء غير معروف';
                $messageType = 'error';
        }
    }
}

// التحقق من وجود جدول الإعدادات وإنشائه إذا لم يكن موجودًا
try {
    $tableExists = $pdo->query("SHOW TABLES LIKE 'system_settings'")->rowCount() > 0;

    if (!$tableExists) {
        // إنشاء جدول إعدادات النظام
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS `system_settings` (
              `id` INT AUTO_INCREMENT PRIMARY KEY,
              `setting_key` VARCHAR(50) NOT NULL UNIQUE COMMENT 'مفتاح الإعداد',
              `setting_value` TEXT DEFAULT NULL COMMENT 'قيمة الإعداد',
              `setting_group` VARCHAR(50) DEFAULT 'general' COMMENT 'مجموعة الإعداد',
              `description` VARCHAR(255) DEFAULT NULL COMMENT 'وصف الإعداد',
              `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
              `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث'
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // إدراج الإعدادات الافتراضية
        $pdo->exec("
            INSERT INTO `system_settings` (`setting_key`, `setting_value`, `setting_group`, `description`) VALUES
            ('system_name', 'نظام طلبات الشحن', 'general', 'اسم النظام'),
            ('company_name', 'مجموعة ابراهيم الاحمدي اليمنية', 'general', 'اسم الشركة'),
            ('admin_email', 'ibrahim@al-ahmadigroup.com', 'general', 'البريد الإلكتروني للمدير'),
            ('support_email', 'support@al-ahmadigroup.com', 'general', 'البريد الإلكتروني للدعم'),
            ('phone', '05318281', 'general', 'رقم الهاتف'),
            ('address', 'اليمن، حضرموت، المكلا', 'general', 'العنوان'),
            ('currency', 'ريال يمني', 'general', 'العملة'),
            ('timezone', 'Asia/Riyadh', 'general', 'المنطقة الزمنية'),
            ('date_format', 'd/m/Y', 'general', 'تنسيق التاريخ'),
            ('time_format', 'H:i', 'general', 'تنسيق الوقت'),
            ('company_logo', 'assets/images/backgrounds/background.png', 'appearance', 'شعار الشركة'),
            ('theme', 'default', 'appearance', 'السمة'),
            ('primary_color', '#4F46E5', 'appearance', 'اللون الرئيسي'),
            ('secondary_color', '#6B7280', 'appearance', 'اللون الثانوي'),
            ('success_color', '#10B981', 'appearance', 'لون النجاح'),
            ('danger_color', '#EF4444', 'appearance', 'لون الخطر'),
            ('warning_color', '#F59E0B', 'appearance', 'لون التحذير'),
            ('info_color', '#3B82F6', 'appearance', 'لون المعلومات'),
            ('logo', 'assets/images/backgrounds/background.png', 'appearance', 'الشعار'),
            ('favicon', 'assets/images/favicon.ico', 'appearance', 'أيقونة الموقع'),
            ('rtl_support', '1', 'appearance', 'دعم الواجهة من اليمين إلى اليسار'),
            ('show_footer', '1', 'appearance', 'عرض تذييل الصفحة'),
            ('email_notifications', '1', 'notifications', 'إشعارات البريد الإلكتروني'),
            ('browser_notifications', '1', 'notifications', 'إشعارات المتصفح'),
            ('order_created', '1', 'notifications', 'إشعار إنشاء طلب جديد'),
            ('order_updated', '1', 'notifications', 'إشعار تحديث طلب'),
            ('order_completed', '1', 'notifications', 'إشعار اكتمال طلب'),
            ('order_cancelled', '1', 'notifications', 'إشعار إلغاء طلب'),
            ('user_registered', '1', 'notifications', 'إشعار تسجيل مستخدم جديد'),
            ('password_reset', '1', 'notifications', 'إشعار إعادة تعيين كلمة المرور'),
            ('system_updates', '1', 'notifications', 'إشعارات تحديثات النظام'),
            ('backup_completed', '1', 'notifications', 'إشعار اكتمال النسخ الاحتياطي')
        ");
    }
} catch (PDOException $e) {
    error_log("خطأ في إنشاء جدول الإعدادات: " . $e->getMessage());
}

// الحصول على الإعدادات من قاعدة البيانات
$generalSettings = [];
$generalSettingsData = getSettingsByGroup('general');
foreach ($generalSettingsData as $key => $data) {
    $generalSettings[$key] = $data['value'];
}

$appearanceSettings = [];
$appearanceSettingsData = getSettingsByGroup('appearance');
foreach ($appearanceSettingsData as $key => $data) {
    $appearanceSettings[$key] = $data['value'];
}

// تعيين شعار الشركة
$company_logo = 'assets/images/Company logo.png';

$notificationSettings = [];
$notificationSettingsData = getSettingsByGroup('notifications');
foreach ($notificationSettingsData as $key => $data) {
    $notificationSettings[$key] = $data['value'] === '1';
}

// الحصول على الأدوار والصلاحيات من قاعدة البيانات
// استخدام دالة getAllRoles من ملف settings_manager.php
$roles = getAllRoles();
// التحقق من وجود دالة getAllPermissions أو getSystemPermissions
if (function_exists('getAllPermissions')) {
    $permissions = getAllPermissions();
} else if (function_exists('getSystemPermissions')) {
    // استخدام دالة getSystemPermissions من ملف permissions_manager.php
    $permissions = getSystemPermissions();
} else {
    $permissions = [];
}

// تنسيق البيانات للعرض
$permissionSettings = [
    'roles' => [],
    'permissions' => $permissions
];

foreach ($roles as $role) {
    $rolePermissions = getRolePermissions($role['id']);
    $permissionNames = array_column($rolePermissions, 'name');

    $permissionSettings['roles'][] = [
        'id' => $role['id'],
        'name' => $role['name'],
        'display_name' => $role['display_name'],
        'permissions' => $permissionNames
    ];
}

// الحصول على وحدات النظام من قاعدة البيانات
$modules = getAllModules();
$moduleSettings = [
    'modules' => $modules
];

// بيانات تجريبية للنسخ الاحتياطية (يمكن استبدالها بقراءة من قاعدة البيانات)
$backups = [
    ['id' => 1, 'name' => 'backup_20230615_120000.zip', 'size' => '2.5 MB', 'date' => '2023-06-15 12:00:00', 'type' => 'تلقائي'],
    ['id' => 2, 'name' => 'backup_20230610_120000.zip', 'size' => '2.4 MB', 'date' => '2023-06-10 12:00:00', 'type' => 'تلقائي'],
    ['id' => 3, 'name' => 'backup_20230605_120000.zip', 'size' => '2.3 MB', 'date' => '2023-06-05 12:00:00', 'type' => 'تلقائي'],
    ['id' => 4, 'name' => 'backup_20230601_150000.zip', 'size' => '2.3 MB', 'date' => '2023-06-01 15:00:00', 'type' => 'يدوي']
];

// دالة لتنسيق التاريخ بالعربية
function formatDateArabic($date) {
    $timestamp = strtotime($date);
    $day = date('d', $timestamp);
    $month = date('m', $timestamp);
    $year = date('Y', $timestamp);
    $time = date('H:i', $timestamp);

    $months = [
        '01' => 'يناير',
        '02' => 'فبراير',
        '03' => 'مارس',
        '04' => 'أبريل',
        '05' => 'مايو',
        '06' => 'يونيو',
        '07' => 'يوليو',
        '08' => 'أغسطس',
        '09' => 'سبتمبر',
        '10' => 'أكتوبر',
        '11' => 'نوفمبر',
        '12' => 'ديسمبر'
    ];

    return $day . ' ' . $months[$month] . ' ' . $year . ' - ' . $time;
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام طلبات الشحن | إعدادات النظام</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="assets/css/unified-design.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/animate.css@4.1.1/animate.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/tippy.js@6.3.7/dist/tippy.css">
  <style>
    /* تنسيق عام للصفحة */
    .settings-page {
      background: linear-gradient(135deg, var(--gray-50) 0%, var(--gray-100) 100%);
    }

    /* تنسيق علامات التبويب */
    .tabs-container {
      position: relative;
      margin-bottom: 2rem;
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      padding: 0.5rem;
    }

    .tabs {
      display: flex;
      overflow-x: auto;
      scrollbar-width: none;
      position: relative;
      z-index: 1;
    }

    .tabs::-webkit-scrollbar {
      display: none;
    }

    .tab {
      padding: 1rem 1.5rem;
      font-weight: 600;
      color: var(--gray-600);
      cursor: pointer;
      white-space: nowrap;
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
      border-radius: var(--border-radius);
      position: relative;
      display: flex;
      align-items: center;
      gap: 0.5rem;
      margin: 0 0.25rem;
    }

    .tab:hover {
      color: var(--primary);
      background-color: var(--primary-50);
    }

    .tab.active {
      color: white;
      background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
      box-shadow: 0 4px 10px rgba(79, 70, 229, 0.3);
      transform: translateY(-2px);
    }

    .tab i {
      font-size: 1.25rem;
      transition: transform 0.3s ease;
    }

    .tab:hover i {
      transform: translateY(-2px);
    }

    .tab-indicator {
      position: absolute;
      height: 3px;
      bottom: 0;
      left: 0;
      background-color: var(--primary);
      transition: all 0.3s ease;
      border-radius: 3px;
    }

    /* تنسيق محتوى التبويب */
    .tab-content {
      display: none;
    }

    .tab-content.active {
      display: block;
      animation: fadeInUp 0.5s ease-out forwards;
    }

    /* تنسيق بطاقات الإعدادات */
    .settings-card {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      margin-bottom: 2rem;
      overflow: hidden;
      border: 1px solid var(--gray-200);
      transition: all 0.3s ease;
    }

    .settings-card:hover {
      box-shadow: var(--shadow-md);
      transform: translateY(-3px);
    }

    .settings-card-header {
      padding: 1.25rem 1.5rem;
      border-bottom: 1px solid var(--gray-200);
      display: flex;
      align-items: center;
      justify-content: space-between;
      background: linear-gradient(to left, var(--gray-50), white);
    }

    .settings-card-title {
      font-weight: 700;
      color: var(--gray-800);
      display: flex;
      align-items: center;
      gap: 0.75rem;
      font-size: 1.1rem;
    }

    .settings-card-title i {
      width: 2.5rem;
      height: 2.5rem;
      display: flex;
      align-items: center;
      justify-content: center;
      background: linear-gradient(135deg, var(--primary-400) 0%, var(--primary-600) 100%);
      color: white;
      border-radius: 50%;
      font-size: 1.25rem;
      box-shadow: 0 4px 10px rgba(79, 70, 229, 0.3);
    }

    .settings-card-body {
      padding: 1.75rem;
    }

    .settings-card-footer {
      padding: 1.25rem 1.5rem;
      border-top: 1px solid var(--gray-200);
      background: var(--gray-50);
      display: flex;
      justify-content: flex-end;
    }

    /* تنسيق مفاتيح التبديل */
    .toggle-switch {
      position: relative;
      display: inline-block;
      width: 52px;
      height: 26px;
    }

    .toggle-switch input {
      opacity: 0;
      width: 0;
      height: 0;
    }

    .toggle-slider {
      position: absolute;
      cursor: pointer;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background-color: var(--gray-300);
      transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
      border-radius: 26px;
      box-shadow: inset 0 2px 4px rgba(0, 0, 0, 0.1);
    }

    .toggle-slider:before {
      position: absolute;
      content: "";
      height: 20px;
      width: 20px;
      left: 3px;
      bottom: 3px;
      background: linear-gradient(135deg, white 0%, #f8f9fa 100%);
      transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
      border-radius: 50%;
      box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
    }

    input:checked + .toggle-slider {
      background: linear-gradient(135deg, var(--primary-400) 0%, var(--primary-600) 100%);
    }

    input:checked + .toggle-slider:before {
      transform: translateX(26px);
      box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
    }

    input:focus + .toggle-slider {
      box-shadow: 0 0 1px var(--primary);
    }

    /* تنسيق مربع اختيار الألوان */
    .color-picker {
      width: 100%;
      height: 42px;
      padding: 0;
      border: 2px solid var(--gray-300);
      border-radius: var(--border-radius);
      cursor: pointer;
      transition: all 0.3s ease;
    }

    .color-picker:hover {
      transform: scale(1.02);
      box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
    }

    .color-picker:focus {
      border-color: var(--primary);
      box-shadow: 0 0 0 3px var(--primary-100);
    }

    /* تنسيق الجدول */
    .data-table-container {
      border-radius: var(--border-radius);
      overflow: hidden;
      box-shadow: var(--shadow);
    }

    .data-table {
      width: 100%;
      border-collapse: separate;
      border-spacing: 0;
      overflow: hidden;
    }

    .data-table th {
      background: linear-gradient(135deg, var(--primary-600) 0%, var(--primary-700) 100%);
      color: white;
      font-weight: 600;
      text-align: right;
      padding: 1rem 1.25rem;
      position: relative;
    }

    .data-table th:after {
      content: '';
      position: absolute;
      bottom: 0;
      left: 0;
      right: 0;
      height: 1px;
      background: rgba(255, 255, 255, 0.2);
    }

    .data-table th:first-child {
      border-top-right-radius: var(--border-radius);
    }

    .data-table th:last-child {
      border-top-left-radius: var(--border-radius);
    }

    .data-table td {
      padding: 1rem 1.25rem;
      border-bottom: 1px solid var(--gray-200);
      transition: all 0.3s ease;
    }

    .data-table tbody tr {
      transition: all 0.3s ease;
    }

    .data-table tbody tr:hover {
      background-color: var(--primary-50);
      transform: translateY(-2px);
      box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    }

    .data-table tbody tr:last-child td {
      border-bottom: none;
    }

    .data-table tbody tr:last-child td:first-child {
      border-bottom-right-radius: var(--border-radius);
    }

    .data-table tbody tr:last-child td:last-child {
      border-bottom-left-radius: var(--border-radius);
    }

    /* تنسيق الحقول */
    .form-input, .form-select, .form-textarea {
      width: 100%;
      padding: 0.75rem 1rem;
      border: 2px solid var(--gray-300);
      border-radius: var(--border-radius);
      transition: all 0.3s ease;
      font-size: 0.95rem;
    }

    .form-input:focus, .form-select:focus, .form-textarea:focus {
      border-color: var(--primary);
      box-shadow: 0 0 0 3px var(--primary-100);
      outline: none;
    }

    .form-label {
      display: block;
      margin-bottom: 0.5rem;
      font-weight: 600;
      color: var(--gray-700);
    }

    .form-group {
      margin-bottom: 1.5rem;
    }

    /* تنسيق الشارات */
    .badge {
      display: inline-flex;
      align-items: center;
      padding: 0.35rem 0.75rem;
      border-radius: 9999px;
      font-size: 0.75rem;
      font-weight: 600;
      line-height: 1;
      transition: all 0.3s ease;
    }

    .badge-primary {
      background-color: var(--primary-100);
      color: var(--primary-700);
    }

    .badge-success {
      background-color: var(--success-100);
      color: var(--success-700);
    }

    .badge-danger {
      background-color: var(--danger-100);
      color: var(--danger-700);
    }

    .badge-warning {
      background-color: var(--warning-100);
      color: var(--warning-700);
    }

    .badge-info {
      background-color: var(--info-100);
      color: var(--info-700);
    }

    /* تنسيق مصفوفة الصلاحيات */
    .permissions-matrix {
      border: 1px solid var(--gray-200);
      border-radius: var(--border-radius);
      overflow: hidden;
    }

    .permissions-matrix-header {
      background: linear-gradient(to left, var(--primary-50), white);
      padding: 1rem;
      border-bottom: 1px solid var(--gray-200);
      font-weight: 600;
    }

    .permissions-matrix-row {
      display: flex;
      border-bottom: 1px solid var(--gray-200);
    }

    .permissions-matrix-row:last-child {
      border-bottom: none;
    }

    .permissions-matrix-cell {
      flex: 1;
      padding: 1rem;
      display: flex;
      align-items: center;
      justify-content: center;
      border-left: 1px solid var(--gray-200);
    }

    .permissions-matrix-cell:last-child {
      border-left: none;
    }

    .permissions-matrix-label {
      flex: 0 0 200px;
      padding: 1rem;
      font-weight: 500;
      background-color: var(--gray-50);
    }

    /* تأثيرات الحركة */
    @keyframes fadeInUp {
      from {
        opacity: 0;
        transform: translateY(20px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    @keyframes pulse {
      0% {
        box-shadow: 0 0 0 0 rgba(79, 70, 229, 0.4);
      }
      70% {
        box-shadow: 0 0 0 10px rgba(79, 70, 229, 0);
      }
      100% {
        box-shadow: 0 0 0 0 rgba(79, 70, 229, 0);
      }
    }

    .animate-fadeInUp {
      animation: fadeInUp 0.5s ease-out forwards;
    }

    .animate-pulse {
      animation: pulse 2s infinite;
    }

    /* تنسيق النوافذ المنبثقة */
    .modal {
      display: none;
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background-color: rgba(0, 0, 0, 0.5);
      z-index: 100;
      overflow-y: auto;
      padding: 2rem 1rem;
      opacity: 0;
      transition: opacity 0.3s ease;
    }

    .modal.active {
      display: flex;
      align-items: center;
      justify-content: center;
      opacity: 1;
    }

    .modal-content {
      background-color: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow-lg);
      width: 100%;
      max-width: 600px;
      max-height: 90vh;
      overflow-y: auto;
      transform: translateY(20px);
      transition: transform 0.3s ease;
      position: relative;
    }

    .modal.active .modal-content {
      transform: translateY(0);
    }

    /* تنسيق الإشعارات */
    .notification {
      position: fixed;
      bottom: 20px;
      left: 20px;
      padding: 1rem 1.5rem;
      border-radius: var(--border-radius);
      background: white;
      box-shadow: var(--shadow-lg);
      display: flex;
      align-items: center;
      gap: 1rem;
      z-index: 1000;
      transform: translateY(100px);
      opacity: 0;
      transition: all 0.3s ease;
    }

    .notification.active {
      transform: translateY(0);
      opacity: 1;
    }

    .notification-success {
      border-right: 4px solid var(--success);
    }

    .notification-error {
      border-right: 4px solid var(--danger);
    }

    .notification-icon {
      width: 2rem;
      height: 2rem;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      font-size: 1rem;
    }

    .notification-success .notification-icon {
      background-color: var(--success-100);
      color: var(--success);
    }

    .notification-error .notification-icon {
      background-color: var(--danger-100);
      color: var(--danger);
    }

    /* تنسيق الأقسام */
    .section-title {
      font-size: 1.25rem;
      font-weight: 700;
      color: var(--gray-800);
      margin-bottom: 1.5rem;
      display: flex;
      align-items: center;
      gap: 0.75rem;
    }

    .section-title i {
      color: var(--primary);
    }

    .section-subtitle {
      font-size: 1rem;
      font-weight: 600;
      color: var(--gray-700);
      margin-bottom: 1rem;
    }

    /* تنسيق الإجراءات السريعة */
    .quick-actions {
      display: grid;
      grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
      gap: 1rem;
      margin-bottom: 2rem;
    }

    .quick-action-card {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      padding: 1.5rem;
      text-align: center;
      transition: all 0.3s ease;
      border: 1px solid var(--gray-200);
      cursor: pointer;
    }

    .quick-action-card:hover {
      transform: translateY(-5px);
      box-shadow: var(--shadow-md);
      border-color: var(--primary-200);
    }

    .quick-action-icon {
      width: 3rem;
      height: 3rem;
      border-radius: 50%;
      background: linear-gradient(135deg, var(--primary-400) 0%, var(--primary-600) 100%);
      color: white;
      display: flex;
      align-items: center;
      justify-content: center;
      font-size: 1.25rem;
      margin: 0 auto 1rem;
      box-shadow: 0 4px 10px rgba(79, 70, 229, 0.3);
    }

    .quick-action-title {
      font-weight: 600;
      color: var(--gray-800);
      margin-bottom: 0.5rem;
    }

    .quick-action-description {
      font-size: 0.875rem;
      color: var(--gray-600);
    }

    /* تنسيق شريط البحث */
    .search-container {
      position: relative;
      margin-bottom: 2rem;
    }

    .search-input {
      width: 100%;
      padding: 0.75rem 1rem 0.75rem 3rem;
      border: 2px solid var(--gray-300);
      border-radius: var(--border-radius);
      transition: all 0.3s ease;
      font-size: 0.95rem;
    }

    .search-input:focus {
      border-color: var(--primary);
      box-shadow: 0 0 0 3px var(--primary-100);
      outline: none;
    }

    .search-icon {
      position: absolute;
      left: 1rem;
      top: 50%;
      transform: translateY(-50%);
      color: var(--gray-500);
      transition: all 0.3s ease;
    }

    .search-input:focus + .search-icon {
      color: var(--primary);
    }
  </style>
</head>

<body>
  <!-- طبقة التراكب للأجهزة المحمولة -->
  <div id="overlay" class="overlay"></div>

  <div class="flex min-h-screen">
    <!-- القائمة الجانبية -->
    <aside class="w-64 hidden md:block">
      <div class="p-6">
        <div class="logo-container flex items-center justify-center mb-8 p-2">
          <img src="<?php echo $company_logo; ?>" alt="شعار الشركة" class="h-40 w-auto object-contain logo-animation">
        </div>

        <div class="user-info mb-8">
          <div class="flex items-center space-x-4 space-x-reverse mb-3">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <h3 class="text-white font-semibold"><?php echo htmlspecialchars($user['full_name']); ?></h3>
              <p class="text-indigo-200 text-sm"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
            </div>
          </div>
          <div class="bg-white/10 h-0.5 rounded-full mb-3"></div>
        </div>

        <nav>
          <ul class="space-y-3">
            <!-- لوحة التحكم -->
            <li>
              <a href="unified-dashboard.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-tachometer-alt ml-3 text-lg"></i>
                <span>لوحة التحكم</span>
              </a>
            </li>

            <!-- إنشاء طلب جديد -->
            <li>
              <a href="create-order-unified.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-plus-circle ml-3 text-lg"></i>
                <span>إنشاء طلب جديد</span>
              </a>
            </li>

            <!-- قائمة الطلبات -->
            <li>
              <a href="orders.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-box ml-3 text-lg"></i>
                <span>قائمة الطلبات</span>
              </a>
            </li>

            <?php if ($isAdmin): ?>
            <!-- قائمة التقارير المنسدلة -->
            <li>
              <a href="#" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white dropdown-toggle" id="reportsDropdown">
                <i class="fas fa-chart-bar ml-3 text-lg"></i>
                <span>التقارير</span>
              </a>
              <ul class="dropdown-menu" id="reportsMenu">
                <li>
                  <a href="weekly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-week ml-2 text-sm"></i>
                    <span>التقارير الأسبوعية</span>
                  </a>
                </li>
                <li>
                  <a href="monthly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white">
                    <i class="fas fa-calendar-alt ml-2 text-sm"></i>
                    <span>التقارير الشهرية</span>
                  </a>
                </li>
              </ul>
            </li>

            <!-- إدارة المستخدمين -->
            <li>
              <a href="users.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-users ml-3 text-lg"></i>
                <span>إدارة المستخدمين</span>
              </a>
            </li>

            <!-- الفروع -->
            <li>
              <a href="branches.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
                <i class="fas fa-code-branch ml-3 text-lg"></i>
                <span>الفروع</span>
              </a>
            </li>

            <!-- إعدادات النظام -->
            <li>
              <a href="settings.php" class="flex items-center p-3 rounded-lg text-white active">
                <i class="fas fa-cog ml-3 text-lg"></i>
                <span>إعدادات النظام</span>
              </a>
            </li>
            <?php endif; ?>
          </ul>

          <div class="pt-6 mt-6 border-t border-white/10">
            <a href="auth/logout.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white">
              <i class="fas fa-sign-out-alt ml-3 text-lg"></i>
              <span>تسجيل الخروج</span>
            </a>
          </div>
        </nav>
      </div>
    </aside>

    <!-- المحتوى الرئيسي -->
    <main class="flex-1 overflow-x-hidden overflow-y-auto settings-page">
      <!-- شريط التنقل العلوي -->
      <header class="bg-white shadow-md sticky top-0 z-30">
        <div class="flex items-center justify-between p-4">
          <!-- زر القائمة للأجهزة المحمولة -->
          <button id="menuBtn" class="md:hidden text-gray-500 hover:text-gray-600 focus:outline-none">
            <i class="fas fa-bars text-xl"></i>
          </button>

          <!-- عنوان الصفحة -->
          <div class="flex items-center">
            <div class="w-10 h-10 rounded-full bg-primary-100 flex items-center justify-center text-primary ml-3 animate-pulse">
              <i class="fas fa-cog text-xl"></i>
            </div>
            <h1 class="text-xl font-bold text-gray-800">إعدادات النظام</h1>
          </div>

          <!-- زر حفظ الإعدادات -->
          <button id="saveSettingsBtn" class="btn btn-primary">
            <i class="fas fa-save ml-2"></i>
            <span>حفظ الإعدادات</span>
          </button>
        </div>
      </header>

      <!-- محتوى الصفحة -->
      <div class="p-6">
        <?php if (!empty($message)): ?>
        <div class="notification notification-<?php echo $messageType === 'success' ? 'success' : 'error'; ?> active mb-6">
          <div class="notification-icon">
            <i class="fas <?php echo $messageType === 'success' ? 'fa-check' : 'fa-exclamation'; ?>"></i>
          </div>
          <div>
            <h3 class="font-semibold"><?php echo $messageType === 'success' ? 'تم بنجاح' : 'خطأ'; ?></h3>
            <p><?php echo $message; ?></p>
          </div>
        </div>
        <?php endif; ?>

        <!-- الإجراءات السريعة -->
        <div class="quick-actions">
          <div class="quick-action-card" id="quickActionGeneral">
            <div class="quick-action-icon">
              <i class="fas fa-cog"></i>
            </div>
            <h3 class="quick-action-title">الإعدادات العامة</h3>
            <p class="quick-action-description">تعديل إعدادات النظام الأساسية</p>
          </div>

          <div class="quick-action-card" id="quickActionBackup">
            <div class="quick-action-icon">
              <i class="fas fa-database"></i>
            </div>
            <h3 class="quick-action-title">النسخ الاحتياطي</h3>
            <p class="quick-action-description">إنشاء واستعادة النسخ الاحتياطية</p>
          </div>
        </div>

        <!-- شريط البحث -->
        <div class="search-container">
          <input type="text" class="search-input" id="settingsSearch" placeholder="ابحث في الإعدادات...">
          <i class="fas fa-search search-icon"></i>
        </div>

        <!-- علامات التبويب -->
        <div class="tabs-container">
          <div class="tabs">
            <div class="tab active" data-tab="general">
              <i class="fas fa-cog"></i>
              <span>الإعدادات العامة</span>
            </div>
            <div class="tab" data-tab="appearance">
              <i class="fas fa-palette"></i>
              <span>المظهر</span>
            </div>

            <div class="tab" data-tab="modules">
              <i class="fas fa-puzzle-piece"></i>
              <span>الوحدات</span>
            </div>
            <div class="tab" data-tab="notifications">
              <i class="fas fa-bell"></i>
              <span>الإشعارات</span>
            </div>
            <div class="tab" data-tab="backup">
              <i class="fas fa-database"></i>
              <span>النسخ الاحتياطي</span>
            </div>
          </div>
        </div>

        <!-- محتوى التبويب: الإعدادات العامة -->
        <div class="tab-content active" id="general-content">
          <form id="generalForm" method="post">
            <input type="hidden" name="action" value="save_general">

            <div class="settings-card">
              <div class="settings-card-header">
                <h3 class="settings-card-title">
                  <i class="fas fa-info-circle ml-2"></i>
                  <span>معلومات النظام</span>
                </h3>
              </div>
              <div class="settings-card-body">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div class="form-group">
                    <label for="system_name" class="form-label">اسم النظام</label>
                    <input type="text" id="system_name" name="system_name" class="form-input" value="<?php echo htmlspecialchars($generalSettings['system_name']); ?>" required>
                  </div>

                  <div class="form-group">
                    <label for="company_name" class="form-label">اسم الشركة</label>
                    <input type="text" id="company_name" name="company_name" class="form-input" value="<?php echo htmlspecialchars($generalSettings['company_name']); ?>" required>
                  </div>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div class="form-group">
                    <label for="admin_email" class="form-label">البريد الإلكتروني للمدير</label>
                    <input type="email" id="admin_email" name="admin_email" class="form-input" value="<?php echo htmlspecialchars($generalSettings['admin_email']); ?>" required>
                  </div>

                  <div class="form-group">
                    <label for="support_email" class="form-label">البريد الإلكتروني للدعم</label>
                    <input type="email" id="support_email" name="support_email" class="form-input" value="<?php echo htmlspecialchars($generalSettings['support_email']); ?>" required>
                  </div>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div class="form-group">
                    <label for="phone" class="form-label">رقم الهاتف</label>
                    <input type="tel" id="phone" name="phone" class="form-input" value="<?php echo htmlspecialchars($generalSettings['phone']); ?>" required>
                  </div>

                  <div class="form-group">
                    <label for="address" class="form-label">العنوان</label>
                    <input type="text" id="address" name="address" class="form-input" value="<?php echo htmlspecialchars($generalSettings['address']); ?>" required>
                  </div>
                </div>
              </div>
            </div>

            <div class="settings-card">
              <div class="settings-card-header">
                <h3 class="settings-card-title">
                  <i class="fas fa-globe ml-2"></i>
                  <span>إعدادات المنطقة</span>
                </h3>
              </div>
              <div class="settings-card-body">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div class="form-group">
                    <label for="currency" class="form-label">العملة</label>
                    <input type="text" id="currency" name="currency" class="form-input" value="<?php echo htmlspecialchars($generalSettings['currency']); ?>" required>
                  </div>

                  <div class="form-group">
                    <label for="timezone" class="form-label">المنطقة الزمنية</label>
                    <select id="timezone" name="timezone" class="form-select" required>
                      <option value="Asia/Riyadh" <?php echo $generalSettings['timezone'] === 'Asia/Riyadh' ? 'selected' : ''; ?>>الرياض (GMT+3)</option>
                      <option value="Asia/Dubai" <?php echo $generalSettings['timezone'] === 'Asia/Dubai' ? 'selected' : ''; ?>>دبي (GMT+4)</option>
                      <option value="Asia/Kuwait" <?php echo $generalSettings['timezone'] === 'Asia/Kuwait' ? 'selected' : ''; ?>>الكويت (GMT+3)</option>
                      <option value="Asia/Bahrain" <?php echo $generalSettings['timezone'] === 'Asia/Bahrain' ? 'selected' : ''; ?>>البحرين (GMT+3)</option>
                      <option value="Asia/Qatar" <?php echo $generalSettings['timezone'] === 'Asia/Qatar' ? 'selected' : ''; ?>>قطر (GMT+3)</option>
                      <option value="Asia/Muscat" <?php echo $generalSettings['timezone'] === 'Asia/Muscat' ? 'selected' : ''; ?>>مسقط (GMT+4)</option>
                    </select>
                  </div>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div class="form-group">
                    <label for="date_format" class="form-label">تنسيق التاريخ</label>
                    <select id="date_format" name="date_format" class="form-select" required>
                      <option value="d/m/Y" <?php echo $generalSettings['date_format'] === 'd/m/Y' ? 'selected' : ''; ?>>DD/MM/YYYY (31/12/2023)</option>
                      <option value="Y-m-d" <?php echo $generalSettings['date_format'] === 'Y-m-d' ? 'selected' : ''; ?>>YYYY-MM-DD (2023-12-31)</option>
                      <option value="m/d/Y" <?php echo $generalSettings['date_format'] === 'm/d/Y' ? 'selected' : ''; ?>>MM/DD/YYYY (12/31/2023)</option>
                      <option value="d.m.Y" <?php echo $generalSettings['date_format'] === 'd.m.Y' ? 'selected' : ''; ?>>DD.MM.YYYY (31.12.2023)</option>
                    </select>
                  </div>

                  <div class="form-group">
                    <label for="time_format" class="form-label">تنسيق الوقت</label>
                    <select id="time_format" name="time_format" class="form-select" required>
                      <option value="H:i" <?php echo $generalSettings['time_format'] === 'H:i' ? 'selected' : ''; ?>>24 ساعة (14:30)</option>
                      <option value="h:i A" <?php echo $generalSettings['time_format'] === 'h:i A' ? 'selected' : ''; ?>>12 ساعة (02:30 PM)</option>
                    </select>
                  </div>
                </div>
              </div>
            </div>
          </form>
        </div>

        <!-- محتوى التبويب: المظهر -->
        <div class="tab-content" id="appearance-content">
          <form id="appearanceForm" method="post">
            <input type="hidden" name="action" value="save_appearance">

            <div class="settings-card">
              <div class="settings-card-header">
                <h3 class="settings-card-title">
                  <i class="fas fa-palette ml-2"></i>
                  <span>الألوان والمظهر</span>
                </h3>
              </div>
              <div class="settings-card-body">
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                  <div class="form-group">
                    <label for="theme" class="form-label">السمة</label>
                    <select id="theme" name="theme" class="form-select" required>
                      <option value="default" <?php echo $appearanceSettings['theme'] === 'default' ? 'selected' : ''; ?>>الافتراضية</option>
                      <option value="light" <?php echo $appearanceSettings['theme'] === 'light' ? 'selected' : ''; ?>>فاتحة</option>
                      <option value="dark" <?php echo $appearanceSettings['theme'] === 'dark' ? 'selected' : ''; ?>>داكنة</option>
                    </select>
                  </div>

                  <div class="form-group">
                    <label for="primary_color" class="form-label">اللون الرئيسي</label>
                    <input type="color" id="primary_color" name="primary_color" class="color-picker" value="<?php echo htmlspecialchars($appearanceSettings['primary_color']); ?>" required>
                  </div>

                  <div class="form-group">
                    <label for="secondary_color" class="form-label">اللون الثانوي</label>
                    <input type="color" id="secondary_color" name="secondary_color" class="color-picker" value="<?php echo htmlspecialchars($appearanceSettings['secondary_color']); ?>" required>
                  </div>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
                  <div class="form-group">
                    <label for="success_color" class="form-label">لون النجاح</label>
                    <input type="color" id="success_color" name="success_color" class="color-picker" value="<?php echo htmlspecialchars($appearanceSettings['success_color']); ?>" required>
                  </div>

                  <div class="form-group">
                    <label for="danger_color" class="form-label">لون الخطر</label>
                    <input type="color" id="danger_color" name="danger_color" class="color-picker" value="<?php echo htmlspecialchars($appearanceSettings['danger_color']); ?>" required>
                  </div>

                  <div class="form-group">
                    <label for="warning_color" class="form-label">لون التحذير</label>
                    <input type="color" id="warning_color" name="warning_color" class="color-picker" value="<?php echo htmlspecialchars($appearanceSettings['warning_color']); ?>" required>
                  </div>

                  <div class="form-group">
                    <label for="info_color" class="form-label">لون المعلومات</label>
                    <input type="color" id="info_color" name="info_color" class="color-picker" value="<?php echo htmlspecialchars($appearanceSettings['info_color']); ?>" required>
                  </div>
                </div>
              </div>
            </div>

            <div class="settings-card">
              <div class="settings-card-header">
                <h3 class="settings-card-title">
                  <i class="fas fa-image ml-2"></i>
                  <span>الشعار والأيقونات</span>
                </h3>
              </div>
              <div class="settings-card-body">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div class="form-group">
                    <label for="logo" class="form-label">شعار الشركة</label>
                    <div class="flex items-center space-x-4 space-x-reverse">
                      <div class="w-24 h-24 border border-gray-300 rounded-lg overflow-hidden">
                        <img src="<?php echo $company_logo; ?>" alt="شعار الشركة" class="w-full h-full object-contain">
                      </div>
                      <div class="flex-1">
                        <input type="file" id="logo_file" name="logo_file" class="hidden">
                        <input type="hidden" id="logo" name="logo" value="assets/images/Company logo.png">
                        <input type="hidden" id="company_logo" name="company_logo" value="assets/images/Company logo.png">
                        <button type="button" id="changeLogo" class="btn btn-secondary btn-sm">تغيير الشعار</button>
                      </div>
                    </div>
                  </div>

                  <div class="form-group">
                    <label for="favicon" class="form-label">أيقونة الموقع</label>
                    <div class="flex items-center space-x-4 space-x-reverse">
                      <div class="w-12 h-12 border border-gray-300 rounded-lg overflow-hidden">
                        <img src="<?php echo htmlspecialchars($appearanceSettings['favicon']); ?>" alt="أيقونة الموقع" class="w-full h-full object-contain">
                      </div>
                      <div class="flex-1">
                        <input type="file" id="favicon_file" name="favicon_file" class="hidden">
                        <input type="hidden" id="favicon" name="favicon" value="<?php echo htmlspecialchars($appearanceSettings['favicon']); ?>">
                        <button type="button" id="changeFavicon" class="btn btn-secondary btn-sm">تغيير الأيقونة</button>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            <div class="settings-card">
              <div class="settings-card-header">
                <h3 class="settings-card-title">
                  <i class="fas fa-sliders-h ml-2"></i>
                  <span>خيارات إضافية</span>
                </h3>
              </div>
              <div class="settings-card-body">
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div class="form-group">
                    <label class="flex items-center justify-between">
                      <span class="form-label">دعم الواجهة من اليمين إلى اليسار (RTL)</span>
                      <label class="toggle-switch">
                        <input type="checkbox" name="rtl_support" <?php echo $appearanceSettings['rtl_support'] ? 'checked' : ''; ?>>
                        <span class="toggle-slider"></span>
                      </label>
                    </label>
                  </div>

                  <div class="form-group">
                    <label class="flex items-center justify-between">
                      <span class="form-label">عرض تذييل الصفحة</span>
                      <label class="toggle-switch">
                        <input type="checkbox" name="show_footer" <?php echo $appearanceSettings['show_footer'] ? 'checked' : ''; ?>>
                        <span class="toggle-slider"></span>
                      </label>
                    </label>
                  </div>
                </div>
              </div>
            </div>
          </form>
        </div>



        <!-- محتوى التبويب: الوحدات -->
        <div class="tab-content" id="modules-content">
          <form id="modulesForm" method="post">
            <input type="hidden" name="action" value="save_modules">

            <div class="settings-card">
              <div class="settings-card-header">
                <h3 class="settings-card-title">
                  <i class="fas fa-puzzle-piece ml-2"></i>
                  <span>إدارة الوحدات</span>
                </h3>
              </div>
              <div class="settings-card-body">
                <p class="text-gray-600 mb-6">يمكنك تفعيل أو تعطيل الوحدات المختلفة في النظام، وتغيير ترتيب ظهورها في القائمة الجانبية.</p>

                <div class="overflow-x-auto">
                  <table class="data-table">
                    <thead>
                      <tr>
                        <th>الترتيب</th>
                        <th>الوحدة</th>
                        <th>الوصف</th>
                        <th>الحالة</th>
                        <th>الإجراءات</th>
                      </tr>
                    </thead>
                    <tbody id="modulesTableBody">
                      <?php foreach ($moduleSettings['modules'] as $module): ?>
                      <tr data-id="<?php echo $module['id']; ?>">
                        <td>
                          <div class="flex items-center">
                            <button type="button" class="move-up-btn p-1 text-gray-500 hover:text-gray-700">
                              <i class="fas fa-arrow-up"></i>
                            </button>
                            <button type="button" class="move-down-btn p-1 text-gray-500 hover:text-gray-700">
                              <i class="fas fa-arrow-down"></i>
                            </button>
                            <input type="hidden" name="module_order[<?php echo $module['id']; ?>]" value="<?php echo $module['order']; ?>">
                            <span class="mr-2"><?php echo $module['order']; ?></span>
                          </div>
                        </td>
                        <td>
                          <div class="flex items-center">
                            <div class="w-8 h-8 rounded-full bg-primary-100 flex items-center justify-center text-primary font-semibold ml-2">
                              <i class="fas fa-<?php
                              switch ($module['name']) {
                                  case 'dashboard':
                                      echo 'tachometer-alt';
                                      break;
                                  case 'orders':
                                      echo 'box';
                                      break;
                                  case 'users':
                                      echo 'users';
                                      break;
                                  case 'branches':
                                      echo 'code-branch';
                                      break;
                                  case 'reports':
                                      echo 'chart-bar';
                                      break;
                                  case 'settings':
                                      echo 'cog';
                                      break;
                                  default:
                                      echo 'puzzle-piece';
                              }
                              ?>"></i>
                            </div>
                            <span class="font-medium"><?php echo htmlspecialchars($module['display_name']); ?></span>
                          </div>
                        </td>
                        <td>
                          <?php
                          $descriptions = [
                              'dashboard' => 'لوحة التحكم الرئيسية للنظام',
                              'orders' => 'إدارة طلبات الشحن',
                              'users' => 'إدارة المستخدمين والصلاحيات',
                              'branches' => 'إدارة فروع الشركة',
                              'reports' => 'تقارير وإحصائيات النظام',
                              'settings' => 'إعدادات النظام المختلفة'
                          ];
                          echo $descriptions[$module['name']] ?? '';
                          ?>
                        </td>
                        <td>
                          <label class="toggle-switch">
                            <input type="checkbox" name="module_status[<?php echo $module['id']; ?>]" <?php echo $module['status'] === 'enabled' ? 'checked' : ''; ?>>
                            <span class="toggle-slider"></span>
                          </label>
                        </td>
                        <td>
                          <div class="flex items-center space-x-2 space-x-reverse">
                            <button type="button" class="edit-module-btn p-1 text-info hover:bg-info/10 rounded-full transition-all" data-id="<?php echo $module['id']; ?>">
                              <i class="fas fa-edit"></i>
                            </button>
                            <?php if (!in_array($module['name'], ['dashboard', 'settings'])): ?>
                            <button type="button" class="delete-module-btn p-1 text-danger hover:bg-danger/10 rounded-full transition-all" data-id="<?php echo $module['id']; ?>">
                              <i class="fas fa-trash-alt"></i>
                            </button>
                            <?php endif; ?>
                          </div>
                        </td>
                      </tr>
                      <?php endforeach; ?>
                    </tbody>
                  </table>
                </div>

                <div class="mt-6">
                  <button type="button" id="addModuleBtn" class="btn btn-primary btn-sm">
                    <i class="fas fa-plus-circle ml-1"></i>
                    <span>إضافة وحدة جديدة</span>
                  </button>
                </div>
              </div>
            </div>
          </form>
        </div>

        <!-- محتوى التبويب: الإشعارات -->
        <div class="tab-content" id="notifications-content">
          <form id="notificationsForm" method="post">
            <input type="hidden" name="action" value="save_notifications">

            <div class="settings-card">
              <div class="settings-card-header">
                <h3 class="settings-card-title">
                  <i class="fas fa-bell ml-2"></i>
                  <span>إعدادات الإشعارات</span>
                </h3>
              </div>
              <div class="settings-card-body">
                <div class="mb-6">
                  <h4 class="text-lg font-semibold mb-4">قنوات الإشعارات</h4>

                  <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">إشعارات البريد الإلكتروني</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="email_notifications" <?php echo $notificationSettings['email_notifications'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">إرسال إشعارات عبر البريد الإلكتروني</p>
                    </div>

                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">إشعارات المتصفح</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="browser_notifications" <?php echo $notificationSettings['browser_notifications'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">عرض إشعارات في المتصفح</p>
                    </div>
                  </div>
                </div>

                <div>
                  <h4 class="text-lg font-semibold mb-4">أحداث الإشعارات</h4>

                  <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">إنشاء طلب جديد</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="order_created" <?php echo $notificationSettings['order_created'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">إرسال إشعار عند إنشاء طلب جديد</p>
                    </div>

                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">تحديث طلب</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="order_updated" <?php echo $notificationSettings['order_updated'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">إرسال إشعار عند تحديث طلب</p>
                    </div>

                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">اكتمال طلب</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="order_completed" <?php echo $notificationSettings['order_completed'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">إرسال إشعار عند اكتمال طلب</p>
                    </div>

                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">إلغاء طلب</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="order_cancelled" <?php echo $notificationSettings['order_cancelled'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">إرسال إشعار عند إلغاء طلب</p>
                    </div>

                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">تسجيل مستخدم جديد</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="user_registered" <?php echo $notificationSettings['user_registered'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">إرسال إشعار عند تسجيل مستخدم جديد</p>
                    </div>

                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">إعادة تعيين كلمة المرور</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="password_reset" <?php echo $notificationSettings['password_reset'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">إرسال إشعار عند إعادة تعيين كلمة المرور</p>
                    </div>

                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">تحديثات النظام</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="system_updates" <?php echo $notificationSettings['system_updates'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">إرسال إشعار عند توفر تحديثات للنظام</p>
                    </div>

                    <div class="form-group">
                      <label class="flex items-center justify-between">
                        <span class="form-label">اكتمال النسخ الاحتياطي</span>
                        <label class="toggle-switch">
                          <input type="checkbox" name="backup_completed" <?php echo $notificationSettings['backup_completed'] ? 'checked' : ''; ?>>
                          <span class="toggle-slider"></span>
                        </label>
                      </label>
                      <p class="text-sm text-gray-500 mt-1">إرسال إشعار عند اكتمال النسخ الاحتياطي</p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </form>
        </div>

        <!-- محتوى التبويب: النسخ الاحتياطي -->
        <div class="tab-content" id="backup-content">
          <form id="backupForm" method="post">
            <input type="hidden" name="action" value="save_backup">

            <div class="settings-card">
              <div class="settings-card-header">
                <h3 class="settings-card-title">
                  <i class="fas fa-database ml-2"></i>
                  <span>النسخ الاحتياطي واستعادة البيانات</span>
                </h3>
              </div>
              <div class="settings-card-body">
                <div class="mb-6">
                  <h4 class="text-lg font-semibold mb-4">إنشاء نسخة احتياطية</h4>

                  <div class="bg-gray-50 p-4 rounded-lg mb-4">
                    <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                      <div>
                        <h5 class="font-medium text-gray-800 mb-1">نسخة احتياطية كاملة</h5>
                        <p class="text-sm text-gray-600">إنشاء نسخة احتياطية كاملة من قاعدة البيانات والملفات</p>
                      </div>
                      <button type="button" id="fullBackupBtn" class="btn btn-primary">
                        <i class="fas fa-download ml-1"></i>
                        <span>إنشاء نسخة احتياطية</span>
                      </button>
                    </div>
                  </div>

                  <div class="bg-gray-50 p-4 rounded-lg">
                    <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                      <div>
                        <h5 class="font-medium text-gray-800 mb-1">نسخة احتياطية لقاعدة البيانات فقط</h5>
                        <p class="text-sm text-gray-600">إنشاء نسخة احتياطية من قاعدة البيانات فقط</p>
                      </div>
                      <button type="button" id="dbBackupBtn" class="btn btn-secondary">
                        <i class="fas fa-database ml-1"></i>
                        <span>إنشاء نسخة احتياطية</span>
                      </button>
                    </div>
                  </div>
                </div>

                <div class="mb-6">
                  <h4 class="text-lg font-semibold mb-4">النسخ الاحتياطية المتوفرة</h4>

                  <div class="overflow-x-auto">
                    <table class="data-table">
                      <thead>
                        <tr>
                          <th>اسم الملف</th>
                          <th>الحجم</th>
                          <th>التاريخ</th>
                          <th>النوع</th>
                          <th>الإجراءات</th>
                        </tr>
                      </thead>
                      <tbody>
                        <?php if (empty($backups)): ?>
                        <tr>
                          <td colspan="5" class="text-center py-8">
                            <div class="flex flex-col items-center justify-center text-gray-500">
                              <i class="fas fa-database text-4xl mb-3 text-gray-300"></i>
                              <p>لا توجد نسخ احتياطية متوفرة</p>
                            </div>
                          </td>
                        </tr>
                        <?php else: ?>
                          <?php foreach ($backups as $backup): ?>
                          <tr>
                            <td class="font-medium"><?php echo htmlspecialchars($backup['name']); ?></td>
                            <td><?php echo htmlspecialchars($backup['size']); ?></td>
                            <td><?php echo formatDateArabic($backup['date']); ?></td>
                            <td>
                              <span class="inline-block px-2 py-1 <?php echo $backup['type'] === 'تلقائي' ? 'bg-info-100 text-info' : 'bg-primary-100 text-primary'; ?> text-xs rounded-full">
                                <?php echo htmlspecialchars($backup['type']); ?>
                              </span>
                            </td>
                            <td>
                              <div class="flex items-center space-x-2 space-x-reverse">
                                <button type="button" class="restore-backup-btn p-1 text-warning hover:bg-warning/10 rounded-full transition-all" title="استعادة النسخة الاحتياطية" data-id="<?php echo $backup['id']; ?>">
                                  <i class="fas fa-undo-alt"></i>
                                </button>
                                <button type="button" class="download-backup-btn p-1 text-info hover:bg-info/10 rounded-full transition-all" title="تنزيل النسخة الاحتياطية" data-id="<?php echo $backup['id']; ?>">
                                  <i class="fas fa-download"></i>
                                </button>
                                <button type="button" class="delete-backup-btn p-1 text-danger hover:bg-danger/10 rounded-full transition-all" title="حذف النسخة الاحتياطية" data-id="<?php echo $backup['id']; ?>">
                                  <i class="fas fa-trash-alt"></i>
                                </button>
                              </div>
                            </td>
                          </tr>
                          <?php endforeach; ?>
                        <?php endif; ?>
                      </tbody>
                    </table>
                  </div>
                </div>

                <div>
                  <h4 class="text-lg font-semibold mb-4">إعدادات النسخ الاحتياطي التلقائي</h4>

                  <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div class="form-group">
                      <label for="auto_backup" class="form-label">النسخ الاحتياطي التلقائي</label>
                      <select id="auto_backup" name="auto_backup" class="form-select">
                        <option value="daily">يومي</option>
                        <option value="weekly" selected>أسبوعي</option>
                        <option value="monthly">شهري</option>
                        <option value="disabled">معطل</option>
                      </select>
                    </div>

                    <div class="form-group">
                      <label for="backup_retention" class="form-label">مدة الاحتفاظ بالنسخ الاحتياطية</label>
                      <select id="backup_retention" name="backup_retention" class="form-select">
                        <option value="7">7 أيام</option>
                        <option value="14">14 يوم</option>
                        <option value="30" selected>30 يوم</option>
                        <option value="90">90 يوم</option>
                        <option value="180">180 يوم</option>
                        <option value="365">سنة</option>
                      </select>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </form>
        </div>
      </div>
    </main>
  </div>

  <!-- نموذج إضافة/تعديل دور -->
  <div id="roleModal" class="modal">
    <div class="modal-content">
      <div class="p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-xl font-semibold text-gray-800" id="roleModalTitle">إضافة دور جديد</h3>
          <button id="closeRoleModal" class="text-gray-400 hover:text-gray-600">
            <i class="fas fa-times text-xl"></i>
          </button>
        </div>

        <form id="roleForm">
          <input type="hidden" id="roleId" value="">

          <div class="form-group">
            <label for="roleName" class="form-label">اسم الدور</label>
            <input type="text" id="roleName" name="role_name" class="form-input" required>
          </div>

          <div class="form-group">
            <label for="roleDisplayName" class="form-label">الاسم المعروض</label>
            <input type="text" id="roleDisplayName" name="role_display_name" class="form-input" required>
          </div>

          <div class="form-group">
            <label class="form-label">الصلاحيات</label>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-2 mt-2">
              <?php foreach ($permissionSettings['permissions'] as $permission): ?>
              <label class="flex items-center space-x-2 space-x-reverse">
                <input type="checkbox" name="permissions[]" value="<?php echo $permission['name']; ?>" class="form-checkbox">
                <span><?php echo htmlspecialchars($permission['display_name']); ?></span>
              </label>
              <?php endforeach; ?>
            </div>
          </div>

          <div class="flex justify-end mt-6">
            <button type="button" id="cancelRoleBtn" class="btn btn-secondary ml-2">إلغاء</button>
            <button type="submit" class="btn btn-primary">حفظ</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- نموذج إضافة/تعديل وحدة -->
  <div id="moduleModal" class="modal">
    <div class="modal-content">
      <div class="p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-xl font-semibold text-gray-800" id="moduleModalTitle">إضافة وحدة جديدة</h3>
          <button id="closeModuleModal" class="text-gray-400 hover:text-gray-600">
            <i class="fas fa-times text-xl"></i>
          </button>
        </div>

        <form id="moduleForm">
          <input type="hidden" id="moduleId" value="">

          <div class="form-group">
            <label for="moduleName" class="form-label">اسم الوحدة (بالإنجليزية)</label>
            <input type="text" id="moduleName" name="module_name" class="form-input" required>
          </div>

          <div class="form-group">
            <label for="moduleDisplayName" class="form-label">الاسم المعروض</label>
            <input type="text" id="moduleDisplayName" name="module_display_name" class="form-input" required>
          </div>

          <div class="form-group">
            <label for="moduleOrder" class="form-label">الترتيب</label>
            <input type="number" id="moduleOrder" name="module_order" class="form-input" min="1" required>
          </div>

          <div class="form-group">
            <label for="moduleStatus" class="form-label">الحالة</label>
            <select id="moduleStatus" name="module_status" class="form-select" required>
              <option value="enabled">مفعل</option>
              <option value="disabled">معطل</option>
            </select>
          </div>

          <div class="flex justify-end mt-6">
            <button type="button" id="cancelModuleBtn" class="btn btn-secondary ml-2">إلغاء</button>
            <button type="submit" class="btn btn-primary">حفظ</button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- نافذة إضافة/تعديل الدور -->
  <div id="roleModal" class="modal">
    <div class="modal-content animate-fadeInUp">
      <div class="settings-card-header">
        <h3 class="settings-card-title" id="roleModalTitle">
          <i class="fas fa-user-tag"></i>
          <span>إضافة دور جديد</span>
        </h3>
        <button type="button" id="closeRoleModal" class="btn btn-flat-danger btn-sm">
          <i class="fas fa-times"></i>
        </button>
      </div>
      <div class="settings-card-body">
        <form id="roleForm" method="post">
          <input type="hidden" name="action" value="save_permissions">
          <input type="hidden" name="role_id" id="roleId" value="">

          <div class="form-group">
            <label for="roleName" class="form-label">اسم الدور</label>
            <input type="text" id="roleName" name="new_role_name" class="form-input" placeholder="مثال: editor" required>
            <small class="text-gray-500">يجب أن يكون اسم الدور فريدًا ويتكون من أحرف إنجليزية وأرقام وشرطات سفلية فقط</small>
          </div>

          <div class="form-group">
            <label for="roleDisplayName" class="form-label">الاسم المعروض</label>
            <input type="text" id="roleDisplayName" name="new_role_display_name" class="form-input" placeholder="مثال: محرر" required>
          </div>

          <div class="form-group">
            <label for="roleDescription" class="form-label">وصف الدور</label>
            <textarea id="roleDescription" name="new_role_description" class="form-textarea" rows="3" placeholder="وصف مختصر للدور وصلاحياته"></textarea>
          </div>

          <div class="form-group">
            <label class="form-label">الصلاحيات</label>
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-2">
              <?php foreach ($permissionSettings['permissions'] as $permission): ?>
              <div class="flex items-center">
                <input type="checkbox" id="perm_<?php echo $permission['id']; ?>" name="role_permissions[]" value="<?php echo $permission['id']; ?>" class="ml-2">
                <label for="perm_<?php echo $permission['id']; ?>" class="text-sm"><?php echo htmlspecialchars($permission['display_name']); ?></label>
              </div>
              <?php endforeach; ?>
            </div>
          </div>
        </form>
      </div>
      <div class="settings-card-footer">
        <button type="button" id="cancelRoleBtn" class="btn btn-secondary ml-2">إلغاء</button>
        <button type="submit" form="roleForm" class="btn btn-primary">حفظ</button>
      </div>
    </div>
  </div>

  <!-- نافذة إضافة/تعديل الصلاحية -->
  <div id="permissionModal" class="modal">
    <div class="modal-content animate-fadeInUp">
      <div class="settings-card-header">
        <h3 class="settings-card-title" id="permissionModalTitle">
          <i class="fas fa-key"></i>
          <span>إضافة صلاحية جديدة</span>
        </h3>
        <button type="button" id="closePermissionModal" class="btn btn-flat-danger btn-sm">
          <i class="fas fa-times"></i>
        </button>
      </div>
      <div class="settings-card-body">
        <form id="permissionForm" method="post">
          <input type="hidden" name="action" value="save_permissions">
          <input type="hidden" name="permission_id" id="permissionId" value="">

          <div class="form-group">
            <label for="permissionName" class="form-label">اسم الصلاحية</label>
            <input type="text" id="permissionName" name="new_permission_name" class="form-input" placeholder="مثال: edit_posts" required>
            <small class="text-gray-500">يجب أن يكون اسم الصلاحية فريدًا ويتكون من أحرف إنجليزية وأرقام وشرطات سفلية فقط</small>
          </div>

          <div class="form-group">
            <label for="permissionDisplayName" class="form-label">الاسم المعروض</label>
            <input type="text" id="permissionDisplayName" name="new_permission_display_name" class="form-input" placeholder="مثال: تعديل المنشورات" required>
          </div>

          <div class="form-group">
            <label for="permissionCategory" class="form-label">الفئة</label>
            <select id="permissionCategory" name="new_permission_category" class="form-select">
              <option value="general">عام</option>
              <option value="dashboard">لوحة التحكم</option>
              <option value="orders">الطلبات</option>
              <option value="users">المستخدمين</option>
              <option value="branches">الفروع</option>
              <option value="reports">التقارير</option>
              <option value="settings">الإعدادات</option>
              <option value="data">البيانات</option>
            </select>
          </div>

          <div class="form-group">
            <label for="permissionDescription" class="form-label">وصف الصلاحية</label>
            <textarea id="permissionDescription" name="new_permission_description" class="form-textarea" rows="3" placeholder="وصف مختصر للصلاحية"></textarea>
          </div>
        </form>
      </div>
      <div class="settings-card-footer">
        <button type="button" id="cancelPermissionBtn" class="btn btn-secondary ml-2">إلغاء</button>
        <button type="submit" form="permissionForm" class="btn btn-primary">حفظ</button>
      </div>
    </div>
  </div>

  <!-- سكريبت القائمة الجانبية للأجهزة المحمولة -->
  <script src="https://cdn.jsdelivr.net/npm/tippy.js@6.3.7/dist/tippy.umd.min.js"></script>
  <script>
    // تهيئة التلميحات
    document.addEventListener('DOMContentLoaded', function() {
      tippy('[data-tippy-content]', {
        placement: 'top',
        arrow: true,
        theme: 'light',
        duration: [300, 200],
        animation: 'scale'
      });
    });

    // تبديل القائمة الجانبية للأجهزة المحمولة
    const menuBtn = document.getElementById('menuBtn');
    const sidebar = document.querySelector('aside');
    const overlay = document.getElementById('overlay');

    if (menuBtn && sidebar && overlay) {
      menuBtn.addEventListener('click', () => {
        sidebar.classList.toggle('hidden');
        sidebar.classList.toggle('fixed');
        sidebar.classList.toggle('inset-0');
        sidebar.classList.toggle('z-40');
        sidebar.classList.toggle('w-64');
        overlay.classList.toggle('hidden');
      });

      overlay.addEventListener('click', () => {
        sidebar.classList.add('hidden');
        sidebar.classList.remove('fixed', 'inset-0', 'z-40', 'w-64');
        overlay.classList.add('hidden');
      });
    }

    // تبديل القوائم المنسدلة
    const dropdownToggles = document.querySelectorAll('.dropdown-toggle');

    dropdownToggles.forEach(toggle => {
      toggle.addEventListener('click', (e) => {
        e.preventDefault();
        const menuId = toggle.getAttribute('id');
        const menu = document.getElementById(menuId.replace('Dropdown', 'Menu'));

        if (menu) {
          menu.classList.toggle('active');
        }
      });
    });

    // تبديل علامات التبويب
    const tabs = document.querySelectorAll('.tab');
    const tabContents = document.querySelectorAll('.tab-content');

    tabs.forEach(tab => {
      tab.addEventListener('click', () => {
        const tabId = tab.getAttribute('data-tab');

        // إزالة الفئة النشطة من جميع علامات التبويب
        tabs.forEach(t => t.classList.remove('active'));

        // إضافة الفئة النشطة إلى علامة التبويب المحددة
        tab.classList.add('active');

        // إخفاء جميع محتويات التبويب
        tabContents.forEach(content => content.classList.remove('active'));

        // إظهار محتوى التبويب المحدد
        document.getElementById(`${tabId}-content`).classList.add('active');
      });
    });

    // الإجراءات السريعة
    const quickActionGeneral = document.getElementById('quickActionGeneral');
    const quickActionBackup = document.getElementById('quickActionBackup');

    if (quickActionGeneral) {
      quickActionGeneral.addEventListener('click', () => {
        // تنشيط تبويب الإعدادات العامة
        const generalTab = document.querySelector('.tab[data-tab="general"]');
        if (generalTab) {
          generalTab.click();
        }
      });
    }



    if (quickActionBackup) {
      quickActionBackup.addEventListener('click', () => {
        // تنشيط تبويب النسخ الاحتياطي
        const backupTab = document.querySelector('.tab[data-tab="backup"]');
        if (backupTab) {
          backupTab.click();
        }
      });
    }

    // شريط البحث
    const settingsSearch = document.getElementById('settingsSearch');

    if (settingsSearch) {
      settingsSearch.addEventListener('input', (e) => {
        const searchTerm = e.target.value.toLowerCase();

        // البحث في عناوين البطاقات
        const cardTitles = document.querySelectorAll('.settings-card-title');
        const formLabels = document.querySelectorAll('.form-label');
        const permissionCards = document.querySelectorAll('.permissions-matrix-label');

        // إعادة تعيين جميع البطاقات
        document.querySelectorAll('.settings-card').forEach(card => {
          card.style.display = 'block';
        });

        if (searchTerm.length > 0) {
          // البحث في عناوين البطاقات
          let foundInCards = false;
          cardTitles.forEach(title => {
            const card = title.closest('.settings-card');
            const titleText = title.textContent.toLowerCase();

            if (titleText.includes(searchTerm)) {
              foundInCards = true;
              card.style.display = 'block';

              // تمييز النص المطابق
              const span = title.querySelector('span');
              if (span) {
                const originalText = span.textContent;
                const highlightedText = originalText.replace(
                  new RegExp(searchTerm, 'gi'),
                  match => `<mark class="bg-yellow-200">${match}</mark>`
                );
                span.innerHTML = highlightedText;
              }
            } else {
              card.style.display = 'none';
            }
          });

          // إذا لم يتم العثور على نتائج في العناوين، ابحث في التسميات
          if (!foundInCards) {
            formLabels.forEach(label => {
              const labelText = label.textContent.toLowerCase();

              if (labelText.includes(searchTerm)) {
                const card = label.closest('.settings-card');
                if (card) {
                  card.style.display = 'block';
                }
              }
            });

            permissionCards.forEach(label => {
              const labelText = label.textContent.toLowerCase();

              if (labelText.includes(searchTerm)) {
                const card = label.closest('.settings-card');
                if (card) {
                  card.style.display = 'block';
                }
              }
            });
          }
        }
      });
    }

    // زر حفظ الإعدادات
    const saveSettingsBtn = document.getElementById('saveSettingsBtn');

    if (saveSettingsBtn) {
      saveSettingsBtn.addEventListener('click', () => {
        // الحصول على التبويب النشط
        const activeTab = document.querySelector('.tab.active');
        const activeTabId = activeTab.getAttribute('data-tab');

        // تحديد النموذج المناسب للتبويب النشط
        let form;

        switch (activeTabId) {
          case 'general':
            form = document.getElementById('generalForm');
            break;
          case 'appearance':
            form = document.getElementById('appearanceForm');
            break;

          case 'modules':
            form = document.getElementById('modulesForm');
            break;
          case 'notifications':
            form = document.getElementById('notificationsForm');
            break;
          case 'backup':
            form = document.getElementById('backupForm');
            break;
        }

        // إرسال النموذج
        if (form) {
          // إظهار إشعار بالحفظ
          showNotification('جاري حفظ الإعدادات...', 'success');

          // إرسال النموذج بعد تأخير قصير
          setTimeout(() => {
            form.submit();
          }, 500);
        }
      });
    }



    // ترتيب الوحدات
    const moveUpBtns = document.querySelectorAll('.move-up-btn');
    const moveDownBtns = document.querySelectorAll('.move-down-btn');

    moveUpBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        const row = btn.closest('tr');
        const prevRow = row.previousElementSibling;

        if (prevRow) {
          // تبديل الصفوف
          row.parentNode.insertBefore(row, prevRow);

          // تحديث الترتيب
          updateModuleOrder();
        }
      });
    });

    moveDownBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        const row = btn.closest('tr');
        const nextRow = row.nextElementSibling;

        if (nextRow) {
          // تبديل الصفوف
          row.parentNode.insertBefore(nextRow, row);

          // تحديث الترتيب
          updateModuleOrder();
        }
      });
    });

    function updateModuleOrder() {
      const rows = document.querySelectorAll('#modulesTableBody tr');

      rows.forEach((row, index) => {
        const orderInput = row.querySelector('input[name^="module_order"]');
        const orderSpan = row.querySelector('span');

        if (orderInput && orderSpan) {
          orderInput.value = index + 1;
          orderSpan.textContent = index + 1;
        }
      });
    }

    // دالة لإظهار الإشعارات
    function showNotification(message, type = 'success') {
      // إنشاء عنصر الإشعار
      const notification = document.createElement('div');
      notification.className = `notification notification-${type}`;

      // إنشاء محتوى الإشعار
      notification.innerHTML = `
        <div class="notification-icon">
          <i class="fas ${type === 'success' ? 'fa-check' : 'fa-exclamation'}"></i>
        </div>
        <div>
          <h3 class="font-semibold">${type === 'success' ? 'تم بنجاح' : 'خطأ'}</h3>
          <p>${message}</p>
        </div>
      `;

      // إضافة الإشعار إلى الصفحة
      document.body.appendChild(notification);

      // إظهار الإشعار بعد تأخير قصير
      setTimeout(() => {
        notification.classList.add('active');
      }, 100);

      // إخفاء الإشعار بعد 3 ثوانٍ
      setTimeout(() => {
        notification.classList.remove('active');

        // إزالة الإشعار من الصفحة بعد انتهاء التأثير
        setTimeout(() => {
          document.body.removeChild(notification);
        }, 300);
      }, 3000);
    }





        // إرسال طلب AJAX للحصول على صلاحيات الدور
        fetch(`ajax/get_role_permissions.php?id=${roleId}`)
          .then(response => response.json())
          .then(data => {
            if (data.status === 'success') {
              // إنشاء قائمة بالصلاحيات
              let permissionsList = '';
              data.permissions.forEach(permission => {
                permissionsList += `<li class="mb-2">
                  <div class="flex items-center">
                    <div class="w-6 h-6 rounded-full bg-primary-100 flex items-center justify-center text-primary ml-2">
                      <i class="fas fa-key text-xs"></i>
                    </div>
                    <span class="font-medium">${permission.display_name}</span>
                  </div>
                  <div class="text-xs text-gray-500 mr-8">${permission.description || 'لا يوجد وصف'}</div>
                </li>`;
              });

              // إنشاء نافذة منبثقة لعرض الصلاحيات
              const modal = document.createElement('div');
              modal.className = 'modal active';
              modal.innerHTML = `
                <div class="modal-content animate-fadeInUp" style="max-width: 500px;">
                  <div class="settings-card-header">
                    <h3 class="settings-card-title">
                      <i class="fas fa-user-shield"></i>
                      <span>صلاحيات الدور: ${roleName}</span>
                    </h3>
                    <button type="button" class="btn btn-flat-danger btn-sm close-modal">
                      <i class="fas fa-times"></i>
                    </button>
                  </div>
                  <div class="settings-card-body">
                    <p class="mb-4">يمتلك هذا الدور الصلاحيات التالية:</p>
                    <ul class="list-none">
                      ${permissionsList || '<li>لا توجد صلاحيات</li>'}
                    </ul>
                  </div>
                  <div class="settings-card-footer">
                    <button type="button" class="btn btn-primary close-modal">إغلاق</button>
                  </div>
                </div>
              `;

              // إضافة النافذة إلى الصفحة
              document.body.appendChild(modal);

              // إضافة حدث لإغلاق النافذة
              modal.querySelectorAll('.close-modal').forEach(btn => {
                btn.addEventListener('click', () => {
                  modal.classList.remove('active');
                  setTimeout(() => {
                    document.body.removeChild(modal);
                  }, 300);
                });
              });
            } else {
              showNotification('حدث خطأ أثناء تحميل صلاحيات الدور', 'error');
            }
          })
          .catch(error => {
            console.error('Error:', error);
            showNotification('حدث خطأ أثناء تحميل صلاحيات الدور', 'error');

            // في حالة عدم وجود ملف AJAX، نعرض الصلاحيات من الصفحة
            const permissionBadges = btn.closest('tr').querySelector('td:nth-child(3)').querySelectorAll('.badge');
            let permissionsList = '';

            permissionBadges.forEach(badge => {
              permissionsList += `<li class="mb-2">
                <div class="flex items-center">
                  <div class="w-6 h-6 rounded-full bg-primary-100 flex items-center justify-center text-primary ml-2">
                    <i class="fas fa-key text-xs"></i>
                  </div>
                  <span class="font-medium">${badge.textContent.trim()}</span>
                </div>
              </li>`;
            });

            // إنشاء نافذة منبثقة لعرض الصلاحيات
            const modal = document.createElement('div');
            modal.className = 'modal active';
            modal.innerHTML = `
              <div class="modal-content animate-fadeInUp" style="max-width: 500px;">
                <div class="settings-card-header">
                  <h3 class="settings-card-title">
                    <i class="fas fa-user-shield"></i>
                    <span>صلاحيات الدور: ${roleName}</span>
                  </h3>
                  <button type="button" class="btn btn-flat-danger btn-sm close-modal">
                    <i class="fas fa-times"></i>
                  </button>
                </div>
                <div class="settings-card-body">
                  <p class="mb-4">يمتلك هذا الدور الصلاحيات التالية:</p>
                  <ul class="list-none">
                    ${permissionsList || '<li>لا توجد صلاحيات</li>'}
                  </ul>
                </div>
                <div class="settings-card-footer">
                  <button type="button" class="btn btn-primary close-modal">إغلاق</button>
                </div>
              </div>
            `;

            // إضافة النافذة إلى الصفحة
            document.body.appendChild(modal);

            // إضافة حدث لإغلاق النافذة
            modal.querySelectorAll('.close-modal').forEach(btn => {
              btn.addEventListener('click', () => {
                modal.classList.remove('active');
                setTimeout(() => {
                  document.body.removeChild(modal);
                }, 300);
              });
            });
          });
      });
    });

    // تعديل الصلاحية
    editPermissionBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        const permissionId = btn.getAttribute('data-id');
        const permissionName = btn.closest('.bg-white').querySelector('.font-medium').textContent.trim();

        // إظهار إشعار بتحميل البيانات
        showNotification('جاري تحميل بيانات الصلاحية...', 'success');

        // إرسال طلب AJAX للحصول على بيانات الصلاحية
        fetch(`ajax/get_permission.php?id=${permissionId}`)
          .then(response => response.json())
          .then(data => {
            if (data.status === 'success') {
              // تعبئة النموذج ببيانات الصلاحية
              document.getElementById('permissionId').value = data.permission.id;
              document.getElementById('permissionName').value = data.permission.name;
              document.getElementById('permissionName').disabled = true; // لا يمكن تغيير اسم الصلاحية
              document.getElementById('permissionDisplayName').value = data.permission.display_name;
              document.getElementById('permissionCategory').value = data.permission.category;
              document.getElementById('permissionDescription').value = data.permission.description || '';

              // تغيير اسم الحقل لتحديث الصلاحية بدلاً من إضافة صلاحية جديدة
              document.querySelector('#permissionForm input[name="new_permission_name"]').name = 'permission_name';
              document.querySelector('#permissionForm input[name="new_permission_display_name"]').name = 'permission_display_name';
              document.querySelector('#permissionForm select[name="new_permission_category"]').name = 'permission_category';
              document.querySelector('#permissionForm textarea[name="new_permission_description"]').name = 'permission_description';

              // تعيين عنوان النافذة
              document.querySelector('#permissionModalTitle span').textContent = 'تعديل الصلاحية: ' + data.permission.display_name;

              // إظهار النافذة
              permissionModal.classList.add('active');
            } else {
              showNotification('حدث خطأ أثناء تحميل بيانات الصلاحية', 'error');
            }
          })
          .catch(error => {
            console.error('Error:', error);
            showNotification('حدث خطأ أثناء تحميل بيانات الصلاحية', 'error');

            // في حالة عدم وجود ملف AJAX، نقوم بتعبئة النموذج بالبيانات المتاحة
            document.getElementById('permissionId').value = permissionId;
            document.getElementById('permissionName').value = '';
            document.getElementById('permissionName').disabled = true;
            document.getElementById('permissionDisplayName').value = permissionName;
            document.getElementById('permissionCategory').value = 'general';
            document.getElementById('permissionDescription').value = '';

            // تغيير اسم الحقل لتحديث الصلاحية بدلاً من إضافة صلاحية جديدة
            document.querySelector('#permissionForm input[name="new_permission_name"]').name = 'permission_name';
            document.querySelector('#permissionForm input[name="new_permission_display_name"]').name = 'permission_display_name';
            document.querySelector('#permissionForm select[name="new_permission_category"]').name = 'permission_category';
            document.querySelector('#permissionForm textarea[name="new_permission_description"]').name = 'permission_description';

            // تعيين عنوان النافذة
            document.querySelector('#permissionModalTitle span').textContent = 'تعديل الصلاحية: ' + permissionName;

            // إظهار النافذة
            permissionModal.classList.add('active');
          });
      });
    });

    // حذف الصلاحية
    deletePermissionBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        const permissionId = btn.getAttribute('data-id');
        const permissionName = btn.closest('.bg-white').querySelector('.font-medium').textContent.trim();

        if (confirm(`هل أنت متأكد من حذف الصلاحية "${permissionName}"؟`)) {
          // إنشاء نموذج لإرسال طلب حذف الصلاحية
          const form = document.createElement('form');
          form.method = 'POST';
          form.style.display = 'none';

          const actionInput = document.createElement('input');
          actionInput.type = 'hidden';
          actionInput.name = 'action';
          actionInput.value = 'save_permissions';

          const permissionIdInput = document.createElement('input');
          permissionIdInput.type = 'hidden';
          permissionIdInput.name = 'delete_permission_id';
          permissionIdInput.value = permissionId;

          form.appendChild(actionInput);
          form.appendChild(permissionIdInput);
          document.body.appendChild(form);

          // إرسال النموذج
          form.submit();
        }
      });
    });

    // إضافة دور جديد
    const addRoleBtn = document.getElementById('addRoleBtn');
    if (addRoleBtn) {
      addRoleBtn.addEventListener('click', () => {
        // إعادة تعيين النموذج
        const roleForm = document.getElementById('roleForm');
        if (roleForm) {
          roleForm.reset();
        }

        // التأكد من أن أسماء الحقول صحيحة لإضافة دور جديد
        const nameInput = document.querySelector('#roleForm input[name="role_name"]');
        if (nameInput) nameInput.name = 'new_role_name';

        const displayNameInput = document.querySelector('#roleForm input[name="role_display_name"]');
        if (displayNameInput) displayNameInput.name = 'new_role_display_name';

        const descriptionTextarea = document.querySelector('#roleForm textarea[name="role_description"]');
        if (descriptionTextarea) descriptionTextarea.name = 'new_role_description';

        // تمكين حقل اسم الدور
        document.getElementById('roleName').disabled = false;

        // تعيين معرف الدور
        document.getElementById('roleId').value = '';

        // تعيين عنوان النافذة
        document.querySelector('#roleModalTitle span').textContent = 'إضافة دور جديد';

        // إظهار النافذة
        roleModal.classList.add('active');
      });
    }

    // إضافة صلاحية جديدة
    const addPermissionBtn = document.getElementById('addPermissionBtn');
    if (addPermissionBtn) {
      addPermissionBtn.addEventListener('click', () => {
        // إعادة تعيين النموذج
        const permissionForm = document.getElementById('permissionForm');
        if (permissionForm) {
          permissionForm.reset();
        }

        // التأكد من أن أسماء الحقول صحيحة لإضافة صلاحية جديدة
        const nameInput = document.querySelector('#permissionForm input[name="permission_name"]');
        if (nameInput) nameInput.name = 'new_permission_name';

        const displayNameInput = document.querySelector('#permissionForm input[name="permission_display_name"]');
        if (displayNameInput) displayNameInput.name = 'new_permission_display_name';

        const categorySelect = document.querySelector('#permissionForm select[name="permission_category"]');
        if (categorySelect) categorySelect.name = 'new_permission_category';

        const descriptionTextarea = document.querySelector('#permissionForm textarea[name="permission_description"]');
        if (descriptionTextarea) descriptionTextarea.name = 'new_permission_description';

        // تمكين حقل اسم الصلاحية
        document.getElementById('permissionName').disabled = false;

        // تعيين معرف الصلاحية
        document.getElementById('permissionId').value = '';

        // تعيين عنوان النافذة
        document.querySelector('#permissionModalTitle span').textContent = 'إضافة صلاحية جديدة';

        // إظهار النافذة
        permissionModal.classList.add('active');
      });
    }

    // إغلاق النوافذ المنبثقة
    const closeRoleModal = document.getElementById('closeRoleModal');
    const cancelRoleBtn = document.getElementById('cancelRoleBtn');

    if (closeRoleModal && roleModal) {
      closeRoleModal.addEventListener('click', () => {
        roleModal.classList.remove('active');
      });
    }

    if (cancelRoleBtn && roleModal) {
      cancelRoleBtn.addEventListener('click', () => {
        roleModal.classList.remove('active');
      });
    }

    const closePermissionModal = document.getElementById('closePermissionModal');
    const cancelPermissionBtn = document.getElementById('cancelPermissionBtn');

    if (closePermissionModal && permissionModal) {
      closePermissionModal.addEventListener('click', () => {
        permissionModal.classList.remove('active');
      });
    }

    if (cancelPermissionBtn && permissionModal) {
      cancelPermissionBtn.addEventListener('click', () => {
        permissionModal.classList.remove('active');
      });
    }

    // معالجة نموذج الدور
    const roleForm = document.getElementById('roleForm');
    if (roleForm) {
      roleForm.addEventListener('submit', (e) => {
        e.preventDefault();

        // إظهار إشعار بحفظ البيانات
        showNotification('جاري حفظ بيانات الدور...', 'success');

        // إرسال النموذج
        roleForm.submit();
      });
    }

    // معالجة نموذج الصلاحية
    const permissionForm = document.getElementById('permissionForm');
    if (permissionForm) {
      permissionForm.addEventListener('submit', (e) => {
        e.preventDefault();

        // إظهار إشعار بحفظ البيانات
        showNotification('جاري حفظ بيانات الصلاحية...', 'success');

        // إرسال النموذج
        permissionForm.submit();
      });
    }

    // إخفاء الإشعارات تلقائيًا بعد 5 ثوانٍ
    const notifications = document.querySelectorAll('.notification');
    notifications.forEach(notification => {
      setTimeout(() => {
        notification.classList.remove('active');
      }, 5000);
    });
  </script>
</body>
</html>