<?php
/**
 * التقارير الأسبوعية
 * تعرض هذه الصفحة التقارير الأسبوعية للنظام
 * تم ربط الصفحة بقاعدة البيانات الفعلية لعرض البيانات الحقيقية
 *
 * @version 1.1
 * @author فريق التطوير
 * @update تم ربط الصفحة بقاعدة البيانات الفعلية
 */

// تضمين ملف التحقق من تسجيل الدخول
require_once 'check_auth.php';

// ملاحظة: ملف check_auth.php يقوم بتضمين ملف الاتصال بقاعدة البيانات (db_connection.php)
// لذلك لا نحتاج إلى تضمين ملف database.php هنا
// require_once '../config/database.php';

// التحقق من وجود اتصال بقاعدة البيانات
if (!isset($conn) || !$conn) {
    die("خطأ: لم يتم العثور على اتصال بقاعدة البيانات");
}

// تفعيل وضع التصحيح للتحقق من الاستعلامات
// يمكن تغيير هذا المتغير إلى false لتعطيل عرض معلومات التصحيح في الإنتاج
$debug_mode = isset($_GET['debug']) && $_GET['debug'] === '1' ? true : false;

// دالة لعرض معلومات التصحيح
function debug_log($message, $data = null) {
    global $debug_mode;
    if ($debug_mode) {
        echo "<div style='background-color: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px; padding: 10px; margin: 10px 0; direction: ltr; text-align: left;'>";
        echo "<strong>Debug:</strong> " . htmlspecialchars($message) . "<br>";
        if ($data !== null) {
            echo "<pre>" . htmlspecialchars(print_r($data, true)) . "</pre>";
        }
        echo "</div>";
    }
}

// التحقق من وجود بيانات في جدول الطلبات
try {
    $checkOrdersStmt = $conn->query("SELECT COUNT(*) FROM orders");
    $ordersCount = $checkOrdersStmt->fetchColumn();

    // عرض معلومات التصحيح
    debug_log("التحقق من وجود بيانات في جدول الطلبات", [
        'orders_count' => $ordersCount
    ]);

    // إذا لم تكن هناك طلبات، قم بإضافة بعض البيانات التجريبية
    if ($ordersCount == 0) {
        // إضافة بيانات تجريبية للطلبات
        $conn->exec("
            INSERT INTO orders (
                order_number, client_name, service_type, branch_id,
                departure_area, delivery_area, goods_name, goods_type,
                weight, service_fees, amount_paid, remaining_amount,
                order_upload_date, scheduled_delivery_date, recipient_name,
                phone_number, employee_name, amount_agreed_with_customer,
                amount_agreed_with_agent, profit, payment_status, status,
                created_at, updated_at, user_id
            ) VALUES
            ('ORD-001', 'أحمد محمد', 'شحن بحري', 1, 'الصين', 'اليمن', 'ملابس', 'بضائع عامة',
             50.5, 1500, 1000, 500, '".date('Y-m-d', strtotime('-3 days'))."', '".date('Y-m-d', strtotime('+7 days'))."',
             'علي أحمد', '777123456', 'محمد علي', 1500, 1200, 300, 'مدفوع جزئياً', 'قيد المعالجة',
             NOW(), NOW(), 1),

            ('ORD-002', 'محمد علي', 'شحن جوي', 2, 'دبي', 'اليمن', 'إلكترونيات', 'أجهزة',
             20.3, 2000, 2000, 0, '".date('Y-m-d', strtotime('-2 days'))."', '".date('Y-m-d', strtotime('+5 days'))."',
             'خالد محمد', '777654321', 'فاطمة أحمد', 2000, 1800, 200, 'مدفوع', 'قيد التوصيل',
             NOW(), NOW(), 1),

            ('ORD-003', 'فاطمة أحمد', 'شحن بري', 3, 'السعودية', 'اليمن', 'أثاث', 'أثاث منزلي',
             150.0, 3000, 3000, 0, '".date('Y-m-d', strtotime('-5 days'))."', '".date('Y-m-d', strtotime('+2 days'))."',
             'سعيد عمر', '777987654', 'خالد عبدالله', 3000, 2500, 500, 'مدفوع', 'تم التسليم',
             NOW(), NOW(), 1),

            ('ORD-004', 'خالد عبدالله', 'شحن بحري', 4, 'الصين', 'اليمن', 'قطع غيار', 'قطع غيار سيارات',
             80.7, 1800, 1000, 800, '".date('Y-m-d', strtotime('-1 day'))."', '".date('Y-m-d', strtotime('+10 days'))."',
             'محمد سالم', '777456789', 'أحمد محمد', 1800, 1500, 300, 'مدفوع جزئياً', 'قيد المعالجة',
             NOW(), NOW(), 1)
        ");

        echo "<div class='alert alert-info'>تم إضافة بيانات تجريبية للطلبات</div>";
    }
} catch (PDOException $e) {
    error_log("خطأ في التحقق من جدول الطلبات: " . $e->getMessage());
}

/**
 * دالة للتخزين المؤقت للاستعلامات
 * تستخدم هذه الدالة لتخزين نتائج الاستعلامات المتكررة وتحسين الأداء
 *
 * @param string $key مفتاح التخزين المؤقت
 * @param callable $callback دالة لتنفيذ الاستعلام إذا لم يكن موجودًا في التخزين المؤقت
 * @param int $ttl مدة صلاحية التخزين المؤقت بالثواني (الافتراضي: 300 ثانية = 5 دقائق)
 * @return mixed نتيجة الاستعلام
 */
function queryCache($key, $callback, $ttl = 300) {
    $cacheDir = __DIR__ . '/../cache';
    $cacheFile = $cacheDir . '/' . md5($key) . '.cache';

    // إنشاء مجلد التخزين المؤقت إذا لم يكن موجودًا
    if (!file_exists($cacheDir)) {
        mkdir($cacheDir, 0777, true);
    }

    // التحقق مما إذا كان ملف التخزين المؤقت موجودًا وصالحًا
    if (file_exists($cacheFile) && (time() - filemtime($cacheFile) < $ttl)) {
        // استرجاع البيانات من التخزين المؤقت
        $data = file_get_contents($cacheFile);
        return unserialize($data);
    }

    // تنفيذ الاستعلام
    $result = $callback();

    // حفظ النتيجة في التخزين المؤقت
    file_put_contents($cacheFile, serialize($result));

    return $result;
}

// تحديد نوع التقرير (مستخدم أو إدارة)
$report_type = isset($_GET['type']) ? $_GET['type'] : 'user';
$is_admin_report = ($report_type === 'admin');

// التحقق من الصلاحيات للتقارير الإدارية
if ($is_admin_report && (!isset($_SESSION['role']) || ($_SESSION['role'] !== 'admin' && $_SESSION['role'] !== 'manager'))) {
    // إعادة توجيه المستخدم إلى صفحة التقارير العادية
    header('Location: weekly_reports.php?type=user');
    exit;
}

// عنوان الصفحة
$page_title = $is_admin_report ? 'التقارير الأسبوعية للإدارة' : 'التقارير الأسبوعية';

// الحصول على الأسبوع الحالي
$current_week = isset($_GET['week']) ? intval($_GET['week']) : date('W');
$current_year = isset($_GET['year']) ? intval($_GET['year']) : date('Y');

// التحقق من صحة الأسبوع والسنة
if ($current_week < 1 || $current_week > 53) {
    $current_week = date('W');
}
if ($current_year < 2020 || $current_year > 2030) {
    $current_year = date('Y');
}

// الحصول على تاريخ بداية ونهاية الأسبوع
$dto = new DateTime();
$dto->setISODate($current_year, $current_week);
$week_start = $dto->format('Y-m-d');
$dto->modify('+6 days');
$week_end = $dto->format('Y-m-d');

// عرض معلومات التصحيح
debug_log("معلومات الأسبوع", [
    'current_week' => $current_week,
    'current_year' => $current_year,
    'week_start' => $week_start,
    'week_end' => $week_end,
    'today' => date('Y-m-d')
]);

// تنسيق التاريخ بالعربية
function formatArabicDate($date) {
    $months = [
        '01' => 'يناير',
        '02' => 'فبراير',
        '03' => 'مارس',
        '04' => 'أبريل',
        '05' => 'مايو',
        '06' => 'يونيو',
        '07' => 'يوليو',
        '08' => 'أغسطس',
        '09' => 'سبتمبر',
        '10' => 'أكتوبر',
        '11' => 'نوفمبر',
        '12' => 'ديسمبر'
    ];

    $date_parts = explode('-', $date);
    return $date_parts[2] . '/' . $date_parts[1] . '/' . $date_parts[0];
}

// تنسيق فترة الأسبوع
$week_period = formatArabicDate($week_start) . ' إلى ' . formatArabicDate($week_end);

// تحديد معرف المستخدم الحالي
$current_user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;
$current_user_role = isset($_SESSION['role']) ? $_SESSION['role'] : 'user';

// الحصول على بيانات الإحصائيات من قاعدة البيانات
$stats = [
    'total_orders' => 0,
    'total_growth' => 0,
    'completed_orders' => 0,
    'completed_percentage' => 0,
    'completed_growth' => 0,
    'pending_orders' => 0,
    'pending_percentage' => 0,
    'pending_growth' => 0,
    'daily_average' => 0,
    'daily_average_growth' => 0,
    'prev_week_orders' => 0,
    'busiest_days' => []
];

try {
    // استخدام التخزين المؤقت لتحسين الأداء
    $weeklyStats = queryCache("weekly_stats_{$current_week}_{$current_year}_{$current_user_id}", function() use ($conn, $week_start, $week_end, $current_user_id, $current_user_role) {

    // عرض معلومات التصحيح
    debug_log("جاري تنفيذ استعلام إحصائيات الأسبوع", [
        'week_start' => $week_start,
        'week_end' => $week_end,
        'current_user_id' => $current_user_id,
        'current_user_role' => $current_user_role
    ]);
        // استعلام للحصول على إحصائيات الأسبوع الحالي
        // استخدام FORCE INDEX لتحسين الأداء
        $sql = "
            SELECT
                COUNT(*) as total_orders,
                SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as completed_orders,
                SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending_orders,
                SUM(CASE WHEN status = 'في الطريق' THEN 1 ELSE 0 END) as in_transit_orders,
                SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled_orders,
                COUNT(DISTINCT DATE(order_upload_date)) as total_days
            FROM orders
            WHERE order_upload_date BETWEEN ? AND ?
        ";

        // إضافة شرط المستخدم إذا لم يكن مدير
        $params = [$week_start, $week_end];

        // عرض معلومات التصحيح للاستعلام
        debug_log("استعلام الإحصائيات", [
            'week_start' => $week_start,
            'week_end' => $week_end,
            'sql' => $sql
        ]);

        if ($current_user_role !== 'admin' && $current_user_role !== 'manager') {
            $sql .= " AND user_id = ?";
            $params[] = $current_user_id;
        }

        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        // حساب النسب المئوية
        $total_orders = (int)$result['total_orders'];
        $completed_orders = (int)$result['completed_orders'];
        $pending_orders = (int)$result['pending_orders'];
        $in_transit_orders = (int)$result['in_transit_orders'];
        $cancelled_orders = (int)$result['cancelled_orders'];
        $total_days = (int)$result['total_days'] ?: 1; // تجنب القسمة على صفر

        $completed_percentage = $total_orders > 0 ? round(($completed_orders / $total_orders) * 100) : 0;
        $pending_percentage = $total_orders > 0 ? round(($pending_orders / $total_orders) * 100) : 0;
        $daily_average = $total_orders > 0 ? round($total_orders / $total_days, 1) : 0;

        // الحصول على إحصائيات الأسبوع السابق
        $prev_week_start = date('Y-m-d', strtotime('-1 week', strtotime($week_start)));
        $prev_week_end = date('Y-m-d', strtotime('-1 week', strtotime($week_end)));

        $prev_sql = "
            SELECT
                COUNT(*) as total_orders,
                SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as completed_orders,
                SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending_orders,
                COUNT(DISTINCT DATE(order_upload_date)) as total_days
            FROM orders FORCE INDEX (idx_orders_upload_date, idx_orders_status)
            WHERE order_upload_date BETWEEN ? AND ?
        ";

        // إضافة شرط المستخدم إذا لم يكن مدير
        $prev_params = [$prev_week_start, $prev_week_end];

        // عرض معلومات التصحيح للاستعلام
        debug_log("استعلام الأسبوع السابق", [
            'prev_week_start' => $prev_week_start,
            'prev_week_end' => $prev_week_end,
            'sql' => $prev_sql
        ]);

        if ($current_user_role !== 'admin' && $current_user_role !== 'manager') {
            $prev_sql .= " AND user_id = ?";
            $prev_params[] = $current_user_id;
        }

        $prev_stmt = $conn->prepare($prev_sql);
        $prev_stmt->execute($prev_params);
        $prev_result = $prev_stmt->fetch(PDO::FETCH_ASSOC);

        $prev_total_orders = (int)$prev_result['total_orders'];
        $prev_completed_orders = (int)$prev_result['completed_orders'];
        $prev_pending_orders = (int)$prev_result['pending_orders'];
        $prev_total_days = (int)$prev_result['total_days'] ?: 1; // تجنب القسمة على صفر
        $prev_daily_average = $prev_total_orders > 0 ? round($prev_total_orders / $prev_total_days, 1) : 0;

        // حساب نسب النمو
        $total_growth = $prev_total_orders > 0 ? round((($total_orders - $prev_total_orders) / $prev_total_orders) * 100) : 100;
        $completed_growth = $prev_completed_orders > 0 ? round((($completed_orders - $prev_completed_orders) / $prev_completed_orders) * 100) : 100;
        $pending_growth = $prev_pending_orders > 0 ? round((($pending_orders - $prev_pending_orders) / $prev_pending_orders) * 100) : 0;
        $daily_average_growth = $prev_daily_average > 0 ? round((($daily_average - $prev_daily_average) / $prev_daily_average) * 100) : 100;

        // الحصول على أكثر أيام الأسبوع ازدحامًا
        $busiest_days_sql = "
            SELECT
                DAYNAME(order_upload_date) as day_name,
                COUNT(*) as orders_count
            FROM orders FORCE INDEX (idx_orders_upload_date)
            WHERE order_upload_date BETWEEN ? AND ?
        ";

        // إضافة شرط المستخدم إذا لم يكن مدير
        $busiest_params = [$week_start, $week_end];

        // عرض معلومات التصحيح للاستعلام
        debug_log("استعلام أكثر أيام الأسبوع ازدحامًا", [
            'week_start' => $week_start,
            'week_end' => $week_end,
            'sql' => $busiest_days_sql
        ]);

        if ($current_user_role !== 'admin' && $current_user_role !== 'manager') {
            $busiest_days_sql .= " AND user_id = ?";
            $busiest_params[] = $current_user_id;
        }

        $busiest_days_sql .= " GROUP BY DAYNAME(order_upload_date) ORDER BY orders_count DESC LIMIT 3";

        $busiest_stmt = $conn->prepare($busiest_days_sql);
        $busiest_stmt->execute($busiest_params);
        $busiest_days = $busiest_stmt->fetchAll(PDO::FETCH_ASSOC);

        // تحويل أسماء الأيام إلى العربية
        $days_translation = [
            'Sunday' => 'الأحد',
            'Monday' => 'الاثنين',
            'Tuesday' => 'الثلاثاء',
            'Wednesday' => 'الأربعاء',
            'Thursday' => 'الخميس',
            'Friday' => 'الجمعة',
            'Saturday' => 'السبت'
        ];

        $formatted_busiest_days = [];
        foreach ($busiest_days as $day) {
            $formatted_busiest_days[] = [
                'day' => $days_translation[$day['day_name']] ?? $day['day_name'],
                'count' => (int)$day['orders_count']
            ];
        }

        return [
            'total_orders' => $total_orders,
            'total_growth' => $total_growth,
            'completed_orders' => $completed_orders,
            'completed_percentage' => $completed_percentage,
            'completed_growth' => $completed_growth,
            'pending_orders' => $pending_orders,
            'pending_percentage' => $pending_percentage,
            'pending_growth' => $pending_growth,
            'in_transit_orders' => $in_transit_orders,
            'cancelled_orders' => $cancelled_orders,
            'daily_average' => $daily_average,
            'daily_average_growth' => $daily_average_growth,
            'prev_week_orders' => $prev_total_orders,
            'busiest_days' => $formatted_busiest_days
        ];
    }); // استخدام التخزين المؤقت

    // تحديث الإحصائيات
    $stats = $weeklyStats;

    // عرض معلومات التصحيح
    debug_log("نتائج استعلام إحصائيات الأسبوع", $stats);

} catch (PDOException $e) {
    // تسجيل الخطأ
    $error_message = "خطأ في استعلام إحصائيات التقارير الأسبوعية: " . $e->getMessage();
    error_log($error_message);

    // عرض رسالة الخطأ في وضع التصحيح
    debug_log("خطأ في استعلام إحصائيات الأسبوع", [
        'error' => $e->getMessage(),
        'code' => $e->getCode(),
        'query' => isset($sql) ? $sql : 'غير متاح'
    ]);

    // عرض رسالة خطأ للمستخدم
    echo '<div class="alert alert-danger mb-4">
        <i class="fas fa-exclamation-triangle ml-2"></i>
        <strong>خطأ:</strong> حدث خطأ أثناء استرجاع بيانات الإحصائيات. يرجى المحاولة مرة أخرى لاحقًا.
    </div>';
}

// تحديد ما إذا كان المستخدم يمكنه رؤية تقارير الإدارة
$can_view_admin_reports = ($current_user_role === 'admin' || $current_user_role === 'manager');
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام طلبات الشحن | <?php echo $page_title; ?></title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.css">
  <link rel="stylesheet" href="../assets/css/unified-design.css">
  <style>
    /* تصميم الرسوم البيانية */
    .chart-container {
      position: relative;
      height: 300px;
      width: 100%;
      background-color: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      padding: 1rem;
      transition: var(--transition);
    }

    .chart-container:hover {
      box-shadow: var(--shadow-md);
      transform: translateY(-3px);
    }

    /* تصميم البطاقات الإحصائية */
    .stat-card {
      background-color: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      transition: var(--transition);
      position: relative;
      overflow: hidden;
      border: 1px solid var(--gray-200);
    }

    .stat-card:hover {
      transform: translateY(-5px);
      box-shadow: var(--shadow-md);
    }

    .stat-icon {
      width: 48px;
      height: 48px;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      transition: transform 0.3s ease;
    }

    .stat-card:hover .stat-icon {
      transform: scale(1.1);
    }

    .progress-bar {
      height: 6px;
      border-radius: 3px;
      background-color: var(--gray-200);
      margin-top: 8px;
      overflow: hidden;
    }

    .progress-value {
      height: 6px;
      border-radius: 3px;
      transition: width 1s ease-in-out;
    }

    .badge {
      display: inline-flex;
      align-items: center;
      padding: 0.25rem 0.5rem;
      border-radius: 9999px;
      font-size: 0.75rem;
      font-weight: 500;
      box-shadow: var(--shadow-sm);
    }

    .badge-success {
      background-color: var(--success-50);
      color: var(--success);
    }

    .badge-danger {
      background-color: var(--danger-50);
      color: var(--danger);
    }

    .circle-number {
      width: 24px;
      height: 24px;
      border-radius: 50%;
      display: flex;
      align-items: center;
      justify-content: center;
      background-color: var(--primary-100);
      color: var(--primary);
      font-size: 0.75rem;
      font-weight: 600;
      box-shadow: var(--shadow-sm);
    }

    /* تأثيرات الانتقال */
    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(10px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    .fade-in {
      animation: fadeIn 0.5s ease forwards;
    }

    .slide-up {
      animation: fadeIn 0.5s ease forwards;
    }

    .slide-right {
      animation: fadeIn 0.5s ease forwards;
    }

    /* تصميم الجدول */
    .data-table {
      width: 100%;
      border-collapse: separate;
      border-spacing: 0;
    }

    .data-table th {
      background-color: var(--gray-50);
      font-weight: 600;
      text-align: right;
      padding: 0.75rem 1rem;
      border-bottom: 2px solid var(--gray-200);
    }

    .data-table td {
      padding: 0.75rem 1rem;
      border-bottom: 1px solid var(--gray-200);
      vertical-align: middle;
    }

    .data-table tr:last-child td {
      border-bottom: none;
    }

    .data-table tr:hover td {
      background-color: var(--gray-50);
    }
  </style>
</head>

<body class="bg-gray-100">
  <!-- زر القائمة للجوال -->
  <button id="mobileMenuButton" class="fixed top-4 right-4 z-50 bg-primary text-white p-2 rounded-lg shadow-lg md:hidden">
    <i class="fas fa-bars"></i>
  </button>

  <!-- الغطاء المتداخل للقائمة الجانبية في الجوال -->
  <div id="overlay" class="overlay"></div>

  <div class="flex min-h-screen">
    <!-- تضمين القائمة الجانبية المشتركة -->
    <?php include_once '../includes/sidebar.php'; ?>

    <!-- المحتوى الرئيسي -->
    <main class="main-content flex-1 p-6 md:p-8">
      <!-- رأس الصفحة -->
      <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-8 animate-fadeIn" style="animation-delay: 0.1s;">
        <div>
          <h1 class="text-3xl font-bold text-gray-800 flex items-center">
            <span class="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">
              <?php echo $page_title; ?>
            </span>
          </h1>
          <p class="text-gray-500 mt-2 flex items-center">
            <i class="fas fa-calendar-alt ml-2 text-primary"></i>
            <span>للفترة من <?php echo $week_period; ?></span>
          </p>
        </div>

        <div class="mt-4 md:mt-0 animate-fadeIn" style="animation-delay: 0.2s;">
          <div class="flex items-center space-x-2 space-x-reverse">
            <button onclick="window.print()" class="btn btn-outline-primary">
              <i class="fas fa-print ml-2"></i>
              <span>طباعة</span>
            </button>
            <button onclick="exportToPDF()" class="btn btn-outline-primary">
              <i class="fas fa-file-pdf ml-2"></i>
              <span>PDF</span>
            </button>
            <button onclick="exportToExcel()" class="btn btn-outline-primary">
              <i class="fas fa-file-excel ml-2"></i>
              <span>Excel</span>
            </button>
            <?php if ($current_user_role === 'admin'): ?>
            <a href="?<?php echo http_build_query(array_merge($_GET, ['debug' => $debug_mode ? '0' : '1'])); ?>" class="btn <?php echo $debug_mode ? 'btn-danger' : 'btn-outline-secondary'; ?>">
              <i class="fas fa-bug ml-2"></i>
              <span><?php echo $debug_mode ? 'إيقاف وضع التصحيح' : 'تفعيل وضع التصحيح'; ?></span>
            </a>
            <?php endif; ?>
          </div>
        </div>
      </div>

    <!-- اختيار الأسبوع والسنة -->
    <div class="card mb-6 animate-fadeIn" style="animation-delay: 0.3s;">
      <div class="card-header flex items-center">
        <i class="fas fa-filter text-primary ml-2"></i>
        <span>تصفية التقرير</span>
      </div>
      <div class="card-body">
        <div class="flex flex-wrap items-center justify-between">
          <div class="flex flex-wrap items-center gap-4">
            <div>
              <label for="weekSelector" class="block text-gray-500 mb-1 text-sm">الأسبوع</label>
              <select id="weekSelector" class="bg-gray-50 border border-gray-300 text-gray-700 py-2 px-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-primary transition-all duration-300">
                <?php for ($i = 1; $i <= 52; $i++): ?>
                  <option value="<?php echo $i; ?>" <?php echo $i == $current_week ? 'selected' : ''; ?>>
                    الأسبوع <?php echo $i; ?>
                  </option>
                <?php endfor; ?>
              </select>
            </div>
            <div>
              <label for="yearSelector" class="block text-gray-500 mb-1 text-sm">السنة</label>
              <select id="yearSelector" class="bg-gray-50 border border-gray-300 text-gray-700 py-2 px-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-primary transition-all duration-300">
                <?php for ($i = date('Y') - 2; $i <= date('Y') + 2; $i++): ?>
                  <option value="<?php echo $i; ?>" <?php echo $i == $current_year ? 'selected' : ''; ?>>
                    <?php echo $i; ?>
                  </option>
                <?php endfor; ?>
              </select>
            </div>
            <div>
              <label for="branchSelector" class="block text-gray-500 mb-1 text-sm">الفرع</label>
              <select id="branchSelector" class="bg-gray-50 border border-gray-300 text-gray-700 py-2 px-4 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-primary transition-all duration-300">
                <option value="all" selected>جميع الفروع</option>
                <?php
                // الحصول على قائمة الفروع من قاعدة البيانات
                try {
                    $branches = queryCache('branches_list', function() use ($conn) {
                        $stmt = $conn->query("SELECT id, name FROM branches ORDER BY name");
                        return $stmt->fetchAll(PDO::FETCH_ASSOC);
                    }, 3600); // تخزين مؤقت لمدة ساعة

                    // عرض الفروع
                    foreach ($branches as $branch) {
                        $selected = (isset($_GET['branch']) && $_GET['branch'] == $branch['id']) ? 'selected' : '';
                        echo "<option value=\"{$branch['id']}\" {$selected}>{$branch['name']}</option>";
                    }
                } catch (PDOException $e) {
                    error_log("خطأ في استعلام قائمة الفروع: " . $e->getMessage());
                }
                ?>
              </select>
            </div>
          </div>
          <button id="showReportBtn" class="btn btn-primary mt-4 md:mt-0">
            <i class="fas fa-search ml-2"></i>
            <span>عرض التقرير</span>
          </button>
        </div>
      </div>
    </div>

    <!-- حاوية الأخطاء -->
    <div id="errorContainer" class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-6 hidden" role="alert">
      <strong class="font-bold ml-1">خطأ!</strong>
      <span class="block sm:inline">حدث خطأ أثناء جلب البيانات.</span>
    </div>

    <!-- بطاقات الإحصائيات -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
      <!-- إجمالي الطلبات -->
      <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.4s;">
        <div class="flex flex-col">
          <div class="icon-container mb-4" style="background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);">
            <i class="fas fa-box text-xl"></i>
          </div>
          <p class="stat-label">إجمالي الطلبات</p>
          <p id="totalOrders" class="stat-value">0</p>
          <!-- مؤشر التحميل -->
          <div class="loading-indicator flex items-center justify-center py-2" style="display: none;">
            <div class="animate-spin rounded-full h-5 w-5 border-b-2 border-primary mr-2"></div>
            <span class="text-sm text-gray-500">جاري التحميل...</span>
          </div>
          <div class="flex items-center mt-2 text-sm text-green-600">
            <i class="fas fa-arrow-up ml-1"></i>
            <span><?php echo $stats['total_growth']; ?>% زيادة</span>
            <span class="text-gray-400 mr-2 text-xs">مقارنة بالأسبوع السابق</span>
          </div>
          <div class="progress-bar mt-3">
            <div class="progress-value" style="width: 100%; background: linear-gradient(to right, #3b82f6, #1d4ed8);"></div>
          </div>
        </div>
      </div>

      <!-- الطلبات المكتملة -->
      <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.5s;">
        <div class="flex flex-col">
          <div class="icon-container mb-4" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
            <i class="fas fa-check text-xl"></i>
          </div>
          <p class="stat-label">الطلبات المكتملة</p>
          <div class="flex items-baseline">
            <p id="completedOrders" class="stat-value">0</p>
            <span id="completedPercentage" class="text-sm text-gray-500 mr-2">(0%)</span>
          </div>
          <!-- مؤشر التحميل -->
          <div class="loading-indicator flex items-center justify-center py-2" style="display: none;">
            <div class="animate-spin rounded-full h-5 w-5 border-b-2 border-primary mr-2"></div>
            <span class="text-sm text-gray-500">جاري التحميل...</span>
          </div>
          <div class="flex items-center mt-2 text-sm text-green-600">
            <i class="fas fa-arrow-up ml-1"></i>
            <span><?php echo $stats['completed_growth']; ?>% زيادة</span>
            <span class="text-gray-400 mr-2 text-xs">مقارنة بالأسبوع السابق</span>
          </div>
          <div class="progress-bar mt-3">
            <div id="completedProgressBar" class="progress-value" style="width: 0%; background: linear-gradient(to right, #10b981, #059669);"></div>
          </div>
        </div>
      </div>

      <!-- الطلبات المعلقة -->
      <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.6s;">
        <div class="flex flex-col">
          <div class="icon-container mb-4" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
            <i class="fas fa-clock text-xl"></i>
          </div>
          <p class="stat-label">الطلبات المعلقة</p>
          <div class="flex items-baseline">
            <p id="pendingOrders" class="stat-value">0</p>
            <span id="pendingPercentage" class="text-sm text-gray-500 mr-2">(0%)</span>
          </div>
          <!-- مؤشر التحميل -->
          <div class="loading-indicator flex items-center justify-center py-2" style="display: none;">
            <div class="animate-spin rounded-full h-5 w-5 border-b-2 border-primary mr-2"></div>
            <span class="text-sm text-gray-500">جاري التحميل...</span>
          </div>
          <div class="flex items-center mt-2 text-sm text-red-600">
            <i class="fas fa-arrow-down ml-1"></i>
            <span><?php echo abs($stats['pending_growth']); ?>% انخفاض</span>
            <span class="text-gray-400 mr-2 text-xs">مقارنة بالأسبوع السابق</span>
          </div>
          <div class="progress-bar mt-3">
            <div id="pendingProgressBar" class="progress-value" style="width: 0%; background: linear-gradient(to right, #f59e0b, #d97706);"></div>
          </div>
        </div>
      </div>

      <!-- متوسط الطلبات اليومية -->
      <div class="stat-card p-6 animate-fadeIn" style="animation-delay: 0.7s;">
        <div class="flex flex-col">
          <div class="icon-container mb-4" style="background: linear-gradient(135deg, #8b5cf6 0%, #6d28d9 100%);">
            <i class="fas fa-chart-line text-xl"></i>
          </div>
          <p class="stat-label">متوسط الطلبات اليومية</p>
          <div class="flex items-baseline">
            <p id="dailyAverage" class="stat-value">0</p>
            <span class="text-sm text-gray-500 mr-2">طلب/يوم</span>
          </div>
          <!-- مؤشر التحميل -->
          <div class="loading-indicator flex items-center justify-center py-2" style="display: none;">
            <div class="animate-spin rounded-full h-5 w-5 border-b-2 border-primary mr-2"></div>
            <span class="text-sm text-gray-500">جاري التحميل...</span>
          </div>
          <div class="flex items-center mt-2 text-sm text-green-600">
            <i class="fas fa-arrow-up ml-1"></i>
            <span><?php echo $stats['daily_average_growth']; ?>% زيادة</span>
            <span class="text-gray-400 mr-2 text-xs">مقارنة بالأسبوع السابق</span>
          </div>
          <div class="progress-bar mt-3">
            <div class="progress-value" style="width: 80%; background: linear-gradient(to right, #8b5cf6, #6d28d9);"></div>
          </div>
        </div>
      </div>
    </div>





    <!-- جدول طلبات الأسبوع -->
    <div class="card mb-8 animate-fadeIn" style="animation-delay: 1.1s;">
      <div class="card-header flex justify-between items-center">
        <div class="flex items-center">
          <i class="fas fa-list text-primary ml-2"></i>
          <span>طلبات الأسبوع الحالي</span>
        </div>
        <a href="../orders/orders.php" class="text-primary hover:text-primary-dark flex items-center transition-colors duration-300">
          <span>عرض كل الطلبات</span>
          <i class="fas fa-chevron-left mr-1"></i>
        </a>
      </div>
      <div class="card-body">
        <div class="overflow-x-auto">
          <table class="data-table">
            <thead>
              <tr>
                <th>رقم الطلب</th>
                <th>العميل</th>
                <th>التاريخ</th>
                <th>المبلغ</th>
                <th>الموظف المسؤول</th>
                <th>الحالة</th>
                <th>الإجراءات</th>
              </tr>
            </thead>
            <tbody>
              <tr id="loading-row" style="display: none;">
                <td colspan="7" class="text-center py-8">
                  <div class="flex flex-col items-center justify-center text-gray-500">
                    <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-primary mb-3"></div>
                    <p>جاري تحميل الطلبات...</p>
                  </div>
                </td>
              </tr>
              <tr id="no-orders-row" style="<?php echo empty($weekly_orders) ? '' : 'display: none;'; ?>">
                <td colspan="7" class="text-center py-8">
                  <div class="flex flex-col items-center justify-center text-gray-500">
                    <i class="fas fa-inbox text-4xl mb-3 text-gray-300"></i>
                    <p>لا توجد طلبات في هذا الأسبوع</p>
                  </div>
                </td>
              </tr>
              <!-- طلبات الأسبوع الحالي -->
              <?php
              // الحصول على طلبات الأسبوع الحالي من قاعدة البيانات
              try {
                  // تعطيل التخزين المؤقت مؤقتًا للتأكد من أن البيانات تأتي من قاعدة البيانات مباشرة
                  // استخدام التخزين المؤقت لتحسين الأداء
                  $weekly_orders = queryCache("weekly_orders_{$current_week}_{$current_year}_{$current_user_id}", function() use ($conn, $week_start, $week_end, $current_user_id, $current_user_role) {

                  // عرض معلومات التصحيح
                  debug_log("جاري تنفيذ استعلام الطلبات الأسبوعية", [
                      'week_start' => $week_start,
                      'week_end' => $week_end,
                      'current_user_id' => $current_user_id,
                      'current_user_role' => $current_user_role
                  ]);
                      // استعلام للحصول على طلبات الأسبوع الحالي
                      // استخدام FORCE INDEX لتحسين الأداء
                      $sql = "
                          SELECT
                              o.id,
                              o.order_number,
                              o.client_name,
                              o.order_upload_date,
                              o.amount_agreed_with_customer,
                              o.status,
                              o.employee_name,
                              b.name as branch_name
                          FROM
                              orders o
                          LEFT JOIN
                              branches b ON o.branch_id = b.id
                          WHERE
                              o.order_upload_date BETWEEN ? AND ?
                      ";

                      // عرض معلومات التصحيح للتواريخ
                      debug_log("تواريخ الاستعلام", [
                          'week_start' => $week_start,
                          'week_end' => $week_end,
                          'current_date' => date('Y-m-d')
                      ]);

                      // إضافة شرط المستخدم إذا لم يكن مدير
                      $params = [$week_start, $week_end];

                      // عرض معلومات التصحيح للتواريخ
                      debug_log("معلمات الاستعلام", [
                          'week_start' => $week_start,
                          'week_end' => $week_end,
                          'sql' => $sql
                      ]);

                      if ($current_user_role !== 'admin' && $current_user_role !== 'manager') {
                          $sql .= " AND o.user_id = ?";
                          $params[] = $current_user_id;
                      }

                      // إضافة شرط الفرع إذا تم تحديده
                      if (isset($_GET['branch']) && $_GET['branch'] !== 'all') {
                          $sql .= " AND o.branch_id = ?";
                          $params[] = $_GET['branch'];
                      }

                      $sql .= " ORDER BY o.order_upload_date DESC";

                      $stmt = $conn->prepare($sql);
                      $stmt->execute($params);
                      return $stmt->fetchAll(PDO::FETCH_ASSOC);
                  }); // استخدام التخزين المؤقت

                  // عرض معلومات التصحيح
                  debug_log("نتائج استعلام الطلبات الأسبوعية", [
                      'count' => count($weekly_orders),
                      'first_5_orders' => array_slice($weekly_orders, 0, 5)
                  ]);

                  // التحقق من وجود طلبات
                  if (count($weekly_orders) == 0) {
                      echo "
                      <tr>
                          <td colspan='7' class='text-center py-8'>
                              <div class='flex flex-col items-center justify-center space-y-3'>
                                  <i class='fas fa-inbox text-gray-400 text-5xl'></i>
                                  <p class='text-gray-500 text-lg'>لا توجد طلبات في هذا الأسبوع</p>
                                  <p class='text-gray-400 text-sm'>يمكنك تغيير الأسبوع أو السنة للبحث عن طلبات أخرى</p>
                              </div>
                          </td>
                      </tr>
                      ";
                  }

                  // عرض الطلبات
                  foreach ($weekly_orders as $order) {
                      // تحديد لون وأيقونة حالة الطلب
                      $statusColor = '';
                      $statusIcon = '';
                      $statusText = $order['status'];

                      switch($order['status']) {
                          case 'تم التسليم':
                              $statusColor = 'var(--success-50)';
                              $statusIcon = 'fa-check-circle';
                              break;
                          case 'قيد المعالجة':
                              $statusColor = 'var(--warning-50)';
                              $statusIcon = 'fa-clock';
                              break;
                          case 'في الطريق':
                              $statusColor = 'var(--info-50)';
                              $statusIcon = 'fa-truck';
                              break;
                          case 'ملغى':
                              $statusColor = 'var(--danger-50)';
                              $statusIcon = 'fa-times-circle';
                              break;
                          default:
                              $statusColor = 'var(--gray-50)';
                              $statusIcon = 'fa-question-circle';
                      }

                      // تنسيق التاريخ
                      $orderDate = new DateTime($order['order_upload_date']);
                      $formattedDate = $orderDate->format('Y-m-d');

                      // التحقق من صحة التاريخ
                      if ($formattedDate < $week_start || $formattedDate > $week_end) {
                          debug_log("تحذير: تاريخ الطلب خارج نطاق الأسبوع المحدد", [
                              'order_id' => $order['id'],
                              'order_date' => $formattedDate,
                              'week_start' => $week_start,
                              'week_end' => $week_end
                          ]);
                      }

                      // تنسيق المبلغ
                      $amount = floatval($order['amount_agreed_with_customer']) ?: 0;
                      $formattedAmount = number_format($amount, 2) . ' ر.س';

                      // التحقق من صحة المبلغ
                      if ($amount <= 0) {
                          debug_log("تحذير: مبلغ الطلب غير صحيح", [
                              'order_id' => $order['id'],
                              'order_number' => $order['order_number'],
                              'amount' => $amount,
                              'raw_amount' => $order['amount_agreed_with_customer']
                          ]);
                      }

                      // تحضير اسم الموظف
                      $employeeName = 'غير محدد';
                      if (isset($order['employee_name']) && !empty($order['employee_name'])) {
                          $employeeName = $order['employee_name'];
                      }

                      // تحضير لون الحالة
                      $statusColorValue = str_replace('50', '700', $statusColor);

                      // عرض صف الطلب
                      echo "
                      <tr>
                          <td class='font-medium'>{$order['order_number']}</td>
                          <td>{$order['client_name']}</td>
                          <td>{$formattedDate}</td>
                          <td>{$formattedAmount}</td>
                          <td>{$employeeName}</td>
                          <td>
                              <span class='filter-btn' style='background-color: {$statusColor}; color: {$statusColorValue};'>
                                  <i class='fas {$statusIcon} ml-1'></i>
                                  {$statusText}
                              </span>
                          </td>
                          <td>
                              <div class='btn-group'>
                                  <a href='../orders/order-details.php?id={$order['id']}' class='btn btn-flat-info btn-sm'>
                                      <i class='fas fa-eye'></i>
                                  </a>
                                  <a href='../orders/edit-order.php?id={$order['id']}' class='btn btn-flat-primary btn-sm'>
                                      <i class='fas fa-edit'></i>
                                  </a>
                              </div>
                          </td>
                      </tr>
                      ";
                  }
              } catch (PDOException $e) {
                  // تسجيل الخطأ
                  $error_message = "خطأ في استعلام طلبات الأسبوع: " . $e->getMessage();
                  error_log($error_message);

                  // عرض رسالة الخطأ في وضع التصحيح
                  debug_log("خطأ في استعلام الطلبات الأسبوعية", [
                      'error' => $e->getMessage(),
                      'code' => $e->getCode(),
                      'query' => isset($sql) ? $sql : 'غير متاح',
                      'params' => isset($params) ? $params : 'غير متاح'
                  ]);

                  // عرض رسالة خطأ للمستخدم
                  echo "
                  <tr>
                      <td colspan='7' class='text-center py-4'>
                          <div class='alert alert-danger mb-0'>
                              <i class='fas fa-exclamation-triangle ml-2'></i>
                              <strong>خطأ:</strong> حدث خطأ أثناء استرجاع بيانات الطلبات. يرجى المحاولة مرة أخرى لاحقًا.
                          </div>
                      </td>
                  </tr>
                  ";
              }
              ?>
            </tbody>
          </table>
        </div>
      </div>
    </div>
    </main>
  </div>

  <!-- سكريبت التطبيق -->
  <script>
    // متغيرات عامة
    let reportData = null;
    const reportType = '<?php echo $report_type; ?>';
    const currentWeek = <?php echo $current_week; ?>;
    const currentYear = <?php echo $current_year; ?>;
    const currentUserId = <?php echo $current_user_id; ?>;
    const currentUserRole = '<?php echo $current_user_role; ?>';
    const canViewAdminReports = <?php echo $can_view_admin_reports ? 'true' : 'false'; ?>;

    document.addEventListener('DOMContentLoaded', function() {
      // تحديث واجهة المستخدم بالبيانات الموجودة بالفعل
      updateUIFromPHP();

      // تهيئة زر عرض التقرير
      initReportButton();

      // تهيئة تأثيرات التمرير
      initScrollAnimations();

      // تهيئة أزرار التصدير
      initExportButtons();
    });

    // دالة جلب بيانات التقرير
    function fetchReportData() {
      // إظهار مؤشر التحميل
      showLoading(true);

      // الحصول على الفرع المحدد
      const selectedBranch = document.getElementById('branchSelector').value;

      // بناء عنوان URL للطلب
      const url = `../api/reports.php?report_type=weekly&data_type=${reportType}&year=${currentYear}&period=${currentWeek}&branch=${selectedBranch}`;

      // إرسال الطلب
      fetch(url)
        .then(response => {
          if (!response.ok) {
            throw new Error('فشل في جلب بيانات التقرير');
          }
          return response.json();
        })
        .then(data => {
          if (data.status === 'success') {
            // تخزين البيانات
            reportData = data.data;

            // تحديث واجهة المستخدم
            updateUI(reportData);

            // تهيئة الرسوم البيانية
            initCharts(reportData);
          } else {
            throw new Error(data.message || 'فشل في جلب بيانات التقرير');
          }
        })
        .catch(error => {
          console.error('خطأ:', error);
          showError(error.message);
        })
        .finally(() => {
          // إخفاء مؤشر التحميل
          showLoading(false);
        });
    }

    // دالة تحديث واجهة المستخدم من البيانات الموجودة بالفعل
    function updateUIFromPHP() {
      // تحديث بطاقات الإحصائيات
      document.getElementById('totalOrders').textContent = <?php echo $stats['total_orders']; ?>;
      document.getElementById('completedOrders').textContent = <?php echo $stats['completed_orders']; ?>;
      document.getElementById('pendingOrders').textContent = <?php echo $stats['pending_orders']; ?>;
      document.getElementById('dailyAverage').textContent = <?php echo $stats['daily_average']; ?>;

      // تحديث النسب المئوية
      document.getElementById('completedPercentage').textContent = `(<?php echo $stats['completed_percentage']; ?>%)`;
      document.getElementById('pendingPercentage').textContent = `(<?php echo $stats['pending_percentage']; ?>%)`;

      // تحديث شرائط التقدم
      document.getElementById('completedProgressBar').style.width = `<?php echo $stats['completed_percentage']; ?>%`;
      document.getElementById('pendingProgressBar').style.width = `<?php echo $stats['pending_percentage']; ?>%`;
    }

    // دالة تحديث واجهة المستخدم من البيانات المستلمة من API
    function updateUI(data) {
      // تحديث بطاقات الإحصائيات
      document.getElementById('totalOrders').textContent = data.stats.total_orders;
      document.getElementById('completedOrders').textContent = data.stats.completed_orders;
      document.getElementById('pendingOrders').textContent = data.stats.pending_orders;
      document.getElementById('dailyAverage').textContent = data.stats.daily_average;

      // تحديث النسب المئوية
      document.getElementById('completedPercentage').textContent = `(${data.stats.completed_percentage}%)`;
      document.getElementById('pendingPercentage').textContent = `(${data.stats.pending_percentage}%)`;

      // تحديث شرائط التقدم
      document.getElementById('completedProgressBar').style.width = `${data.stats.completed_percentage}%`;
      document.getElementById('pendingProgressBar').style.width = `${data.stats.pending_percentage}%`;

      // تحديث جدول الطلبات
      updateOrdersTable(data.weekly_orders);
    }

    // دالة تحديث جدول الطلبات
    function updateOrdersTable(orders) {
      const tableBody = document.querySelector('.data-table tbody');
      const loadingRow = document.getElementById('loading-row');
      const noOrdersRow = document.getElementById('no-orders-row');

      // إخفاء صف التحميل
      if (loadingRow) {
        loadingRow.style.display = 'none';
      }

      // التحقق من وجود طلبات
      if (!orders || orders.length === 0) {
        if (noOrdersRow) {
          noOrdersRow.style.display = 'table-row';
        }
        return;
      }

      // إخفاء صف "لا توجد طلبات"
      if (noOrdersRow) {
        noOrdersRow.style.display = 'none';
      }

      // إنشاء صفوف الطلبات
      let ordersHTML = '';

      orders.forEach(order => {
        // تحديد لون وأيقونة حالة الطلب
        let statusColor, statusIcon, statusText;

        switch(order.status) {
          case 'تم التسليم':
            statusColor = 'var(--success-50)';
            statusIcon = 'fa-check-circle';
            statusText = 'تم التسليم';
            break;
          case 'قيد المعالجة':
            statusColor = 'var(--warning-50)';
            statusIcon = 'fa-clock';
            statusText = 'قيد المعالجة';
            break;
          case 'في الطريق':
            statusColor = 'var(--info-50)';
            statusIcon = 'fa-truck';
            statusText = 'في الطريق';
            break;
          case 'ملغى':
            statusColor = 'var(--danger-50)';
            statusIcon = 'fa-times-circle';
            statusText = 'ملغى';
            break;
          default:
            statusColor = 'var(--gray-50)';
            statusIcon = 'fa-question-circle';
            statusText = order.status || 'غير معروف';
        }

        // تنسيق التاريخ
        const orderDate = new Date(order.order_upload_date);
        const formattedDate = orderDate.toLocaleDateString('ar-SA');

        // تنسيق المبلغ
        const amount = parseFloat(order.amount_agreed_with_customer) || 0;
        const formattedAmount = amount.toLocaleString('ar-SA') + ' ر.س';

        // إنشاء صف الطلب
        ordersHTML += `
          <tr>
            <td class="font-medium">${order.order_number}</td>
            <td>${order.client_name}</td>
            <td>${formattedDate}</td>
            <td>${formattedAmount}</td>
            <td>${order.employee_name || 'غير محدد'}</td>
            <td>
              <span class="filter-btn" style="background-color: ${statusColor}; color: ${statusColor.replace('50', '700')};">
                <i class="fas ${statusIcon} ml-1"></i>
                ${statusText}
              </span>
            </td>
            <td>
              <div class="btn-group">
                <a href="../orders/order-details.php?id=${order.id}" class="btn btn-flat-info btn-sm">
                  <i class="fas fa-eye"></i>
                </a>
                <a href="../orders/edit-order.php?id=${order.id}" class="btn btn-flat-primary btn-sm">
                  <i class="fas fa-edit"></i>
                </a>
              </div>
            </td>
          </tr>
        `;
      });

      // إضافة الصفوف إلى الجدول
      tableBody.innerHTML = loadingRow.outerHTML + noOrdersRow.outerHTML + ordersHTML;
    }

    // دالة إظهار/إخفاء مؤشر التحميل
    function showLoading(show) {
      const loadingIndicators = document.querySelectorAll('.loading-indicator');
      loadingIndicators.forEach(indicator => {
        indicator.style.display = show ? 'flex' : 'none';
      });

      const contentElements = document.querySelectorAll('.content-element');
      contentElements.forEach(element => {
        element.style.display = show ? 'none' : 'block';
      });
    }

    // دالة إظهار رسالة خطأ
    function showError(message) {
      const errorContainer = document.getElementById('errorContainer');
      if (errorContainer) {
        errorContainer.textContent = message;
        errorContainer.style.display = 'block';
      } else {
        alert(message);
      }
    }

    // تهيئة الرسوم البيانية
    function initCharts() {
      // تم حذف الرسوم البيانية
    }

    // تهيئة زر عرض التقرير
    function initReportButton() {
      const showReportBtn = document.getElementById('showReportBtn');
      const weekSelector = document.getElementById('weekSelector');
      const yearSelector = document.getElementById('yearSelector');
      const branchSelector = document.getElementById('branchSelector');

      if (showReportBtn && weekSelector && yearSelector) {
        showReportBtn.addEventListener('click', function() {
          // إضافة تأثير نقرة الزر
          addRippleEffect(this, event);

          const selectedWeek = weekSelector.value;
          const selectedYear = yearSelector.value;
          const selectedBranch = branchSelector ? branchSelector.value : 'all';
          const reportType = '<?php echo $report_type; ?>';

          // إظهار مؤشر التحميل
          showReportBtn.innerHTML = '<i class="fas fa-spinner fa-spin ml-2"></i> جاري التحميل...';
          showReportBtn.disabled = true;

          // إعادة توجيه إلى نفس الصفحة مع المعلمات الجديدة بعد تأخير قصير
          setTimeout(function() {
            window.location.href = `weekly_reports.php?type=${reportType}&week=${selectedWeek}&year=${selectedYear}&branch=${selectedBranch}`;
          }, 500);
        });
      }

      // تهيئة حقول التصفية
      initFilterFields();
    }

    // تهيئة حقول التصفية
    function initFilterFields() {
      const weekSelector = document.getElementById('weekSelector');
      const yearSelector = document.getElementById('yearSelector');
      const branchSelector = document.getElementById('branchSelector');

      // إضافة تأثيرات التغيير للحقول
      [weekSelector, yearSelector, branchSelector].forEach(selector => {
        if (selector) {
          selector.addEventListener('change', function() {
            this.classList.add('ring-2');
            setTimeout(() => {
              this.classList.remove('ring-2');
            }, 300);
          });
        }
      });
    }

    // تهيئة تأثيرات التمرير
    function initScrollAnimations() {
      // تحديد العناصر التي تحتاج إلى تأثيرات التمرير
      const animatedElements = document.querySelectorAll('.slide-up, .fade-in, .slide-right');

      // تهيئة مراقب التمرير
      const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
          if (entry.isIntersecting) {
            entry.target.style.opacity = '1';
            entry.target.style.transform = 'translateY(0)';
            observer.unobserve(entry.target);
          }
        });
      }, {
        threshold: 0.1
      });

      // إضافة العناصر إلى المراقب
      animatedElements.forEach(element => {
        element.style.opacity = '0';
        element.style.transform = 'translateY(20px)';
        element.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
        observer.observe(element);
      });
    }

    // تهيئة أزرار التصدير
    function initExportButtons() {
      // تصدير إلى PDF
      const pdfButton = document.querySelector('button[onclick="exportToPDF()"]');
      if (pdfButton) {
        pdfButton.onclick = function(event) {
          event.preventDefault();
          addRippleEffect(this, event);
          exportToPDF();
        };
      }

      // تصدير إلى Excel
      const excelButton = document.querySelector('button i.fa-file-excel').closest('button');
      if (excelButton) {
        excelButton.onclick = function(event) {
          event.preventDefault();
          addRippleEffect(this, event);
          exportToExcel();
        };
      }

      // طباعة التقرير
      const printButton = document.querySelector('button[onclick="window.print()"]');
      if (printButton) {
        printButton.onclick = function(event) {
          event.preventDefault();
          addRippleEffect(this, event);
          setTimeout(() => {
            window.print();
          }, 300);
        };
      }
    }

    // إضافة تأثير نقرة الزر
    function addRippleEffect(button, event) {
      const ripple = document.createElement('span');
      const rect = button.getBoundingClientRect();

      const x = event.clientX - rect.left;
      const y = event.clientY - rect.top;

      ripple.style.position = 'absolute';
      ripple.style.width = '5px';
      ripple.style.height = '5px';
      ripple.style.borderRadius = '50%';
      ripple.style.backgroundColor = 'rgba(255, 255, 255, 0.5)';
      ripple.style.left = x + 'px';
      ripple.style.top = y + 'px';
      ripple.style.transform = 'scale(0)';
      ripple.style.animation = 'ripple 0.6s ease-out';

      button.style.position = 'relative';
      button.style.overflow = 'hidden';
      button.appendChild(ripple);

      setTimeout(() => {
        ripple.remove();
      }, 600);
    }

    // دالة تصدير التقرير إلى PDF
    function exportToPDF() {
      // هنا يمكن إضافة كود لتصدير التقرير إلى PDF
      const button = document.querySelector('button[onclick="exportToPDF()"]');
      if (button) {
        const originalText = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin ml-2"></i> جاري التصدير...';
        button.disabled = true;

        setTimeout(() => {
          button.innerHTML = '<i class="fas fa-check ml-2"></i> تم التصدير';

          setTimeout(() => {
            button.innerHTML = originalText;
            button.disabled = false;
          }, 1500);
        }, 1500);
      }
    }

    // دالة تصدير التقرير إلى Excel
    function exportToExcel() {
      // هنا يمكن إضافة كود لتصدير التقرير إلى Excel
      const button = document.querySelector('button i.fa-file-excel').closest('button');
      if (button) {
        const originalText = button.innerHTML;
        button.innerHTML = '<i class="fas fa-spinner fa-spin ml-2"></i> جاري التصدير...';
        button.disabled = true;

        setTimeout(() => {
          button.innerHTML = '<i class="fas fa-check ml-2"></i> تم التصدير';

          setTimeout(() => {
            button.innerHTML = originalText;
            button.disabled = false;
          }, 1500);
        }, 1500);
      }
    }

    // تهيئة تأثير الـ ripple للأزرار
    @keyframes ripple {
      0% {
        transform: scale(0);
        opacity: 0.5;
      }
      100% {
        transform: scale(40);
        opacity: 0;
      }
    }
  </script>

  <!-- تم نقل تهيئة القائمة الجانبية إلى ملف sidebar.php -->
</body>

</html>
