<?php
/**
 * نموذج الطلب
 * يتعامل مع بيانات الطلبات في قاعدة البيانات
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Models;

use App\Core\Database;

class Order {
    private $db;
    private $table = 'orders';

    /**
     * إنشاء نموذج الطلب
     */
    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * الحصول على جميع الطلبات
     */
    public function getAll($limit = null, $offset = 0, $filters = []) {
        $sql = "SELECT o.*, b.name as branch_name 
                FROM {$this->table} o 
                LEFT JOIN branches b ON o.branch_id = b.id";
        
        $params = [];
        $whereClause = [];
        
        // إضافة الفلاتر
        if (!empty($filters)) {
            foreach ($filters as $key => $value) {
                if ($value !== null && $value !== '') {
                    switch ($key) {
                        case 'branch_id':
                            $whereClause[] = "o.branch_id = ?";
                            $params[] = $value;
                            break;
                        case 'status':
                            $whereClause[] = "o.status = ?";
                            $params[] = $value;
                            break;
                        case 'customer_name':
                            $whereClause[] = "o.customer_name LIKE ?";
                            $params[] = "%{$value}%";
                            break;
                        case 'order_number':
                            $whereClause[] = "o.order_number LIKE ?";
                            $params[] = "%{$value}%";
                            break;
                        case 'start_date':
                            $whereClause[] = "o.created_at >= ?";
                            $params[] = $value;
                            break;
                        case 'end_date':
                            $whereClause[] = "o.created_at <= ?";
                            $params[] = $value . ' 23:59:59';
                            break;
                    }
                }
            }
        }
        
        // إضافة شرط WHERE إذا كان هناك فلاتر
        if (!empty($whereClause)) {
            $sql .= " WHERE " . implode(" AND ", $whereClause);
        }
        
        // ترتيب النتائج
        $sql .= " ORDER BY o.created_at DESC";
        
        // إضافة حدود للنتائج
        if ($limit !== null) {
            $sql .= " LIMIT ?, ?";
            $params[] = $offset;
            $params[] = $limit;
        }
        
        return $this->db->fetchAll($sql, $params);
    }

    /**
     * الحصول على عدد الطلبات
     */
    public function count($filters = []) {
        $sql = "SELECT COUNT(*) FROM {$this->table} o";
        
        $params = [];
        $whereClause = [];
        
        // إضافة الفلاتر
        if (!empty($filters)) {
            foreach ($filters as $key => $value) {
                if ($value !== null && $value !== '') {
                    switch ($key) {
                        case 'branch_id':
                            $whereClause[] = "o.branch_id = ?";
                            $params[] = $value;
                            break;
                        case 'status':
                            $whereClause[] = "o.status = ?";
                            $params[] = $value;
                            break;
                        case 'customer_name':
                            $whereClause[] = "o.customer_name LIKE ?";
                            $params[] = "%{$value}%";
                            break;
                        case 'order_number':
                            $whereClause[] = "o.order_number LIKE ?";
                            $params[] = "%{$value}%";
                            break;
                        case 'start_date':
                            $whereClause[] = "o.created_at >= ?";
                            $params[] = $value;
                            break;
                        case 'end_date':
                            $whereClause[] = "o.created_at <= ?";
                            $params[] = $value . ' 23:59:59';
                            break;
                    }
                }
            }
        }
        
        // إضافة شرط WHERE إذا كان هناك فلاتر
        if (!empty($whereClause)) {
            $sql .= " WHERE " . implode(" AND ", $whereClause);
        }
        
        return $this->db->fetchColumn($sql, $params);
    }

    /**
     * البحث عن طلب بواسطة المعرف
     */
    public function findById($id) {
        $sql = "SELECT o.*, b.name as branch_name 
                FROM {$this->table} o 
                LEFT JOIN branches b ON o.branch_id = b.id 
                WHERE o.id = ?";
        
        return $this->db->fetchOne($sql, [$id]);
    }

    /**
     * البحث عن طلب بواسطة رقم الطلب
     */
    public function findByOrderNumber($orderNumber) {
        $sql = "SELECT o.*, b.name as branch_name 
                FROM {$this->table} o 
                LEFT JOIN branches b ON o.branch_id = b.id 
                WHERE o.order_number = ?";
        
        return $this->db->fetchOne($sql, [$orderNumber]);
    }

    /**
     * إنشاء طلب جديد
     */
    public function create($data) {
        // إضافة تاريخ الإنشاء
        $data['created_at'] = date('Y-m-d H:i:s');
        
        // إدراج الطلب في قاعدة البيانات
        $orderId = $this->db->insert($this->table, $data);
        
        if ($orderId) {
            return ['success' => true, 'order_id' => $orderId];
        } else {
            return ['success' => false, 'message' => 'فشل إنشاء الطلب'];
        }
    }

    /**
     * تحديث بيانات الطلب
     */
    public function update($id, $data) {
        // التحقق من وجود الطلب
        $order = $this->findById($id);
        if (!$order) {
            return ['success' => false, 'message' => 'الطلب غير موجود'];
        }
        
        // إضافة تاريخ التحديث
        $data['updated_at'] = date('Y-m-d H:i:s');
        
        // تحديث بيانات الطلب
        $success = $this->db->update($this->table, $data, 'id = ?', [$id]);
        
        if ($success) {
            return ['success' => true];
        } else {
            return ['success' => false, 'message' => 'فشل تحديث بيانات الطلب'];
        }
    }

    /**
     * حذف طلب
     */
    public function delete($id) {
        // التحقق من وجود الطلب
        $order = $this->findById($id);
        if (!$order) {
            return ['success' => false, 'message' => 'الطلب غير موجود'];
        }
        
        // حذف الطلب
        $success = $this->db->delete($this->table, 'id = ?', [$id]);
        
        if ($success) {
            return ['success' => true];
        } else {
            return ['success' => false, 'message' => 'فشل حذف الطلب'];
        }
    }

    /**
     * الحصول على إحصائيات الطلبات
     */
    public function getStats($branchId = null, $startDate = null, $endDate = null) {
        $sql = "SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN status = 'قيد المعالجة' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'قيد التوصيل' THEN 1 ELSE 0 END) as in_transit,
                SUM(CASE WHEN status = 'تم التسليم' THEN 1 ELSE 0 END) as delivered,
                SUM(CASE WHEN status = 'ملغى' THEN 1 ELSE 0 END) as cancelled
                FROM {$this->table}
                WHERE 1=1";
        
        $params = [];
        
        // إضافة فلتر الفرع
        if ($branchId !== null) {
            $sql .= " AND branch_id = ?";
            $params[] = $branchId;
        }
        
        // إضافة فلتر تاريخ البداية
        if ($startDate !== null) {
            $sql .= " AND created_at >= ?";
            $params[] = $startDate;
        }
        
        // إضافة فلتر تاريخ النهاية
        if ($endDate !== null) {
            $sql .= " AND created_at <= ?";
            $params[] = $endDate . ' 23:59:59';
        }
        
        return $this->db->fetchOne($sql, $params);
    }

    /**
     * الحصول على رقم طلب جديد
     */
    public function getNewOrderNumber($branchCode) {
        try {
            // البحث عن آخر رقم تسلسلي للفرع
            $stmt = $this->db->query("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?", [$branchCode]);
            $result = $stmt->fetch();
            
            // الحصول على السنة الحالية (آخر رقمين)
            $year = date('y');
            
            if ($result) {
                $lastNumber = $result['last_sequence_number'];
                $newNumber = $lastNumber + 1;
                
                // تحديث الرقم التسلسلي في قاعدة البيانات
                $this->db->update('order_sequences', ['last_sequence_number' => $newNumber], 'branch_code = ?', [$branchCode]);
                
                // تنسيق الرقم التسلسلي
                $sequenceNumber = str_pad($newNumber, 2, '0', STR_PAD_LEFT);
                
                // إنشاء رقم الطلب الكامل
                $newOrderNumber = $branchCode . $year . '-' . $sequenceNumber;
                
                return $newOrderNumber;
            } else {
                // إذا لم يتم العثور على رمز الفرع، قم بإنشاء سجل جديد
                $this->db->insert('order_sequences', [
                    'branch_code' => $branchCode,
                    'last_sequence_number' => 1
                ]);
                
                return $branchCode . $year . '-01';
            }
        } catch (\Exception $e) {
            // في حالة حدوث خطأ
            error_log("خطأ في الحصول على رقم طلب جديد: " . $e->getMessage());
            return false;
        }
    }
}
