<?php
/**
 * فئة التوجيه
 * تتعامل مع توجيه الطلبات إلى المتحكمات المناسبة
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Core;

class Router {
    private $routes = [];
    private static $instance;

    /**
     * إنشاء كائن التوجيه
     */
    private function __construct() {
        // تهيئة المسارات
        $this->initRoutes();
    }

    /**
     * الحصول على نسخة واحدة من الفئة (نمط Singleton)
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * تهيئة المسارات الافتراضية
     */
    private function initRoutes() {
        // المسارات الرئيسية
        $this->addRoute('GET', '/', 'AuthController@showLogin');
        $this->addRoute('GET', '/login', 'AuthController@showLogin');
        $this->addRoute('POST', '/login', 'AuthController@login');
        $this->addRoute('GET', '/logout', 'AuthController@logout');
        
        // لوحة التحكم
        $this->addRoute('GET', '/dashboard', 'DashboardController@index');
        
        // الطلبات
        $this->addRoute('GET', '/orders', 'OrderController@index');
        $this->addRoute('GET', '/orders/create', 'OrderController@create');
        $this->addRoute('POST', '/orders/store', 'OrderController@store');
        $this->addRoute('GET', '/orders/view/:id', 'OrderController@view');
        
        // التقارير
        $this->addRoute('GET', '/reports/weekly', 'ReportController@weekly');
        $this->addRoute('GET', '/reports/monthly', 'ReportController@monthly');
        
        // المستخدمين
        $this->addRoute('GET', '/users', 'UserController@index');
        $this->addRoute('GET', '/users/create', 'UserController@create');
        $this->addRoute('POST', '/users/store', 'UserController@store');
        $this->addRoute('GET', '/users/edit/:id', 'UserController@edit');
        $this->addRoute('POST', '/users/update/:id', 'UserController@update');
        $this->addRoute('POST', '/users/delete/:id', 'UserController@delete');
        
        // الفروع
        $this->addRoute('GET', '/branches', 'BranchController@index');
        $this->addRoute('GET', '/branches/create', 'BranchController@create');
        $this->addRoute('POST', '/branches/store', 'BranchController@store');
        $this->addRoute('GET', '/branches/edit/:id', 'BranchController@edit');
        $this->addRoute('POST', '/branches/update/:id', 'BranchController@update');
        
        // الإعدادات
        $this->addRoute('GET', '/settings', 'SettingsController@index');
        $this->addRoute('POST', '/settings/update', 'SettingsController@update');
    }

    /**
     * إضافة مسار جديد
     */
    public function addRoute($method, $path, $handler) {
        $this->routes[] = [
            'method' => $method,
            'path' => $path,
            'handler' => $handler
        ];
    }

    /**
     * تطابق المسار مع المسارات المسجلة
     */
    private function matchRoute($method, $path) {
        foreach ($this->routes as $route) {
            if ($route['method'] !== $method) {
                continue;
            }
            
            // تحويل مسار المسار إلى تعبير منتظم
            $pattern = $this->convertRouteToRegex($route['path']);
            
            // التحقق من تطابق المسار
            if (preg_match($pattern, $path, $matches)) {
                // إزالة المطابقة الكاملة
                array_shift($matches);
                
                return [
                    'handler' => $route['handler'],
                    'params' => $matches
                ];
            }
        }
        
        return null;
    }

    /**
     * تحويل مسار المسار إلى تعبير منتظم
     */
    private function convertRouteToRegex($route) {
        // استبدال المعلمات بنمط التعبير المنتظم
        $pattern = preg_replace('/:[a-zA-Z0-9]+/', '([^/]+)', $route);
        
        // إضافة بداية ونهاية التعبير المنتظم
        $pattern = '/^' . str_replace('/', '\/', $pattern) . '$/';
        
        return $pattern;
    }

    /**
     * توجيه الطلب الحالي
     */
    public function dispatch() {
        // الحصول على طريقة الطلب والمسار
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        
        // إزالة المسار الأساسي من المسار
        $basePath = dirname($_SERVER['SCRIPT_NAME']);
        if ($basePath !== '/' && $basePath !== '\\') {
            $path = substr($path, strlen($basePath));
        }
        
        // التأكد من أن المسار يبدأ بـ /
        if (empty($path) || $path[0] !== '/') {
            $path = '/' . $path;
        }
        
        // مطابقة المسار
        $match = $this->matchRoute($method, $path);
        
        if ($match) {
            // تقسيم المتحكم والإجراء
            list($controller, $action) = explode('@', $match['handler']);
            
            // إضافة مساحة الاسم
            $controllerClass = "\\App\\Controllers\\{$controller}";
            
            // التحقق من وجود الفئة والطريقة
            if (class_exists($controllerClass)) {
                $controllerInstance = new $controllerClass();
                
                if (method_exists($controllerInstance, $action)) {
                    // استدعاء الإجراء مع المعلمات
                    return call_user_func_array([$controllerInstance, $action], $match['params']);
                }
            }
        }
        
        // إذا لم يتم العثور على مسار مطابق، عرض صفحة 404
        $this->notFound();
    }

    /**
     * عرض صفحة 404
     */
    private function notFound() {
        header("HTTP/1.0 404 Not Found");
        include dirname(__DIR__) . '/views/errors/404.php';
        exit;
    }

    /**
     * إعادة توجيه إلى مسار آخر
     */
    public static function redirect($path, $message = null, $type = 'success') {
        if ($message !== null) {
            $_SESSION['flash_message'] = $message;
            $_SESSION['flash_type'] = $type;
        }
        
        header("Location: {$path}");
        exit;
    }
}
