<?php
/**
 * متحكم المستخدمين
 * يتعامل مع إدارة المستخدمين (إنشاء، عرض، تعديل، حذف)
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Controllers;

use App\Core\Auth;
use App\Models\User;
use App\Models\Branch;

class UserController {
    private $auth;
    private $userModel;
    private $branchModel;

    /**
     * إنشاء متحكم المستخدمين
     */
    public function __construct() {
        $this->auth = Auth::getInstance();
        $this->userModel = new User();
        $this->branchModel = new Branch();

        // التحقق من تسجيل الدخول
        if (!$this->auth->isLoggedIn()) {
            header('Location: /login');
            exit;
        }

        // التحقق من الصلاحيات
        $user = $this->auth->getUser();
        if ($user['role'] !== 'admin') {
            $_SESSION['flash_message'] = 'ليس لديك صلاحية للوصول إلى هذه الصفحة.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /dashboard');
            exit;
        }
    }

    /**
     * عرض قائمة المستخدمين
     */
    public function index() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // الحصول على جميع المستخدمين
        $users = $this->userModel->getAll();

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'إدارة المستخدمين';
        $activePage = 'users';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;
        $branches = $this->branchModel->getAll();

        // عرض صفحة إدارة المستخدمين
        include dirname(__DIR__) . '/views/users/manage.php';
    }

    /**
     * عرض صفحة إنشاء مستخدم جديد
     */
    public function create() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // الحصول على قائمة الفروع
        $branches = $this->branchModel->getActive();

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'إنشاء مستخدم جديد';
        $activePage = 'users';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;

        // عرض صفحة إنشاء مستخدم جديد
        include dirname(__DIR__) . '/views/users/create.php';
    }

    /**
     * حفظ مستخدم جديد
     */
    public function store() {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /users/create');
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/create');
            exit;
        }

        // الحصول على بيانات النموذج
        $username = $_POST['username'] ?? '';
        $fullName = $_POST['full_name'] ?? '';
        $email = $_POST['email'] ?? '';
        $password = $_POST['password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        $role = $_POST['role'] ?? 'user';
        $branchId = $_POST['branch_id'] ?? null;
        $isActive = isset($_POST['is_active']) ? 1 : 0;

        // التحقق من البيانات
        if (empty($username) || empty($fullName) || empty($email) || empty($password)) {
            $_SESSION['flash_message'] = 'يرجى ملء جميع الحقول المطلوبة.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/create');
            exit;
        }

        // التحقق من تطابق كلمة المرور
        if ($password !== $confirmPassword) {
            $_SESSION['flash_message'] = 'كلمة المرور وتأكيد كلمة المرور غير متطابقين.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/create');
            exit;
        }

        // التحقق من صحة البريد الإلكتروني
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $_SESSION['flash_message'] = 'البريد الإلكتروني غير صالح.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/create');
            exit;
        }

        // التحقق من طول كلمة المرور
        if (strlen($password) < 8) {
            $_SESSION['flash_message'] = 'يجب أن تكون كلمة المرور 8 أحرف على الأقل.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/create');
            exit;
        }

        // إعداد بيانات المستخدم
        $userData = [
            'username' => $username,
            'full_name' => $fullName,
            'email' => $email,
            'password' => $password,
            'role' => $role,
            'branch_id' => $branchId,
            'is_active' => $isActive
        ];

        // إنشاء المستخدم
        $result = $this->userModel->create($userData);

        if ($result['success']) {
            $_SESSION['flash_message'] = 'تم إنشاء المستخدم بنجاح.';
            $_SESSION['flash_type'] = 'success';
            header('Location: /users');
            exit;
        } else {
            $_SESSION['flash_message'] = 'فشل إنشاء المستخدم: ' . $result['message'];
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/create');
            exit;
        }
    }

    /**
     * عرض صفحة تعديل المستخدم
     */
    public function edit($id) {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // الحصول على بيانات المستخدم المراد تعديله
        $editUser = $this->userModel->findById($id);

        // التحقق من وجود المستخدم
        if (!$editUser) {
            $_SESSION['flash_message'] = 'المستخدم غير موجود.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users');
            exit;
        }

        // الحصول على قائمة الفروع
        $branches = $this->branchModel->getActive();

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'تعديل المستخدم: ' . $editUser['full_name'];
        $activePage = 'users';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;

        // عرض صفحة تعديل المستخدم
        include dirname(__DIR__) . '/views/users/edit.php';
    }

    /**
     * تحديث بيانات المستخدم
     */
    public function update($id) {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /users/edit/' . $id);
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/edit/' . $id);
            exit;
        }

        // الحصول على بيانات المستخدم المراد تعديله
        $editUser = $this->userModel->findById($id);

        // التحقق من وجود المستخدم
        if (!$editUser) {
            $_SESSION['flash_message'] = 'المستخدم غير موجود.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users');
            exit;
        }

        // الحصول على بيانات النموذج
        $username = $_POST['username'] ?? '';
        $fullName = $_POST['full_name'] ?? '';
        $email = $_POST['email'] ?? '';
        $password = $_POST['password'] ?? '';
        $confirmPassword = $_POST['confirm_password'] ?? '';
        $role = $_POST['role'] ?? 'user';
        $branchId = $_POST['branch_id'] ?? null;
        $isActive = isset($_POST['is_active']) ? 1 : 0;

        // التحقق من البيانات
        if (empty($username) || empty($fullName) || empty($email)) {
            $_SESSION['flash_message'] = 'يرجى ملء جميع الحقول المطلوبة.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/edit/' . $id);
            exit;
        }

        // التحقق من صحة البريد الإلكتروني
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $_SESSION['flash_message'] = 'البريد الإلكتروني غير صالح.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/edit/' . $id);
            exit;
        }

        // إعداد بيانات المستخدم
        $userData = [
            'username' => $username,
            'full_name' => $fullName,
            'email' => $email,
            'role' => $role,
            'branch_id' => $branchId,
            'is_active' => $isActive
        ];

        // التحقق من كلمة المرور إذا تم تغييرها
        if (!empty($password)) {
            // التحقق من تطابق كلمة المرور
            if ($password !== $confirmPassword) {
                $_SESSION['flash_message'] = 'كلمة المرور وتأكيد كلمة المرور غير متطابقين.';
                $_SESSION['flash_type'] = 'error';
                header('Location: /users/edit/' . $id);
                exit;
            }

            // التحقق من طول كلمة المرور
            if (strlen($password) < 8) {
                $_SESSION['flash_message'] = 'يجب أن تكون كلمة المرور 8 أحرف على الأقل.';
                $_SESSION['flash_type'] = 'error';
                header('Location: /users/edit/' . $id);
                exit;
            }

            // إضافة كلمة المرور إلى بيانات المستخدم
            $userData['password'] = $password;
        }

        // تحديث بيانات المستخدم
        $result = $this->userModel->update($id, $userData);

        if ($result['success']) {
            $_SESSION['flash_message'] = 'تم تحديث بيانات المستخدم بنجاح.';
            $_SESSION['flash_type'] = 'success';
            header('Location: /users');
            exit;
        } else {
            $_SESSION['flash_message'] = 'فشل تحديث بيانات المستخدم: ' . $result['message'];
            $_SESSION['flash_type'] = 'error';
            header('Location: /users/edit/' . $id);
            exit;
        }
    }

    /**
     * حذف المستخدم
     */
    public function delete($id) {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /users');
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users');
            exit;
        }

        // الحصول على بيانات المستخدم الحالي
        $currentUser = $this->auth->getUser();

        // التحقق من عدم حذف المستخدم الحالي
        if ($id == $currentUser['id']) {
            $_SESSION['flash_message'] = 'لا يمكنك حذف حسابك الحالي.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /users');
            exit;
        }

        // حذف المستخدم
        $result = $this->userModel->delete($id);

        if ($result['success']) {
            $_SESSION['flash_message'] = 'تم حذف المستخدم بنجاح.';
            $_SESSION['flash_type'] = 'success';
        } else {
            $_SESSION['flash_message'] = 'فشل حذف المستخدم: ' . $result['message'];
            $_SESSION['flash_type'] = 'error';
        }

        header('Location: /users');
        exit;
    }
}
