<?php
/**
 * متحكم الإعدادات
 * يتعامل مع إدارة إعدادات النظام
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Controllers;

use App\Core\Auth;

class SettingsController {
    private $auth;
    private $configFile;

    /**
     * إنشاء متحكم الإعدادات
     */
    public function __construct() {
        $this->auth = Auth::getInstance();
        $this->configFile = dirname(__DIR__, 2) . '/config/config.php';

        // التحقق من تسجيل الدخول
        if (!$this->auth->isLoggedIn()) {
            header('Location: /login');
            exit;
        }

        // التحقق من الصلاحيات
        $user = $this->auth->getUser();
        if ($user['role'] !== 'admin') {
            $_SESSION['flash_message'] = 'ليس لديك صلاحية للوصول إلى هذه الصفحة.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /dashboard');
            exit;
        }
    }

    /**
     * عرض صفحة الإعدادات
     */
    public function index() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // الحصول على الإعدادات الحالية
        $config = require $this->configFile;

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'إعدادات النظام';
        $activePage = 'settings';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;

        // عرض صفحة الإعدادات
        include dirname(__DIR__) . '/views/settings/index.php';
    }

    /**
     * تحديث الإعدادات
     */
    public function update() {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /settings');
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /settings');
            exit;
        }

        // الحصول على الإعدادات الحالية
        $config = require $this->configFile;

        // الحصول على بيانات النموذج
        $systemName = $_POST['system_name'] ?? $config['system_name'];
        $companyName = $_POST['company_name'] ?? $config['company_name'];
        $companyAddress = $_POST['company_address'] ?? $config['company_address'];
        $companyPhone = $_POST['company_phone'] ?? $config['company_phone'];
        $companyEmail = $_POST['company_email'] ?? $config['company_email'];
        $useHttps = isset($_POST['use_https']) ? true : false;
        $cacheEnabled = isset($_POST['cache_enabled']) ? true : false;
        $cacheLifetime = $_POST['cache_lifetime'] ?? $config['cache_lifetime'];
        $uploadMaxSize = $_POST['upload_max_size'] ?? $config['upload_max_size'];
        $csrfProtection = isset($_POST['csrf_protection']) ? true : false;
        $passwordMinLength = $_POST['password_min_length'] ?? $config['password_min_length'];
        $mailFrom = $_POST['mail_from'] ?? $config['mail_from'];
        $mailFromName = $_POST['mail_from_name'] ?? $config['mail_from_name'];
        $reportsPerPage = $_POST['reports_per_page'] ?? $config['reports_per_page'];

        // تحديث الإعدادات
        $config['system_name'] = $systemName;
        $config['company_name'] = $companyName;
        $config['company_address'] = $companyAddress;
        $config['company_phone'] = $companyPhone;
        $config['company_email'] = $companyEmail;
        $config['use_https'] = $useHttps;
        $config['cache_enabled'] = $cacheEnabled;
        $config['cache_lifetime'] = (int)$cacheLifetime;
        $config['upload_max_size'] = (int)$uploadMaxSize;
        $config['csrf_protection'] = $csrfProtection;
        $config['password_min_length'] = (int)$passwordMinLength;
        $config['mail_from'] = $mailFrom;
        $config['mail_from_name'] = $mailFromName;
        $config['reports_per_page'] = (int)$reportsPerPage;

        // حفظ الإعدادات
        $success = $this->saveConfig($config);

        if ($success) {
            $_SESSION['flash_message'] = 'تم تحديث الإعدادات بنجاح.';
            $_SESSION['flash_type'] = 'success';
        } else {
            $_SESSION['flash_message'] = 'فشل تحديث الإعدادات. يرجى التحقق من صلاحيات الملف.';
            $_SESSION['flash_type'] = 'error';
        }

        header('Location: /settings');
        exit;
    }

    /**
     * حفظ الإعدادات في ملف التكوين
     */
    private function saveConfig($config) {
        // التحقق من إمكانية الكتابة في ملف التكوين
        if (!is_writable($this->configFile)) {
            return false;
        }

        // إنشاء محتوى ملف التكوين
        $content = "<?php\n/**\n * ملف التكوين المركزي للنظام\n * يحتوي على جميع إعدادات النظام\n */\n\n";

        // إضافة تعريف وضع التطوير
        $content .= "// تحديد وضع التطوير (تفعيل لعرض الأخطاء)\ndefine('DEVELOPMENT_MODE', " . (DEVELOPMENT_MODE ? 'true' : 'false') . ");\n\n";

        // إضافة إعدادات عرض الأخطاء
        $content .= "// تفعيل عرض الأخطاء للتصحيح\nif (DEVELOPMENT_MODE) {\n";
        $content .= "    // وضع التطوير: عرض جميع الأخطاء\n";
        $content .= "    ini_set('display_errors', 1);\n";
        $content .= "    ini_set('display_startup_errors', 1);\n";
        $content .= "    error_reporting(E_ALL);\n";
        $content .= "} else {\n";
        $content .= "    // وضع الإنتاج: إخفاء الأخطاء\n";
        $content .= "    ini_set('display_errors', 0);\n";
        $content .= "    ini_set('display_startup_errors', 0);\n";
        $content .= "    error_reporting(E_ALL & ~E_NOTICE & ~E_DEPRECATED & ~E_STRICT);\n\n";
        $content .= "    // تسجيل الأخطاء في ملف\n";
        $content .= "    ini_set('log_errors', 1);\n";
        $content .= "    ini_set('error_log', dirname(__DIR__) . '/storage/logs/error.log');\n";
        $content .= "}\n\n";

        // إضافة إعدادات المنطقة الزمنية والترميز
        $content .= "// تعيين منطقة زمنية افتراضية\ndate_default_timezone_set('Asia/Riyadh');\n\n";
        $content .= "// تعيين ترميز افتراضي\nini_set('default_charset', 'UTF-8');\n\n";
        $content .= "// تعيين حد الذاكرة\nini_set('memory_limit', '256M');\n\n";
        $content .= "// تعيين وقت التنفيذ الأقصى\nset_time_limit(300);\n\n";

        // إضافة الإعدادات كمصفوفة
        $content .= "// معلومات النظام\nreturn [\n";

        // إضافة معلومات النظام
        $content .= "    // معلومات النظام\n";
        $content .= "    'system_name' => '" . addslashes($config['system_name']) . "',\n";
        $content .= "    'company_name' => '" . addslashes($config['company_name']) . "',\n";
        $content .= "    'company_logo' => '" . addslashes($config['company_logo']) . "',\n";
        $content .= "    'company_address' => '" . addslashes($config['company_address']) . "',\n";
        $content .= "    'company_phone' => '" . addslashes($config['company_phone']) . "',\n";
        $content .= "    'company_email' => '" . addslashes($config['company_email']) . "',\n";
        $content .= "    \n";

        // إضافة إعدادات URL
        $content .= "    // إعدادات URL\n";
        $content .= "    'base_url' => '" . addslashes($config['base_url']) . "',\n";
        $content .= "    'use_https' => " . ($config['use_https'] ? 'true' : 'false') . ",\n";
        $content .= "    \n";

        // إضافة إعدادات المسار
        $content .= "    // إعدادات المسار\n";
        $content .= "    'base_path' => dirname(__DIR__),\n";
        $content .= "    \n";

        // إضافة إعدادات الجلسة
        $content .= "    // إعدادات الجلسة\n";
        $content .= "    'session_name' => '" . addslashes($config['session_name']) . "',\n";
        $content .= "    'session_lifetime' => " . $config['session_lifetime'] . ", // ساعتان\n";
        $content .= "    \n";

        // إضافة إعدادات التخزين المؤقت
        $content .= "    // إعدادات التخزين المؤقت\n";
        $content .= "    'cache_enabled' => " . ($config['cache_enabled'] ? 'true' : 'false') . ",\n";
        $content .= "    'cache_lifetime' => " . $config['cache_lifetime'] . ", // 5 دقائق\n";
        $content .= "    \n";

        // إضافة إعدادات التحميل
        $content .= "    // إعدادات التحميل\n";
        $content .= "    'upload_max_size' => " . $config['upload_max_size'] . ", // 10 ميجابايت\n";
        $content .= "    'allowed_extensions' => " . var_export($config['allowed_extensions'], true) . ",\n";
        $content .= "    \n";

        // إضافة إعدادات الأمان
        $content .= "    // إعدادات الأمان\n";
        $content .= "    'csrf_protection' => " . ($config['csrf_protection'] ? 'true' : 'false') . ",\n";
        $content .= "    'password_min_length' => " . $config['password_min_length'] . ",\n";
        $content .= "    \n";

        // إضافة إعدادات البريد الإلكتروني
        $content .= "    // إعدادات البريد الإلكتروني\n";
        $content .= "    'mail_from' => '" . addslashes($config['mail_from']) . "',\n";
        $content .= "    'mail_from_name' => '" . addslashes($config['mail_from_name']) . "',\n";
        $content .= "    \n";

        // إضافة إعدادات التقارير
        $content .= "    // إعدادات التقارير\n";
        $content .= "    'reports_per_page' => " . $config['reports_per_page'] . ",\n";
        $content .= "    \n";

        // إضافة إعدادات الفروع
        $content .= "    // إعدادات الفروع\n";
        $content .= "    'branches' => " . var_export($config['branches'], true) . ",\n";
        $content .= "    \n";

        // إضافة إعدادات حالات الطلب
        $content .= "    // إعدادات حالات الطلب\n";
        $content .= "    'order_statuses' => " . var_export($config['order_statuses'], true) . "\n";

        $content .= "];";

        // حفظ الملف
        return file_put_contents($this->configFile, $content) !== false;
    }

    /**
     * تنظيف ملفات التخزين المؤقت
     */
    public function clearCache() {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /settings');
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /settings');
            exit;
        }

        // حذف ملفات التخزين المؤقت
        $cacheDir = dirname(__DIR__, 2) . '/cache';
        $success = $this->deleteDirectory($cacheDir);

        if ($success) {
            // إعادة إنشاء مجلد التخزين المؤقت
            mkdir($cacheDir, 0777, true);

            $_SESSION['flash_message'] = 'تم تنظيف ملفات التخزين المؤقت بنجاح.';
            $_SESSION['flash_type'] = 'success';
        } else {
            $_SESSION['flash_message'] = 'فشل تنظيف ملفات التخزين المؤقت. يرجى التحقق من صلاحيات المجلد.';
            $_SESSION['flash_type'] = 'error';
        }

        header('Location: /settings');
        exit;
    }

    /**
     * حذف مجلد وجميع محتوياته
     */
    private function deleteDirectory($dir) {
        if (!file_exists($dir)) {
            return true;
        }

        if (!is_dir($dir)) {
            return unlink($dir);
        }

        foreach (scandir($dir) as $item) {
            if ($item == '.' || $item == '..') {
                continue;
            }

            if (!$this->deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {
                return false;
            }
        }

        return rmdir($dir);
    }
}
