<?php
/**
 * متحكم التقارير
 * يتعامل مع عرض التقارير الأسبوعية والشهرية
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Controllers;

use App\Core\Auth;
use App\Models\Order;
use App\Models\Branch;

class ReportController {
    private $auth;
    private $orderModel;
    private $branchModel;

    /**
     * إنشاء متحكم التقارير
     */
    public function __construct() {
        $this->auth = Auth::getInstance();
        $this->orderModel = new Order();
        $this->branchModel = new Branch();

        // التحقق من تسجيل الدخول
        if (!$this->auth->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
    }

    /**
     * عرض التقارير الأسبوعية
     */
    public function weekly() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // تحديد ما إذا كان المستخدم مديرًا
        $isAdmin = ($user['role'] === 'admin' || $user['role'] === 'manager');

        // تحديد الأسبوع الحالي
        $currentWeek = isset($_GET['week']) ? (int)$_GET['week'] : date('W');
        $currentYear = isset($_GET['year']) ? (int)$_GET['year'] : date('Y');

        // التحقق من صحة الأسبوع والسنة
        if ($currentWeek < 1 || $currentWeek > 53) {
            $currentWeek = date('W');
        }

        if ($currentYear < 2020 || $currentYear > date('Y') + 1) {
            $currentYear = date('Y');
        }

        // حساب تاريخ بداية ونهاية الأسبوع
        $startDate = new \DateTime();
        $startDate->setISODate($currentYear, $currentWeek, 1); // 1 = الاثنين

        $endDate = clone $startDate;
        $endDate->modify('+6 days'); // +6 يوم = الأحد

        // تنسيق التواريخ
        $startDateFormatted = $startDate->format('Y-m-d');
        $endDateFormatted = $endDate->format('Y-m-d');

        // تحديد معلمات الفلترة
        $filters = [
            'start_date' => $startDateFormatted,
            'end_date' => $endDateFormatted
        ];

        // إذا كان المستخدم ليس مديرًا، فلتر الطلبات حسب الفرع
        if (!$isAdmin) {
            $filters['branch_id'] = $user['branch_id'];
        }

        // الحصول على معلمات البحث
        $branchId = $_GET['branch_id'] ?? '';

        // إضافة معلمات البحث إلى الفلاتر
        if (!empty($branchId) && $isAdmin) {
            $filters['branch_id'] = $branchId;
        }

        // الحصول على الطلبات الأسبوعية
        $weeklyOrders = $this->orderModel->getAll(null, 0, $filters);

        // الحصول على إحصائيات الطلبات الأسبوعية
        $weeklyStats = $this->orderModel->getStats(
            $filters['branch_id'] ?? null,
            $startDateFormatted,
            $endDateFormatted
        );

        // الحصول على قائمة الفروع للفلترة
        $branches = $isAdmin ? $this->branchModel->getActive() : [];

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'التقارير الأسبوعية: الأسبوع ' . $currentWeek . ' - ' . $currentYear;
        $activePage = 'weekly-reports';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;
        $config = require dirname(__DIR__, 2) . '/config/config.php';

        // عرض صفحة التقارير الأسبوعية
        include dirname(__DIR__) . '/views/reports/weekly.php';
    }

    /**
     * عرض التقارير الشهرية
     */
    public function monthly() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // تحديد ما إذا كان المستخدم مديرًا
        $isAdmin = ($user['role'] === 'admin' || $user['role'] === 'manager');

        // تحديد الشهر الحالي
        $currentMonth = isset($_GET['month']) ? (int)$_GET['month'] : date('n');
        $currentYear = isset($_GET['year']) ? (int)$_GET['year'] : date('Y');

        // التحقق من صحة الشهر والسنة
        if ($currentMonth < 1 || $currentMonth > 12) {
            $currentMonth = date('n');
        }

        if ($currentYear < 2020 || $currentYear > date('Y') + 1) {
            $currentYear = date('Y');
        }

        // حساب تاريخ بداية ونهاية الشهر
        $startDate = new \DateTime("$currentYear-$currentMonth-01");
        $endDate = clone $startDate;
        $endDate->modify('last day of this month');

        // تنسيق التواريخ
        $startDateFormatted = $startDate->format('Y-m-d');
        $endDateFormatted = $endDate->format('Y-m-d');

        // تحديد معلمات الفلترة
        $filters = [
            'start_date' => $startDateFormatted,
            'end_date' => $endDateFormatted
        ];

        // إذا كان المستخدم ليس مديرًا، فلتر الطلبات حسب الفرع
        if (!$isAdmin) {
            $filters['branch_id'] = $user['branch_id'];
        }

        // الحصول على معلمات البحث
        $branchId = $_GET['branch_id'] ?? '';

        // إضافة معلمات البحث إلى الفلاتر
        if (!empty($branchId) && $isAdmin) {
            $filters['branch_id'] = $branchId;
        }

        // الحصول على الطلبات الشهرية
        $monthlyOrders = $this->orderModel->getAll(null, 0, $filters);

        // الحصول على إحصائيات الطلبات الشهرية
        $monthlyStats = $this->orderModel->getStats(
            $filters['branch_id'] ?? null,
            $startDateFormatted,
            $endDateFormatted
        );

        // الحصول على قائمة الفروع للفلترة
        $branches = $isAdmin ? $this->branchModel->getActive() : [];

        // الحصول على أسماء الأشهر بالعربية
        $arabicMonths = [
            1 => 'يناير',
            2 => 'فبراير',
            3 => 'مارس',
            4 => 'أبريل',
            5 => 'مايو',
            6 => 'يونيو',
            7 => 'يوليو',
            8 => 'أغسطس',
            9 => 'سبتمبر',
            10 => 'أكتوبر',
            11 => 'نوفمبر',
            12 => 'ديسمبر'
        ];

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'التقارير الشهرية: ' . $arabicMonths[$currentMonth] . ' ' . $currentYear;
        $activePage = 'monthly-reports';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;
        $config = require dirname(__DIR__, 2) . '/config/config.php';

        // عرض صفحة التقارير الشهرية
        include dirname(__DIR__) . '/views/reports/monthly.php';
    }

    /**
     * تصدير التقرير إلى ملف Excel
     */
    public function exportExcel() {
        // التحقق من المعلمات
        $type = $_GET['type'] ?? '';
        $startDate = $_GET['start_date'] ?? '';
        $endDate = $_GET['end_date'] ?? '';
        $branchId = $_GET['branch_id'] ?? '';

        if (empty($type) || empty($startDate) || empty($endDate)) {
            $_SESSION['flash_message'] = 'معلمات غير صالحة للتصدير.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /reports/' . ($type === 'weekly' ? 'weekly' : 'monthly'));
            exit;
        }

        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // تحديد ما إذا كان المستخدم مديرًا
        $isAdmin = ($user['role'] === 'admin' || $user['role'] === 'manager');

        // تحديد معلمات الفلترة
        $filters = [
            'start_date' => $startDate,
            'end_date' => $endDate
        ];

        // إذا كان المستخدم ليس مديرًا، فلتر الطلبات حسب الفرع
        if (!$isAdmin) {
            $filters['branch_id'] = $user['branch_id'];
        } elseif (!empty($branchId)) {
            $filters['branch_id'] = $branchId;
        }

        // الحصول على الطلبات
        $orders = $this->orderModel->getAll(null, 0, $filters);

        // الحصول على إحصائيات الطلبات
        $stats = $this->orderModel->getStats(
            $filters['branch_id'] ?? null,
            $startDate,
            $endDate
        );

        // الحصول على معلومات الفرع إذا كان محددًا
        $branch = null;
        if (!empty($filters['branch_id'])) {
            $branch = $this->branchModel->findById($filters['branch_id']);
        }

        // تحديد عنوان التقرير
        $reportTitle = $type === 'weekly' ? 'التقرير الأسبوعي' : 'التقرير الشهري';
        $reportTitle .= ' (' . $startDate . ' - ' . $endDate . ')';

        if ($branch) {
            $reportTitle .= ' - ' . $branch['name'];
        }

        // تصدير البيانات إلى ملف Excel
        // هنا يمكن استخدام مكتبة مثل PhpSpreadsheet لإنشاء ملف Excel

        // في هذا المثال، سنقوم بإنشاء ملف CSV بسيط
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $reportTitle . '.csv"');

        $output = fopen('php://output', 'w');

        // إضافة BOM لدعم الترميز UTF-8
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));

        // إضافة عنوان التقرير
        fputcsv($output, [$reportTitle]);
        fputcsv($output, []);

        // إضافة الإحصائيات
        fputcsv($output, ['إجمالي الطلبات', 'قيد المعالجة', 'قيد التوصيل', 'تم التسليم', 'ملغى']);
        fputcsv($output, [
            $stats['total'] ?? 0,
            $stats['pending'] ?? 0,
            $stats['in_transit'] ?? 0,
            $stats['delivered'] ?? 0,
            $stats['cancelled'] ?? 0
        ]);
        fputcsv($output, []);

        // إضافة رؤوس الأعمدة
        fputcsv($output, ['رقم الطلب', 'اسم العميل', 'الفرع', 'الموظف', 'التاريخ', 'المبلغ', 'الحالة']);

        // إضافة بيانات الطلبات
        foreach ($orders as $order) {
            fputcsv($output, [
                $order['order_number'],
                $order['customer_name'],
                $order['branch_name'],
                $order['employee_name'],
                $order['created_at'],
                $order['amount_paid'],
                $order['status']
            ]);
        }

        fclose($output);
        exit;
    }
}
