<?php
/**
 * متحكم لوحة التحكم
 * يتعامل مع عرض لوحة التحكم والإحصائيات
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Controllers;

use App\Core\Auth;
use App\Models\Order;
use App\Models\User;
use App\Models\Branch;

class DashboardController {
    private $auth;
    private $orderModel;
    private $userModel;
    private $branchModel;

    /**
     * إنشاء متحكم لوحة التحكم
     */
    public function __construct() {
        $this->auth = Auth::getInstance();
        $this->orderModel = new Order();
        $this->userModel = new User();
        $this->branchModel = new Branch();
        
        // التحقق من تسجيل الدخول
        if (!$this->auth->isLoggedIn()) {
            header('Location: /login');
            exit;
        }
    }

    /**
     * عرض لوحة التحكم
     */
    public function index() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();
        
        // تحديد ما إذا كان المستخدم مديرًا
        $isAdmin = ($user['role'] === 'admin' || $user['role'] === 'manager');
        
        // تحديد معلمات الفلترة
        $branchId = $isAdmin ? null : $user['branch_id'];
        $startDate = date('Y-m-d', strtotime('-90 days'));
        $endDate = date('Y-m-d');
        
        // الحصول على إحصائيات الطلبات
        $orderStats = $this->orderModel->getStats($branchId, $startDate, $endDate);
        
        // تعيين قيم الإحصائيات
        $totalOrders = $orderStats['total'] ?? 0;
        $pendingOrders = $orderStats['pending'] ?? 0;
        $inTransitOrders = $orderStats['in_transit'] ?? 0;
        $deliveredOrders = $orderStats['delivered'] ?? 0;
        $cancelledOrders = $orderStats['cancelled'] ?? 0;
        
        // الحصول على عدد المستخدمين والفروع
        $totalUsers = $this->userModel->count();
        $totalBranches = $this->branchModel->count();
        
        // الحصول على أفضل الموظفين أداءً
        $topEmployees = $this->getTopEmployees($branchId);
        
        // الحصول على أفضل الفروع أداءً
        $topBranches = $this->getTopBranches($isAdmin, $branchId);
        
        // الحصول على الطلبات الأخيرة
        $recentOrders = $this->getRecentOrders($branchId);
        
        // عرض صفحة لوحة التحكم
        include dirname(__DIR__) . '/views/dashboard/index.php';
    }

    /**
     * الحصول على أفضل الموظفين أداءً
     */
    private function getTopEmployees($branchId = null) {
        // تحديد معلمات الفلترة
        $startDate = date('Y-m-d', strtotime('-30 days'));
        $endDate = date('Y-m-d');
        
        // استعلام للحصول على أفضل الموظفين أداءً
        $sql = "SELECT
                employee_name as name,
                COUNT(*) as orders,
                COALESCE(SUM(amount_paid), 0) as revenue
            FROM
                orders
            WHERE
                employee_name IS NOT NULL
                AND employee_name != ''
                AND created_at >= ?
                AND created_at <= ?";
        
        $params = [$startDate, $endDate . ' 23:59:59'];
        
        // إضافة فلتر الفرع إذا كان مطلوبًا
        if ($branchId !== null) {
            $sql .= " AND branch_id = ?";
            $params[] = $branchId;
        }
        
        $sql .= " GROUP BY
                employee_name
            ORDER BY
                orders DESC, revenue DESC
            LIMIT 4";
        
        // تنفيذ الاستعلام
        $db = \App\Core\Database::getInstance();
        $result = $db->fetchAll($sql, $params);
        
        // إذا لم يتم العثور على بيانات، استخدم بيانات افتراضية
        if (empty($result)) {
            $result = [
                [
                    'name' => 'أحمد محمد',
                    'orders' => 0,
                    'revenue' => 0
                ],
                [
                    'name' => 'محمد علي',
                    'orders' => 0,
                    'revenue' => 0
                ],
                [
                    'name' => 'فاطمة أحمد',
                    'orders' => 0,
                    'revenue' => 0
                ],
                [
                    'name' => 'خالد عبدالله',
                    'orders' => 0,
                    'revenue' => 0
                ]
            ];
        }
        
        return $result;
    }

    /**
     * الحصول على أفضل الفروع أداءً
     */
    private function getTopBranches($isAdmin, $branchId = null) {
        // إذا كان المستخدم ليس مديرًا، نعرض فقط الفرع الحالي
        if (!$isAdmin) {
            $branch = $this->branchModel->findById($branchId);
            
            if ($branch) {
                // الحصول على إحصائيات الفرع
                $stats = $this->branchModel->getStats($branchId);
                
                return [
                    [
                        'name' => $branch['name'],
                        'orders' => $stats['total_orders'] ?? 0,
                        'revenue' => $stats['total_revenue'] ?? 0
                    ]
                ];
            }
            
            return [
                [
                    'name' => 'الفرع الحالي',
                    'orders' => 0,
                    'revenue' => 0
                ]
            ];
        }
        
        // للمدير، نعرض أفضل الفروع أداءً
        $startDate = date('Y-m-d', strtotime('-30 days'));
        $endDate = date('Y-m-d');
        
        return $this->branchModel->getTopBranches(4, $startDate, $endDate);
    }

    /**
     * الحصول على الطلبات الأخيرة
     */
    private function getRecentOrders($branchId = null) {
        // تحديد معلمات الفلترة
        $filters = [];
        
        if ($branchId !== null) {
            $filters['branch_id'] = $branchId;
        }
        
        // الحصول على آخر 5 طلبات
        return $this->orderModel->getAll(5, 0, $filters);
    }
}
