<?php
/**
 * متحكم الفروع
 * يتعامل مع إدارة الفروع (إنشاء، عرض، تعديل، حذف)
 *
 * @version 1.0
 * @author فريق التطوير
 */

namespace App\Controllers;

use App\Core\Auth;
use App\Models\Branch;

class BranchController {
    private $auth;
    private $branchModel;

    /**
     * إنشاء متحكم الفروع
     */
    public function __construct() {
        $this->auth = Auth::getInstance();
        $this->branchModel = new Branch();

        // التحقق من تسجيل الدخول
        if (!$this->auth->isLoggedIn()) {
            header('Location: /login');
            exit;
        }

        // التحقق من الصلاحيات
        $user = $this->auth->getUser();
        if ($user['role'] !== 'admin') {
            $_SESSION['flash_message'] = 'ليس لديك صلاحية للوصول إلى هذه الصفحة.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /dashboard');
            exit;
        }
    }

    /**
     * عرض قائمة الفروع
     */
    public function index() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // الحصول على جميع الفروع
        $branches = $this->branchModel->getAll();

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'إدارة الفروع';
        $activePage = 'branches';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;

        // عرض صفحة إدارة الفروع
        include dirname(__DIR__) . '/views/branches/manage.php';
    }

    /**
     * عرض صفحة إنشاء فرع جديد
     */
    public function create() {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'إنشاء فرع جديد';
        $activePage = 'branches';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;

        // عرض صفحة إنشاء فرع جديد
        include dirname(__DIR__) . '/views/branches/create.php';
    }

    /**
     * حفظ فرع جديد
     */
    public function store() {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /branches/create');
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches/create');
            exit;
        }

        // الحصول على بيانات النموذج
        $name = $_POST['name'] ?? '';
        $code = $_POST['code'] ?? '';
        $address = $_POST['address'] ?? '';
        $phone = $_POST['phone'] ?? '';
        $email = $_POST['email'] ?? '';
        $isActive = isset($_POST['is_active']) ? 1 : 0;

        // التحقق من البيانات
        if (empty($name) || empty($code)) {
            $_SESSION['flash_message'] = 'يرجى ملء جميع الحقول المطلوبة.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches/create');
            exit;
        }

        // التحقق من صحة رمز الفرع
        if (!preg_match('/^[A-Za-z0-9]{2,5}$/', $code)) {
            $_SESSION['flash_message'] = 'رمز الفرع يجب أن يتكون من 2-5 أحرف أو أرقام.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches/create');
            exit;
        }

        // تحويل رمز الفرع إلى أحرف كبيرة
        $code = strtoupper($code);

        // إعداد بيانات الفرع
        $branchData = [
            'name' => $name,
            'code' => $code,
            'address' => $address,
            'phone' => $phone,
            'email' => $email,
            'is_active' => $isActive
        ];

        // إنشاء الفرع
        $result = $this->branchModel->create($branchData);

        if ($result['success']) {
            $_SESSION['flash_message'] = 'تم إنشاء الفرع بنجاح.';
            $_SESSION['flash_type'] = 'success';
            header('Location: /branches');
            exit;
        } else {
            $_SESSION['flash_message'] = 'فشل إنشاء الفرع: ' . $result['message'];
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches/create');
            exit;
        }
    }

    /**
     * عرض صفحة تعديل الفرع
     */
    public function edit($id) {
        // الحصول على بيانات المستخدم الحالي
        $user = $this->auth->getUser();

        // الحصول على بيانات الفرع المراد تعديله
        $branch = $this->branchModel->findById($id);

        // التحقق من وجود الفرع
        if (!$branch) {
            $_SESSION['flash_message'] = 'الفرع غير موجود.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches');
            exit;
        }

        // تعيين عنوان الصفحة والصفحة النشطة
        $pageTitle = 'تعديل الفرع: ' . $branch['name'];
        $activePage = 'branches';

        // تمرير متغير $auth إلى الصفحة
        $auth = $this->auth;

        // عرض صفحة تعديل الفرع
        include dirname(__DIR__) . '/views/branches/edit.php';
    }

    /**
     * تحديث بيانات الفرع
     */
    public function update($id) {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /branches/edit/' . $id);
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches/edit/' . $id);
            exit;
        }

        // الحصول على بيانات الفرع المراد تعديله
        $branch = $this->branchModel->findById($id);

        // التحقق من وجود الفرع
        if (!$branch) {
            $_SESSION['flash_message'] = 'الفرع غير موجود.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches');
            exit;
        }

        // الحصول على بيانات النموذج
        $name = $_POST['name'] ?? '';
        $code = $_POST['code'] ?? '';
        $address = $_POST['address'] ?? '';
        $phone = $_POST['phone'] ?? '';
        $email = $_POST['email'] ?? '';
        $isActive = isset($_POST['is_active']) ? 1 : 0;

        // التحقق من البيانات
        if (empty($name) || empty($code)) {
            $_SESSION['flash_message'] = 'يرجى ملء جميع الحقول المطلوبة.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches/edit/' . $id);
            exit;
        }

        // التحقق من صحة رمز الفرع
        if (!preg_match('/^[A-Za-z0-9]{2,5}$/', $code)) {
            $_SESSION['flash_message'] = 'رمز الفرع يجب أن يتكون من 2-5 أحرف أو أرقام.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches/edit/' . $id);
            exit;
        }

        // تحويل رمز الفرع إلى أحرف كبيرة
        $code = strtoupper($code);

        // إعداد بيانات الفرع
        $branchData = [
            'name' => $name,
            'code' => $code,
            'address' => $address,
            'phone' => $phone,
            'email' => $email,
            'is_active' => $isActive
        ];

        // تحديث بيانات الفرع
        $result = $this->branchModel->update($id, $branchData);

        if ($result['success']) {
            $_SESSION['flash_message'] = 'تم تحديث بيانات الفرع بنجاح.';
            $_SESSION['flash_type'] = 'success';
            header('Location: /branches');
            exit;
        } else {
            $_SESSION['flash_message'] = 'فشل تحديث بيانات الفرع: ' . $result['message'];
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches/edit/' . $id);
            exit;
        }
    }

    /**
     * حذف الفرع
     */
    public function delete($id) {
        // التحقق من طريقة الطلب
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            header('Location: /branches');
            exit;
        }

        // التحقق من رمز CSRF
        if (!isset($_POST['csrf_token']) || !$this->auth->verifyCSRFToken($_POST['csrf_token'])) {
            $_SESSION['flash_message'] = 'خطأ في التحقق من الأمان. يرجى المحاولة مرة أخرى.';
            $_SESSION['flash_type'] = 'error';
            header('Location: /branches');
            exit;
        }

        // حذف الفرع
        $result = $this->branchModel->delete($id);

        if ($result['success']) {
            $_SESSION['flash_message'] = 'تم حذف الفرع بنجاح.';
            $_SESSION['flash_type'] = 'success';
        } else {
            $_SESSION['flash_message'] = 'فشل حذف الفرع: ' . $result['message'];
            $_SESSION['flash_type'] = 'error';
        }

        header('Location: /branches');
        exit;
    }
}
