<?php
// تضمين ملف التحقق من تسجيل الدخول
require_once 'check_auth.php';

// التحقق من وجود معرف الطلب
if (!isset($_GET['id']) || empty($_GET['id'])) {
    // إعادة توجيه المستخدم إلى صفحة قائمة الطلبات
    header('Location: orders.php');
    exit;
}

// الحصول على معرف الطلب
$orderId = (int)$_GET['id'];

// الحصول على معلومات الطلب
$stmt = $pdo->prepare("
    SELECT
        o.*,
        b.name as branch_name,
        u.full_name as created_by_name
    FROM orders o
    LEFT JOIN branches b ON o.branch_id = b.id
    LEFT JOIN users u ON o.user_id = u.id
    WHERE o.id = ?
");
$stmt->execute([$orderId]);
$order = $stmt->fetch();

// التحقق من وجود الطلب
if (!$order) {
    // إعادة توجيه المستخدم إلى صفحة قائمة الطلبات
    header('Location: orders.php');
    exit;
}

// التحقق من صلاحيات المستخدم
$canEditOrder = ($currentUser['role'] === 'admin' || $currentUser['role'] === 'manager' || $order['user_id'] == $currentUser['id']);
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>نظام الشحن | تفاصيل الطلب</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
  <link rel="stylesheet" href="../assets/css/unified-design.css">
  <style>
    :root {
      --primary: #4f46e5;
      --primary-dark: #4338ca;
      --primary-light: #818cf8;
      --secondary: #0ea5e9;
      --secondary-dark: #0284c7;
      --secondary-light: #7dd3fc;
      --success: #10b981;
      --success-dark: #059669;
      --success-light: #6ee7b7;
      --danger: #ef4444;
      --danger-dark: #dc2626;
      --danger-light: #fca5a5;
      --warning: #f59e0b;
      --warning-dark: #d97706;
      --warning-light: #fcd34d;
      --info: #3b82f6;
      --info-dark: #2563eb;
      --info-light: #93c5fd;
      --gray-50: #f9fafb;
      --gray-100: #f3f4f6;
      --gray-200: #e5e7eb;
      --gray-300: #d1d5db;
      --gray-400: #9ca3af;
      --gray-500: #6b7280;
      --gray-600: #4b5563;
      --gray-700: #374151;
      --gray-800: #1f2937;
      --gray-900: #111827;

      /* تنسيقات ورقة A4 */
      --a4-width: 210mm;
      --a4-height: 297mm;
      --a4-padding: 15mm;
    }

    /* تنسيق الخط */
    body {
      font-family: 'Tajawal', sans-serif;
      background-color: #f1f5f9;
      color: var(--gray-800);
      line-height: 1.6;
      margin: 0;
      padding: 0;
    }

    /* تنسيق الصفحة الرئيسية */
    .page-container {
      display: flex;
      flex-direction: column;
      align-items: center;
      padding: 2rem;
    }

    /* تنسيق الشريط العلوي */
    .top-navbar {
      width: var(--a4-width);
      background: linear-gradient(135deg, var(--primary), var(--primary-dark));
      color: white;
      padding: 1rem 1.5rem;
      border-radius: 0.5rem;
      margin-bottom: 1.5rem;
      box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
      display: flex;
      justify-content: space-between;
      align-items: center;
      box-sizing: border-box;
    }

    .page-title {
      font-size: 1.25rem;
      font-weight: 700;
      display: flex;
      align-items: center;
      gap: 0.75rem;
    }

    .user-menu {
      display: flex;
      align-items: center;
      gap: 1rem;
    }

    /* تنسيق ورقة A4 */
    .a4-page {
      width: var(--a4-width);
      min-height: var(--a4-height);
      background-color: white;
      box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
      margin-bottom: 2rem;
      position: relative;
      overflow: hidden;
      box-sizing: border-box;
    }

    /* تنسيق ترويسة الصفحة */
    .page-header {
      padding: var(--a4-padding) var(--a4-padding) 1rem;
      border-bottom: 1px solid var(--gray-200);
      display: flex;
      justify-content: space-between;
      align-items: flex-start;
    }

    .company-logo {
      width: 120px;
      height: auto;
    }

    .company-info {
      text-align: left;
    }

    .company-name {
      font-size: 1.5rem;
      font-weight: 700;
      background: linear-gradient(135deg, #4338ca, #3730a3, #4f46e5);
      -webkit-background-clip: text;
      background-clip: text;
      color: transparent;
      margin: 0;
      position: relative;
      display: inline-block;
      text-shadow: 0px 1px 1px rgba(255, 255, 255, 0.5);
      letter-spacing: 0.5px;
      padding-bottom: 5px;
    }

    .company-name::after {
      content: '';
      position: absolute;
      bottom: 0;
      left: 0;
      width: 100%;
      height: 2px;
      background: linear-gradient(90deg, transparent, #4338ca, transparent);
    }

    .company-details {
      font-size: 0.75rem;
      color: var(--gray-600);
      margin: 0.25rem 0;
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }

    .company-details i {
      color: var(--primary);
      font-size: 0.85rem;
      width: 16px;
      text-align: center;
    }

    /* تنسيق عنوان المستند */
    .document-title {
      text-align: center;
      margin: 1.5rem 0;
      padding: 0 var(--a4-padding);
    }

    .document-title h1 {
      font-size: 1.5rem;
      font-weight: 700;
      color: var(--primary-dark);
      margin: 0;
      padding: 0.5rem 2rem;
      border-bottom: 2px solid var(--primary-light);
      display: inline-block;
      position: relative;
      background-color: rgba(79, 70, 229, 0.05);
      border-radius: 0.5rem 0.5rem 0 0;
    }

    .document-title h1::before,
    .document-title h1::after {
      content: '';
      position: absolute;
      bottom: -2px;
      width: 10px;
      height: 10px;
      border-radius: 50%;
      background-color: var(--primary);
    }

    .document-title h1::before {
      left: 0;
    }

    .document-title h1::after {
      right: 0;
    }

    /* تنسيق معلومات الطلب */
    .order-meta-box {
      margin: 0 var(--a4-padding);
      display: flex;
      justify-content: space-between;
      background-color: var(--gray-50);
      border: 1px solid var(--gray-200);
      border-radius: 0.5rem;
      padding: 0.5rem;
      margin-bottom: 1rem;
    }

    .order-meta-column {
      flex: 1;
    }

    .meta-group {
      margin-bottom: 0.5rem;
    }

    .meta-label {
      font-size: 0.7rem;
      color: var(--gray-500);
      margin-bottom: 0.15rem;
    }

    .meta-value {
      font-weight: 600;
      color: var(--gray-800);
      font-size: 0.8rem;
    }

    .order-status {
      padding: 0.25rem 0.75rem;
      border-radius: 1rem;
      font-size: 0.75rem;
      font-weight: 600;
      display: inline-flex;
      align-items: center;
      gap: 0.25rem;
    }

    .status-processing {
      background-color: var(--info-light);
      color: var(--info-dark);
    }

    .status-in-transit {
      background-color: var(--warning-light);
      color: var(--warning-dark);
    }

    .status-delivered {
      background-color: var(--success-light);
      color: var(--success-dark);
    }

    .status-cancelled {
      background-color: var(--danger-light);
      color: var(--danger-dark);
    }

    /* تنسيق محتوى الطلب */
    .page-content {
      padding: 0 var(--a4-padding);
    }

    .content-section {
      margin-bottom: 1rem;
    }

    .section-title {
      font-size: 0.9rem;
      font-weight: 700;
      color: var(--primary-dark);
      margin-bottom: 0.5rem;
      padding-bottom: 0.3rem;
      border-bottom: 1px solid var(--gray-200);
      display: flex;
      align-items: center;
      gap: 0.5rem;
    }

    .section-title i {
      color: var(--primary);
    }

    /* تنسيق الجدول */
    .info-table {
      width: 100%;
      border-collapse: collapse;
      margin-bottom: 0.5rem;
    }

    .info-table th {
      background-color: var(--gray-100);
      padding: 0.4rem;
      text-align: right;
      font-weight: 600;
      color: var(--gray-700);
      border: 1px solid var(--gray-200);
      font-size: 0.7rem;
    }

    .info-table td {
      padding: 0.4rem;
      border: 1px solid var(--gray-200);
      font-size: 0.8rem;
    }

    .info-table tr:nth-child(even) {
      background-color: var(--gray-50);
    }

    /* تنسيق الملاحظات */
    .notes-box {
      background-color: #fffbeb;
      border: 1px solid #fef3c7;
      border-radius: 0.5rem;
      padding: 0.5rem;
      margin-bottom: 0.5rem;
    }

    .notes-title {
      font-weight: 600;
      color: var(--warning-dark);
      margin-bottom: 0.3rem;
      display: flex;
      align-items: center;
      gap: 0.3rem;
      font-size: 0.8rem;
    }

    .notes-box p {
      font-size: 0.8rem;
      margin: 0;
    }

    /* تنسيق تذييل الصفحة */
    .page-footer {
      padding: 1rem var(--a4-padding);
      border-top: 1px solid var(--gray-200);
      margin-top: 2rem;
      font-size: 0.75rem;
      color: var(--gray-500);
      display: flex;
      justify-content: space-between;
      width: 100%;
    }

    .footer-signature {
      display: flex;
      justify-content: space-between;
      margin-top: 1rem;
      margin-bottom: 1rem;
      padding: 0 var(--a4-padding);
    }

    .signature-box {
      text-align: center;
      flex: 1;
      max-width: 150px;
    }

    .signature-line {
      border-top: 1px solid var(--gray-300);
      margin-top: 1rem;
      margin-bottom: 0.3rem;
    }

    .signature-name {
      font-weight: 600;
      font-size: 0.7rem;
    }

    .signature-title {
      font-size: 0.6rem;
      color: var(--gray-500);
    }

    /* تنسيق الشعار المائي */
    .watermark {
      position: absolute;
      top: 50%;
      left: 50%;
      transform: translate(-50%, -50%) rotate(-30deg);
      width: 60%;
      height: auto;
      opacity: 0.05;
      pointer-events: none;
      z-index: 0;
    }

    /* تنسيق أزرار الإجراءات */
    .actions-bar {
      width: var(--a4-width);
      display: flex;
      gap: 0.75rem;
      margin-bottom: 1.5rem;
      justify-content: flex-end;
    }

    .btn {
      display: inline-flex;
      align-items: center;
      gap: 0.5rem;
      padding: 0.5rem 1rem;
      border-radius: 0.25rem;
      font-weight: 600;
      font-size: 0.875rem;
      transition: all 0.2s ease;
      cursor: pointer;
    }

    .btn-primary {
      background-color: var(--primary);
      color: white;
      border: none;
    }

    .btn-primary:hover {
      background-color: var(--primary-dark);
    }

    .btn-secondary {
      background-color: white;
      color: var(--gray-700);
      border: 1px solid var(--gray-300);
    }

    .btn-secondary:hover {
      background-color: var(--gray-50);
    }

    .btn-danger {
      background-color: var(--danger);
      color: white;
      border: none;
    }

    .btn-danger:hover {
      background-color: var(--danger-dark);
    }

    /* تنسيق الطباعة */
    @media print {
      @page {
        size: A4;
        margin: 0;
      }

      body {
        background-color: white;
      }

      .page-container {
        padding: 0;
      }

      .top-navbar, .actions-bar {
        display: none !important;
      }

      .a4-page {
        width: 100%;
        min-height: 100vh;
        box-shadow: none;
        margin: 0;
        padding: 0;
      }

      .page-header, .page-content {
        padding-left: 15mm;
        padding-right: 15mm;
      }

      .page-footer {
        padding: 1rem 15mm;
        margin-top: 2rem;
        position: relative;
        bottom: auto;
      }

      .footer-signature {
        margin-bottom: 3rem;
      }

      .watermark {
        display: block !important;
      }
    }
  </style>
</head>

<body>
  <div class="page-container">
    <!-- الشريط العلوي -->
    <header class="top-navbar">
      <div class="page-title">
        <i class="fas fa-file-invoice"></i>
        <span>تفاصيل الطلب #<?php echo htmlspecialchars($order['order_number']); ?></span>
      </div>
      <div class="user-menu">
        <a href="orders.php" class="btn btn-secondary btn-back">
          <i class="fas fa-arrow-right ml-2"></i>
          <span>العودة للقائمة</span>
        </a>
        <div class="user-avatar">
          <?php echo isset($currentUser['full_name']) ? mb_substr($currentUser['full_name'], 0, 1, 'UTF-8') : 'A'; ?>
        </div>
      </div>
    </header>

    <!-- أزرار الإجراءات -->
    <div class="actions-bar">
      <?php if ($canEditOrder): ?>
      <button id="printOrderBtn" class="btn btn-secondary">
        <i class="fas fa-file-pdf ml-2"></i>
        <span>تحميل PDF</span>
      </button>
      <button id="deleteOrderBtn" class="btn btn-danger">
        <i class="fas fa-trash-alt ml-2"></i>
        <span>حذف</span>
      </button>
      <?php endif; ?>
    </div>

    <!-- ورقة A4 -->
    <div class="a4-page">
      <!-- الشعار المائي -->
      <img src="../assets/images/Company logo.png" alt="شعار الشركة" class="watermark">

      <!-- ترويسة الصفحة -->
      <div class="page-header">
        <div>
          <img src="../assets/images/Company logo.png" alt="شعار الشركة" class="company-logo">
        </div>
        <div class="company-info">
          <h1 class="company-name">
            <span style="font-size: 1.1em; color: #4338ca;">مجموعة</span>
            <span style="font-size: 1.2em; font-weight: 800;">ابراهيم الاحمدي</span>
            <span style="font-size: 0.9em;">اليمنية</span>
          </h1>
          <p class="company-details"><i class="fas fa-map-marker-alt"></i> العنوان: الجمهورية اليمنية، حضرموت، المكلا</p>
          <p class="company-details"><i class="fas fa-phone-alt"></i> هاتف: 05318281</p>
          <p class="company-details"><i class="fas fa-envelope"></i> البريد الإلكتروني: ibrahim@al-ahmadigroup.com</p>
        </div>
      </div>

      <!-- عنوان المستند -->
      <div class="document-title">
        <h1>تفاصيل طلب الشحن</h1>
      </div>

      <!-- معلومات الطلب الأساسية -->
      <div class="order-meta-box">
        <div class="order-meta-column">
          <div class="meta-group">
            <div class="meta-label">رقم الطلب</div>
            <div class="meta-value"><?php echo htmlspecialchars($order['order_number']); ?></div>
          </div>
          <div class="meta-group">
            <div class="meta-label">تاريخ الإنشاء</div>
            <div class="meta-value"><?php echo date('Y-m-d', strtotime($order['created_at'])); ?></div>
          </div>
          <div class="meta-group">
            <div class="meta-label">تم الإنشاء بواسطة</div>
            <div class="meta-value"><?php echo htmlspecialchars($order['created_by_name'] ?? 'غير معروف'); ?></div>
          </div>
        </div>

        <div class="order-meta-column">
          <div class="meta-group">
            <div class="meta-label">الفرع</div>
            <div class="meta-value"><?php echo htmlspecialchars($order['branch_name'] ?? '-'); ?></div>
          </div>
          <div class="meta-group">
            <div class="meta-label">تاريخ التسليم المتوقع</div>
            <div class="meta-value"><?php echo $order['scheduled_delivery_date'] ? date('Y-m-d', strtotime($order['scheduled_delivery_date'])) : '-'; ?></div>
          </div>
          <div class="meta-group">
            <div class="meta-label">مدة التوصيل</div>
            <div class="meta-value"><?php echo $order['delivery_duration'] ? $order['delivery_duration'] . ' يوم' : '-'; ?></div>
          </div>
        </div>

        <div class="order-meta-column">
          <div class="meta-group">
            <div class="meta-label">حالة الطلب</div>
            <?php
            $statusClass = '';
            $statusIcon = '';
            switch ($order['status']) {
                case 'قيد المعالجة':
                    $statusClass = 'status-processing';
                    $statusIcon = 'clock';
                    break;
                case 'في الطريق':
                    $statusClass = 'status-in-transit';
                    $statusIcon = 'truck';
                    break;
                case 'تم التسليم':
                    $statusClass = 'status-delivered';
                    $statusIcon = 'check-circle';
                    break;
                case 'ملغى':
                    $statusClass = 'status-cancelled';
                    $statusIcon = 'times-circle';
                    break;
                default:
                    $statusClass = 'status-processing';
                    $statusIcon = 'question-circle';
            }
            ?>
            <div class="order-status <?php echo $statusClass; ?>">
              <i class="fas fa-<?php echo $statusIcon; ?>"></i>
              <?php echo htmlspecialchars($order['status']); ?>
            </div>
          </div>
          <div class="meta-group">
            <div class="meta-label">حالة الدفع</div>
            <?php
            $paymentStatusClass = '';
            $paymentStatusIcon = '';
            switch ($order['payment_status']) {
                case 'مدفوع':
                    $paymentStatusClass = 'status-delivered';
                    $paymentStatusIcon = 'check-circle';
                    break;
                case 'جزئي':
                    $paymentStatusClass = 'status-in-transit';
                    $paymentStatusIcon = 'dot-circle';
                    break;
                case 'غير مدفوع':
                    $paymentStatusClass = 'status-cancelled';
                    $paymentStatusIcon = 'times-circle';
                    break;
                default:
                    $paymentStatusClass = '';
                    $paymentStatusIcon = 'question-circle';
            }
            ?>
            <div class="order-status <?php echo $paymentStatusClass; ?>">
              <i class="fas fa-<?php echo $paymentStatusIcon; ?>"></i>
              <?php echo htmlspecialchars($order['payment_status'] ?? '-'); ?>
            </div>
          </div>
        </div>
      </div>

      <!-- محتوى الطلب -->
      <div class="page-content">
        <!-- معلومات العميل والمستلم -->
        <div class="content-section">
          <h3 class="section-title">
            <i class="fas fa-user"></i>
            معلومات العميل والمستلم
          </h3>

          <table class="info-table">
            <tr>
              <th width="25%">اسم العميل</th>
              <td width="25%"><?php echo htmlspecialchars($order['client_name']); ?></td>
              <th width="25%">اسم المستلم</th>
              <td width="25%"><?php echo htmlspecialchars($order['recipient_name'] ?? '-'); ?></td>
            </tr>
            <tr>
              <th>رقم الهاتف</th>
              <td dir="ltr"><?php echo htmlspecialchars($order['phone_number'] ?? '-'); ?></td>
              <th>شركة التوصيل</th>
              <td><?php echo htmlspecialchars($order['delivery_company'] ?? '-'); ?></td>
            </tr>
            <tr>
              <th>منطقة المغادرة</th>
              <td><?php echo htmlspecialchars($order['departure_area'] ?? '-'); ?></td>
              <th>منطقة التسليم</th>
              <td><?php echo htmlspecialchars($order['delivery_area'] ?? '-'); ?></td>
            </tr>
            <?php if (!empty($order['second_destination'])): ?>
            <tr>
              <th>الوجهة الثانية</th>
              <td colspan="3"><?php echo htmlspecialchars($order['second_destination']); ?></td>
            </tr>
            <?php endif; ?>
          </table>
        </div>

        <!-- معلومات البضاعة -->
        <div class="content-section">
          <h3 class="section-title">
            <i class="fas fa-box"></i>
            معلومات البضاعة
          </h3>

          <table class="info-table">
            <tr>
              <th width="25%">نوع الخدمة</th>
              <td width="25%"><?php echo htmlspecialchars($order['service_type'] ?? '-'); ?></td>
              <th width="25%">اسم البضاعة</th>
              <td width="25%"><?php echo htmlspecialchars($order['goods_name'] ?? '-'); ?></td>
            </tr>
            <tr>
              <th>نوع البضاعة</th>
              <td><?php echo htmlspecialchars($order['goods_type'] ?? '-'); ?></td>
              <th>بلد المنشأ</th>
              <td><?php echo htmlspecialchars($order['country_of_origin'] ?? '-'); ?></td>
            </tr>
            <tr>
              <th>الوزن</th>
              <td><?php echo $order['weight'] ? $order['weight'] . ' كجم' : '-'; ?></td>
              <th>الكمية</th>
              <td><?php echo $order['quantity'] ? $order['quantity'] . ' ' . ($order['unit'] ?? '') : '-'; ?></td>
            </tr>
          </table>
        </div>

        <!-- المعلومات المالية -->
        <div class="content-section">
          <h3 class="section-title">
            <i class="fas fa-money-bill-wave"></i>
            المعلومات المالية
          </h3>

          <table class="info-table">
            <tr>
              <th width="25%">العملة</th>
              <td width="25%"><?php echo htmlspecialchars($order['currency'] ?? '-'); ?></td>
              <th width="25%">نوع الدفع</th>
              <td width="25%"><?php echo htmlspecialchars($order['payment_type'] ?? '-'); ?></td>
            </tr>
            <tr>
              <th>المبلغ المتفق مع العميل</th>
              <td><?php echo $order['amount_agreed_with_customer'] ? $order['amount_agreed_with_customer'] . ' ' . $order['currency'] : '-'; ?></td>
              <th>المبلغ المتفق مع الوكيل</th>
              <td><?php echo $order['amount_agreed_with_agent'] ? $order['amount_agreed_with_agent'] . ' ' . $order['currency'] : '-'; ?></td>
            </tr>
            <tr>
              <th>المبلغ المدفوع</th>
              <td><?php echo $order['amount_paid'] ? $order['amount_paid'] . ' ' . $order['currency'] : '-'; ?></td>
              <th>المبلغ المتبقي</th>
              <td><?php echo $order['remaining_amount'] ? $order['remaining_amount'] . ' ' . $order['currency'] : '-'; ?></td>
            </tr>
            <tr>
              <th>رسوم الخدمة</th>
              <td><?php echo $order['service_fees'] ? $order['service_fees'] . ' ' . $order['currency'] : '-'; ?></td>
              <th>مصاريف أخرى</th>
              <td><?php echo $order['other_expenses'] ? $order['other_expenses'] . ' ' . $order['currency'] : '-'; ?></td>
            </tr>
            <tr>
              <th>الربح</th>
              <td colspan="3" style="font-weight: 700; <?php echo $order['profit'] >= 0 ? 'color: var(--success-dark);' : 'color: var(--danger-dark);'; ?>">
                <?php echo $order['profit'] ? $order['profit'] . ' ' . $order['currency'] : '-'; ?>
              </td>
            </tr>
          </table>
        </div>

        <!-- ملاحظات -->
        <?php if (!empty($order['notes'])): ?>
        <div class="content-section">
          <h3 class="section-title">
            <i class="fas fa-sticky-note"></i>
            ملاحظات
          </h3>

          <div class="notes-box">
            <div class="notes-title">
              <i class="fas fa-info-circle"></i>
              ملاحظات إضافية
            </div>
            <p><?php echo nl2br(htmlspecialchars($order['notes'])); ?></p>
          </div>
        </div>
        <?php endif; ?>

        <!-- توقيعات -->
        <div class="footer-signature">
          <div class="signature-box">
            <div class="signature-line"></div>
            <div class="signature-name">توقيع المستلم</div>
            <div class="signature-title">المستلم</div>
          </div>

          <div class="signature-box">
            <div class="signature-line"></div>
            <div class="signature-name">توقيع الموظف المسؤول</div>
            <div class="signature-title"><?php echo htmlspecialchars($order['employee_name'] ?? 'الموظف المسؤول'); ?></div>
          </div>

          <div class="signature-box">
            <div class="signature-line"></div>
            <div class="signature-name">ختم الشركة</div>
            <div class="signature-title">مجموعة ابراهيم الاحمدي اليمنية</div>
          </div>
        </div>
      </div>

      <!-- تذييل الصفحة -->
      <div class="page-footer">
        <div>
          <p>تم إنشاء هذا المستند بواسطة نظام الشحن | تاريخ الطباعة: <?php echo date('Y-m-d H:i'); ?></p>
        </div>
        <div>
          <p>جميع الحقوق محفوظة &copy; <?php echo date('Y'); ?> <span style="font-weight: 600; color: var(--primary-dark);">مجموعة ابراهيم الاحمدي اليمنية</span></p>
        </div>
      </div>
    </div><!-- نهاية ورقة A4 -->
  </div>

  <!-- سكريبت الصفحة -->
  <!-- تضمين المكتبات اللازمة -->
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
  <!-- استخدام إصدار أحدث من jsPDF -->
  <script src="https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js"></script>
  <!-- إضافة مكتبة html2canvas -->
  <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
  <!-- إضافة مكتبة للتعامل مع اللغة العربية في PDF -->
  <script src="https://unpkg.com/jspdf-autotable"></script>

  <script>
    document.addEventListener('DOMContentLoaded', function() {
      console.log('DOM Content Loaded');

      // التحقق من تحميل المكتبات
      if (typeof jsPDF === 'undefined') {
        console.error('jsPDF library not loaded');
        Swal.fire({
          title: 'خطأ في تحميل المكتبات',
          text: 'لم يتم تحميل مكتبة jsPDF بشكل صحيح. يرجى تحديث الصفحة والمحاولة مرة أخرى.',
          icon: 'error',
          confirmButtonColor: '#4f46e5',
          confirmButtonText: 'تحديث الصفحة',
        }).then(() => {
          window.location.reload();
        });
        return;
      }

      if (typeof html2canvas === 'undefined') {
        console.error('html2canvas library not loaded');
        Swal.fire({
          title: 'خطأ في تحميل المكتبات',
          text: 'لم يتم تحميل مكتبة html2canvas بشكل صحيح. يرجى تحديث الصفحة والمحاولة مرة أخرى.',
          icon: 'error',
          confirmButtonColor: '#4f46e5',
          confirmButtonText: 'تحديث الصفحة',
        }).then(() => {
          window.location.reload();
        });
        return;
      }

      console.log('Libraries loaded successfully');

      // دالة تحويل الصفحة إلى PDF
      function generatePDF() {
        try {
          console.log('Starting PDF generation...');

          // عرض رسالة انتظار
          Swal.fire({
            title: 'جاري إنشاء ملف PDF...',
            text: 'يرجى الانتظار',
            allowOutsideClick: false,
            didOpen: () => {
              Swal.showLoading();
            }
          });

          // الحصول على عنصر الصفحة
          const element = document.querySelector('.a4-page');

          if (!element) {
            console.error('A4 page element not found');
            Swal.fire({
              title: 'خطأ!',
              text: 'لم يتم العثور على عنصر الصفحة',
              icon: 'error',
              confirmButtonColor: '#4f46e5',
              confirmButtonText: 'حسناً'
            });
            return;
          }

          console.log('A4 page element found');

          // تحضير الصفحة للتحويل
          document.querySelectorAll('.info-table, .content-section, .page-content, .a4-page').forEach(el => {
            el.style.display = 'block';
            el.style.visibility = 'visible';
            el.style.overflow = 'visible';
          });

          document.querySelectorAll('.info-table tr').forEach(el => {
            el.style.display = 'table-row';
            el.style.visibility = 'visible';
          });

          document.querySelectorAll('.info-table th, .info-table td').forEach(el => {
            el.style.display = 'table-cell';
            el.style.visibility = 'visible';
            el.style.color = '#000';
          });

          // الحصول على اسم الملف من رقم الطلب
          const orderNumber = '<?php echo htmlspecialchars($order['order_number']); ?>';
          const fileName = `طلب_شحن_${orderNumber}.pdf`;

          // تحضير اسم الشركة للطباعة
          const companyNameElement = document.querySelector('.company-name');
          const originalHTML = companyNameElement ? companyNameElement.innerHTML : '';

          // تحضير الصفحة بالتنسيق المطلوب كما في الصورة
          if (companyNameElement) {
            companyNameElement.innerHTML = 'مجموعة ابراهيم الاحمدي اليمنية';
            companyNameElement.style.fontSize = '18pt';
            companyNameElement.style.fontWeight = 'bold';
            companyNameElement.style.color = '#4338ca';
            companyNameElement.style.textShadow = 'none';
            companyNameElement.style.background = 'none';
            companyNameElement.style.textAlign = 'center';
            companyNameElement.style.marginBottom = '10px';
            companyNameElement.style.paddingBottom = '5px';
            companyNameElement.style.borderBottom = '2px solid #4338ca';
            companyNameElement.style.display = 'block';
          }

          // تحسين عرض الشعار
          const logoElement = document.querySelector('.company-logo');
          if (logoElement) {
            logoElement.style.maxWidth = '120px';
            logoElement.style.height = 'auto';
            logoElement.style.display = 'block';
          }

          // تحسين عرض الجداول للطباعة لتتناسب مع التصميم في الصورة
          document.querySelectorAll('.info-table').forEach(table => {
            table.style.width = '100%';
            table.style.borderCollapse = 'collapse';
            table.style.border = '1px solid #4338ca';
            table.style.marginBottom = '20px';
            table.style.pageBreakInside = 'avoid';
            table.style.boxShadow = 'none';
          });

          document.querySelectorAll('.info-table th').forEach(cell => {
            cell.style.border = '1px solid #4338ca';
            cell.style.padding = '3mm';
            cell.style.color = '#ffffff';
            cell.style.backgroundColor = '#4338ca';
            cell.style.fontWeight = 'bold';
            cell.style.fontSize = '10pt';
            cell.style.textAlign = 'center';
          });

          document.querySelectorAll('.info-table td').forEach(cell => {
            cell.style.border = '1px solid #4338ca';
            cell.style.padding = '3mm';
            cell.style.color = '#1f2937';
            cell.style.fontSize = '10pt';
            cell.style.textAlign = 'center';
          });

          // تحسين عرض العلامة المائية
          const watermarkElement = document.querySelector('.watermark');
          if (watermarkElement) {
            watermarkElement.style.opacity = '0.08';
            watermarkElement.style.position = 'absolute';
            watermarkElement.style.top = '50%';
            watermarkElement.style.left = '50%';
            watermarkElement.style.transform = 'translate(-50%, -50%) rotate(-30deg)';
            watermarkElement.style.zIndex = '-1';
            watermarkElement.style.width = '60%';
            watermarkElement.style.maxWidth = '500px';
            watermarkElement.style.height = 'auto';
            watermarkElement.style.pointerEvents = 'none';
          }

          // تم تحسين عرض ترويسة الصفحة لاحقاً في الكود

          // تم تحسين عرض عنوان المستند لاحقاً في الكود

          // تحسين عرض عناوين الأقسام لتتناسب مع التصميم في الصورة
          document.querySelectorAll('.section-title').forEach(title => {
            title.style.position = 'relative';
            title.style.fontSize = '14pt';
            title.style.fontWeight = 'bold';
            title.style.color = '#4338ca';
            title.style.marginTop = '20px';
            title.style.marginBottom = '15px';
            title.style.paddingBottom = '5px';
            title.style.textAlign = 'right';
            title.style.borderBottom = 'none';
            title.style.backgroundColor = '#eef2ff'; // لون خلفية فاتح
            title.style.padding = '5px 10px';
            title.style.borderRight = '4px solid #4338ca';
            title.style.borderRadius = '0 5px 5px 0';
          });

          // تم تحسين عرض معلومات الطلب الأساسية لاحقاً في الكود

          // تم تحسين عرض تذييل الصفحة لاحقاً في الكود

          // تم تحسين عرض قسم التوقيعات لاحقاً في الكود

          // تحسين عرض معلومات العميل والمستلم
          document.querySelector('.content-section:nth-child(1)').style.marginBottom = '25px';

          // تحسين عرض معلومات البضاعة
          document.querySelector('.content-section:nth-child(2)').style.marginBottom = '25px';

          // تحسين عرض المعلومات المالية
          document.querySelector('.content-section:nth-child(3)').style.marginBottom = '25px';

          // تحسين عرض صناديق التوقيع
          const signatureBoxes = document.querySelectorAll('.signature-box');
          signatureBoxes.forEach(box => {
            box.style.textAlign = 'center';
            box.style.padding = '10px';
            box.style.minWidth = '120px';

            // تحسين عرض خط التوقيع
            const signatureLine = box.querySelector('.signature-line');
            if (signatureLine) {
              signatureLine.style.marginTop = '25px';
              signatureLine.style.marginBottom = '5px';
              signatureLine.style.borderTop = '1px solid #4338ca';
              signatureLine.style.width = '100%';
            }

            // تحسين عرض اسم الموقع
            const signatureName = box.querySelector('.signature-name');
            if (signatureName) {
              signatureName.style.fontWeight = 'bold';
              signatureName.style.fontSize = '10pt';
              signatureName.style.color = '#1f2937';
            }

            // تحسين عرض منصب الموقع
            const signatureTitle = box.querySelector('.signature-title');
            if (signatureTitle) {
              signatureTitle.style.fontSize = '8pt';
              signatureTitle.style.color = '#6b7280';
            }
          });

          console.log('Starting html2canvas conversion...');

          // تحسين عرض الصفحة قبل التحويل ليكون كورقة A4 رسمية
          const pageElement = document.querySelector('.a4-page');
          if (pageElement) {
            pageElement.style.padding = '15mm'; // هوامش متساوية للورقة الرسمية
            pageElement.style.boxSizing = 'border-box';
            pageElement.style.backgroundColor = '#ffffff';
            pageElement.style.boxShadow = 'none';
            pageElement.style.position = 'relative';
            pageElement.style.minHeight = '297mm'; // ارتفاع ورقة A4
            pageElement.style.width = '210mm'; // عرض ورقة A4
          }

          // تحسين عرض ترويسة الصفحة لتتناسب مع التصميم في الصورة
          const pageHeader = document.querySelector('.page-header');
          if (pageHeader) {
            pageHeader.style.marginBottom = '20px';
            pageHeader.style.paddingBottom = '10px';
            pageHeader.style.borderBottom = '1px solid #4338ca';
            pageHeader.style.display = 'flex';
            pageHeader.style.justifyContent = 'space-between';
            pageHeader.style.alignItems = 'center';

            // تنظيف الترويسة من أي محتوى سابق
            pageHeader.innerHTML = '';

            // إنشاء عنصر الشعار
            const logoDiv = document.createElement('div');
            logoDiv.style.textAlign = 'right';
            logoDiv.style.flex = '1';

            // إضافة الشعار - استخدام الشعار الموجود بالفعل في الصفحة
            const existingLogo = document.querySelector('.company-logo');
            const logoImg = document.createElement('img');

            // استخدام مسار الشعار الموجود إذا كان متاحاً
            if (existingLogo && existingLogo.src) {
              logoImg.src = existingLogo.src;
            } else {
              logoImg.src = '../assets/images/Company logo.png';
            }

            logoImg.style.width = '70px';
            logoImg.style.height = 'auto';
            logoDiv.appendChild(logoImg);

            // إنشاء عنصر اسم الشركة
            const companyDiv = document.createElement('div');
            companyDiv.style.textAlign = 'center';
            companyDiv.style.flex = '2';

            // إضافة اسم الشركة
            const companyName = document.createElement('div');
            companyName.style.fontSize = '20pt';
            companyName.style.fontWeight = 'bold';
            companyName.style.color = '#4338ca';
            companyName.style.marginBottom = '5px';
            companyName.innerHTML = 'مجموعة ابراهيم الاحمدي اليمنية';
            companyDiv.appendChild(companyName);

            // إضافة معلومات الاتصال
            const contactInfo = document.createElement('div');
            contactInfo.style.fontSize = '9pt';
            contactInfo.style.color = '#4b5563';
            contactInfo.style.display = 'flex';
            contactInfo.style.justifyContent = 'center';
            contactInfo.style.gap = '15px';
            contactInfo.style.marginTop = '5px';

            // إضافة العنوان
            const addressSpan = document.createElement('span');
            addressSpan.innerHTML = '<i class="fas fa-map-marker-alt" style="color: #4338ca; margin-left: 5px;"></i> اليمن، حضرموت، المكلا';
            contactInfo.appendChild(addressSpan);

            // إضافة رقم الهاتف
            const phoneSpan = document.createElement('span');
            phoneSpan.innerHTML = '<i class="fas fa-phone" style="color: #4338ca; margin-left: 5px;"></i> 05318281';
            contactInfo.appendChild(phoneSpan);

            // إضافة البريد الإلكتروني
            const emailSpan = document.createElement('span');
            emailSpan.innerHTML = '<i class="fas fa-envelope" style="color: #4338ca; margin-left: 5px;"></i> ibrahim@al-ahmadigroup.com';
            contactInfo.appendChild(emailSpan);

            companyDiv.appendChild(contactInfo);

            // إنشاء عنصر رقم المستند
            const orderNumberDiv = document.createElement('div');
            orderNumberDiv.style.textAlign = 'left';
            orderNumberDiv.style.flex = '1';

            // إضافة رقم المستند
            const orderNumber = '<?php echo htmlspecialchars($order['order_number']); ?>';
            const orderNumberBox = document.createElement('div');
            orderNumberBox.style.border = '1px solid #4338ca';
            orderNumberBox.style.padding = '5px 10px';
            orderNumberBox.style.borderRadius = '5px';
            orderNumberBox.style.display = 'inline-block';
            orderNumberBox.style.backgroundColor = '#f0f0ff';
            orderNumberBox.style.fontSize = '10pt';
            orderNumberBox.style.fontWeight = 'bold';
            orderNumberBox.style.color = '#4338ca';
            orderNumberBox.innerHTML = `<div style="text-align: center; margin-bottom: 3px;">رقم المستند</div><div style="text-align: center;">${orderNumber}</div>`;
            orderNumberDiv.appendChild(orderNumberBox);

            // إضافة تاريخ المستند
            const orderDate = '<?php echo htmlspecialchars(date("Y/m/d", strtotime($order['created_at']))); ?>';
            const dateElement = document.createElement('div');
            dateElement.style.marginTop = '5px';
            dateElement.style.fontSize = '9pt';
            dateElement.style.color = '#4b5563';
            dateElement.style.textAlign = 'left';
            dateElement.innerHTML = `تاريخ الإصدار: ${orderDate}`;
            orderNumberDiv.appendChild(dateElement);

            // إضافة العناصر إلى الترويسة
            pageHeader.appendChild(logoDiv);
            pageHeader.appendChild(companyDiv);
            pageHeader.appendChild(orderNumberDiv);
          }

          // تحسين عرض عنوان المستند ليتناسب مع التصميم في الصورة
          const documentTitle = document.querySelector('.document-title h1');
          if (documentTitle) {
            documentTitle.style.fontSize = '16pt';
            documentTitle.style.fontWeight = 'bold';
            documentTitle.style.color = '#4338ca';
            documentTitle.style.textAlign = 'center';
            documentTitle.style.marginBottom = '10px';
            documentTitle.style.paddingBottom = '5px';
            documentTitle.style.position = 'relative';
            documentTitle.innerHTML = 'تفاصيل طلب الشحن';

            // إضافة زخرفة للعنوان لتتناسب مع التصميم في الصورة
            const titleContainer = document.querySelector('.document-title');
            if (titleContainer) {
              titleContainer.style.position = 'relative';
              titleContainer.style.marginBottom = '20px';
              titleContainer.style.textAlign = 'center';

              // إضافة نقطة زرقاء قبل العنوان
              const dotBefore = document.createElement('span');
              dotBefore.style.display = 'inline-block';
              dotBefore.style.width = '8px';
              dotBefore.style.height = '8px';
              dotBefore.style.backgroundColor = '#4338ca';
              dotBefore.style.borderRadius = '50%';
              dotBefore.style.marginRight = '10px';
              dotBefore.style.verticalAlign = 'middle';

              // إضافة نقطة زرقاء بعد العنوان
              const dotAfter = document.createElement('span');
              dotAfter.style.display = 'inline-block';
              dotAfter.style.width = '8px';
              dotAfter.style.height = '8px';
              dotAfter.style.backgroundColor = '#4338ca';
              dotAfter.style.borderRadius = '50%';
              dotAfter.style.marginLeft = '10px';
              dotAfter.style.verticalAlign = 'middle';

              // إضافة خط أفقي تحت العنوان
              const line = document.createElement('div');
              line.style.width = '250px';
              line.style.height = '1px';
              line.style.backgroundColor = '#4338ca';
              line.style.margin = '10px auto 0';

              // إضافة العناصر إلى العنوان
              documentTitle.insertBefore(dotBefore, documentTitle.firstChild);
              documentTitle.appendChild(dotAfter);
              titleContainer.appendChild(line);
            }
          }

          // تحسين عرض معلومات الطلب الأساسية بشكل رسمي
          const orderMetaBox = document.querySelector('.order-meta-box');
          if (orderMetaBox) {
            orderMetaBox.style.marginBottom = '25px';
            orderMetaBox.style.padding = '15px';
            orderMetaBox.style.backgroundColor = '#f9fafb';
            orderMetaBox.style.border = '1px solid #4338ca';
            orderMetaBox.style.borderRadius = '5px';
            orderMetaBox.style.boxShadow = '0 1px 3px rgba(0, 0, 0, 0.1)';

            // تحسين عرض عناوين البيانات
            const metaLabels = orderMetaBox.querySelectorAll('.meta-label');
            metaLabels.forEach(label => {
              label.style.fontSize = '9pt';
              label.style.fontWeight = 'bold';
              label.style.color = '#4338ca';
              label.style.marginBottom = '3px';
            });

            // تحسين عرض قيم البيانات
            const metaValues = orderMetaBox.querySelectorAll('.meta-value');
            metaValues.forEach(value => {
              value.style.fontSize = '10pt';
              value.style.fontWeight = 'normal';
              value.style.color = '#1f2937';
            });

            // إضافة عنوان لصندوق المعلومات
            const metaTitle = document.createElement('div');
            metaTitle.style.fontSize = '12pt';
            metaTitle.style.fontWeight = 'bold';
            metaTitle.style.color = '#4338ca';
            metaTitle.style.marginBottom = '10px';
            metaTitle.style.paddingBottom = '5px';
            metaTitle.style.borderBottom = '1px solid #4338ca';
            metaTitle.innerHTML = 'معلومات الطلب الأساسية';
            orderMetaBox.prepend(metaTitle);
          }

          // تحسين عرض العلامة المائية
          const watermarkElement = document.querySelector('.watermark');
          if (watermarkElement) {
            watermarkElement.style.opacity = '0.08'; // تقليل الشفافية للعلامة المائية
            watermarkElement.style.position = 'absolute';
            watermarkElement.style.top = '50%';
            watermarkElement.style.left = '50%';
            watermarkElement.style.transform = 'translate(-50%, -50%) rotate(-30deg)'; // إضافة دوران للعلامة المائية
            watermarkElement.style.zIndex = '-1';
            watermarkElement.style.width = '60%'; // تحديد حجم العلامة المائية
            watermarkElement.style.maxWidth = '500px';
            watermarkElement.style.height = 'auto';
            watermarkElement.style.pointerEvents = 'none'; // منع التفاعل مع العلامة المائية
          }

          // تحسين عرض الشعار
          const logoElement = document.querySelector('.company-logo');
          if (logoElement) {
            logoElement.style.maxWidth = '150px';
            logoElement.style.height = 'auto';
            logoElement.style.display = 'block';
            logoElement.style.margin = '0 auto 15px';
          }

          // استخدام نهج جديد لتوليد PDF
          console.log('Using new PDF generation approach...');

          // الحصول على اسم الملف من رقم الطلب
          const orderNumber = '<?php echo htmlspecialchars($order['order_number']); ?>';
          const fileName = `طلب_شحن_${orderNumber}.pdf`;

          // تحضير الصفحة للتحويل
          const clonedElement = element.cloneNode(true);
          document.body.appendChild(clonedElement);
          clonedElement.style.position = 'absolute';
          clonedElement.style.left = '-9999px';
          clonedElement.style.overflow = 'visible';
          clonedElement.style.height = 'auto';
          clonedElement.style.width = '210mm';

          // تحسين عرض الصور
          const images = clonedElement.querySelectorAll('img');
          images.forEach(img => {
            img.style.maxWidth = '100%';
            img.style.height = 'auto';
          });

          // تحويل الصفحة إلى صورة
          html2canvas(clonedElement, {
            scale: 2, // زيادة الجودة
            useCORS: true,
            logging: true,
            allowTaint: true,
            backgroundColor: '#ffffff',
            letterRendering: true, // تحسين عرض النصوص
            scrollY: 0,
            scrollX: 0
          }).then(function(canvas) {
            console.log('Canvas generated successfully');

            try {
              // إزالة النسخة المستنسخة من الصفحة
              document.body.removeChild(clonedElement);

              // استعادة اسم الشركة الأصلي
              if (companyNameElement) {
                companyNameElement.innerHTML = originalHTML;
              }

              console.log('Creating PDF from canvas...');

              // تهيئة jsPDF
              const { jsPDF } = window.jspdf;

              // إنشاء ملف PDF جديد بحجم A4
              const pdf = new jsPDF({
                orientation: 'portrait',
                unit: 'mm',
                format: 'a4',
                compress: true
              });

              // تعيين خصائص PDF
              pdf.setProperties({
                title: `طلب شحن ${orderNumber}`,
                subject: 'تفاصيل طلب الشحن',
                author: 'مجموعة ابراهيم الاحمدي اليمنية',
                keywords: 'طلب شحن, مجموعة الاحمدي',
                creator: 'نظام الشحن'
              });

              // إضافة الصورة إلى ملف PDF
              const imgData = canvas.toDataURL('image/jpeg', 1.0); // أعلى جودة
              console.log('Image data URL created');

              const imgWidth = 210; // عرض A4 بالملم
              const pageHeight = 297; // ارتفاع A4 بالملم
              const imgHeight = canvas.height * imgWidth / canvas.width;

              console.log(`Canvas dimensions: ${canvas.width}x${canvas.height}, PDF dimensions: ${imgWidth}x${imgHeight}`);

              // إضافة الصورة إلى ملف PDF مع ضبط الهوامش
              const margin = 0; // بدون هوامش

              // التحقق مما إذا كانت الصورة أطول من صفحة A4 واحدة
              if (imgHeight <= pageHeight) {
                // إذا كانت الصورة تناسب صفحة واحدة
                pdf.addImage(imgData, 'JPEG', margin, margin, imgWidth - (margin * 2), imgHeight - (margin * 2));
                console.log('Added image to single page PDF');
              } else {
                // إذا كانت الصورة أطول من صفحة واحدة، قم بتقسيمها على عدة صفحات
                let heightLeft = imgHeight;
                let position = 0;
                let page = 1;

                // إضافة الصفحة الأولى
                pdf.addImage(imgData, 'JPEG', margin, margin, imgWidth - (margin * 2), Math.min(pageHeight - (margin * 2), imgHeight));
                heightLeft -= (pageHeight - (margin * 2));
                position = -(pageHeight - (margin * 2));
                console.log(`Added page 1 of multi-page PDF, height left: ${heightLeft}`);

                // إضافة الصفحات الإضافية إذا لزم الأمر
                while (heightLeft > 0) {
                  pdf.addPage();
                  page++;
                  pdf.addImage(imgData, 'JPEG', margin, position, imgWidth - (margin * 2), imgHeight);
                  heightLeft -= (pageHeight - (margin * 2));
                  position -= (pageHeight - (margin * 2));
                  console.log(`Added page ${page} of multi-page PDF, height left: ${heightLeft}`);
                }
              }

              console.log('Saving PDF file...');

              // تحميل ملف PDF
              pdf.save(fileName);

              console.log('PDF saved successfully');

              // إغلاق رسالة الانتظار وإظهار رسالة نجاح
              Swal.fire({
                title: 'تم التحميل!',
                text: 'تم تحميل ملف PDF بنجاح',
                icon: 'success',
                confirmButtonColor: '#4f46e5',
                confirmButtonText: 'حسناً'
              });
            } catch (innerError) {
              console.error('Error in PDF creation after canvas generation:', innerError);
              Swal.fire({
                title: 'خطأ!',
                text: 'حدث خطأ أثناء إنشاء ملف PDF: ' + (innerError.message || 'خطأ غير معروف'),
                icon: 'error',
                confirmButtonColor: '#4f46e5',
                confirmButtonText: 'حسناً'
              });
            }
          }).catch(function(error) {
            console.error('Error in html2canvas:', error);
            Swal.fire({
              title: 'خطأ!',
              text: 'حدث خطأ أثناء تحويل الصفحة إلى صورة: ' + (error.message || 'خطأ غير معروف'),
              icon: 'error',
              confirmButtonColor: '#4f46e5',
              confirmButtonText: 'حسناً'
            });
          });


        } catch (error) {
          console.error('Error in generatePDF function:', error);
          Swal.fire({
            title: 'خطأ!',
            text: 'حدث خطأ أثناء تحضير ملف PDF: ' + (error.message || 'خطأ غير معروف'),
            icon: 'error',
            confirmButtonColor: '#4f46e5',
            confirmButtonText: 'حسناً'
          });
        }
      }

      // دالة الطباعة المباشرة
      function printDirectly() {
        // تحضير الصفحة للطباعة
        document.querySelectorAll('.info-table, .content-section, .page-content, .a4-page').forEach(el => {
          el.style.display = 'block';
          el.style.visibility = 'visible';
          el.style.overflow = 'visible';
        });

        document.querySelectorAll('.info-table tr').forEach(el => {
          el.style.display = 'table-row';
          el.style.visibility = 'visible';
        });

        document.querySelectorAll('.info-table th, .info-table td').forEach(el => {
          el.style.display = 'table-cell';
          el.style.visibility = 'visible';
          el.style.color = '#000';
        });

        // تحسين عرض الجداول للطباعة لتتناسب مع التصميم في الصورة
        document.querySelectorAll('.info-table').forEach(table => {
          table.style.width = '100%';
          table.style.borderCollapse = 'collapse';
          table.style.border = '1px solid #4338ca';
          table.style.marginBottom = '20px';
          table.style.pageBreakInside = 'avoid';
          table.style.boxShadow = 'none';
        });

        document.querySelectorAll('.info-table th').forEach(cell => {
          cell.style.border = '1px solid #4338ca';
          cell.style.padding = '3mm';
          cell.style.color = '#ffffff';
          cell.style.backgroundColor = '#4338ca';
          cell.style.fontWeight = 'bold';
          cell.style.fontSize = '10pt';
          cell.style.textAlign = 'center';
        });

        document.querySelectorAll('.info-table td').forEach(cell => {
          cell.style.border = '1px solid #4338ca';
          cell.style.padding = '3mm';
          cell.style.color = '#1f2937';
          cell.style.fontSize = '10pt';
          cell.style.textAlign = 'center';
        });

        // تطبيق ألوان خاصة لبعض الخلايا (مثل حالة الطلب)
        document.querySelectorAll('.status-cell').forEach(cell => {
          if (cell.textContent.includes('مقبول')) {
            cell.style.backgroundColor = '#dcfce7'; // لون أخضر فاتح
            cell.style.color = '#166534'; // لون أخضر غامق
          } else if (cell.textContent.includes('معلق')) {
            cell.style.backgroundColor = '#fef9c3'; // لون أصفر فاتح
            cell.style.color = '#854d0e'; // لون أصفر غامق
          }
        });

        // تحسين عرض العلامة المائية
        const watermarkElement = document.querySelector('.watermark');
        if (watermarkElement) {
          watermarkElement.style.opacity = '0.08';
          watermarkElement.style.position = 'absolute';
          watermarkElement.style.top = '50%';
          watermarkElement.style.left = '50%';
          watermarkElement.style.transform = 'translate(-50%, -50%) rotate(-30deg)';
          watermarkElement.style.zIndex = '-1';
          watermarkElement.style.width = '60%';
          watermarkElement.style.maxWidth = '500px';
          watermarkElement.style.height = 'auto';
        }

        // تحسين عرض عناوين الأقسام لتتناسب مع التصميم في الصورة
        document.querySelectorAll('.section-title').forEach(title => {
          title.style.position = 'relative';
          title.style.fontSize = '14pt';
          title.style.fontWeight = 'bold';
          title.style.color = '#4338ca';
          title.style.marginTop = '20px';
          title.style.marginBottom = '15px';
          title.style.paddingBottom = '5px';
          title.style.textAlign = 'right';
          title.style.borderBottom = 'none';

          // تنظيف العنوان من أي محتوى إضافي
          const titleText = title.textContent;
          title.innerHTML = '';

          // إضافة أيقونة مميزة قبل العنوان
          const icon = document.createElement('i');
          if (titleText.includes('العميل')) {
            icon.className = 'fas fa-user';
          } else if (titleText.includes('البضاعة')) {
            icon.className = 'fas fa-box';
          } else if (titleText.includes('المالية')) {
            icon.className = 'fas fa-money-bill';
          }
          icon.style.marginLeft = '8px';
          icon.style.color = '#4338ca';
          icon.style.fontSize = '14pt';
          title.appendChild(icon);

          // إضافة نص العنوان
          const textSpan = document.createElement('span');
          textSpan.textContent = titleText.trim();
          title.appendChild(textSpan);

          // إضافة مربع ملون خلف العنوان
          title.style.backgroundColor = '#eef2ff'; // لون خلفية فاتح
          title.style.padding = '5px 10px';
          title.style.borderRight = '4px solid #4338ca';
          title.style.borderRadius = '0 5px 5px 0';
        });

        // تحسين عرض عنوان المستند ليتناسب مع التصميم في الصورة
        const documentTitle = document.querySelector('.document-title h1');
        if (documentTitle) {
          documentTitle.style.fontSize = '16pt';
          documentTitle.style.fontWeight = 'bold';
          documentTitle.style.color = '#4338ca';
          documentTitle.style.textAlign = 'center';
          documentTitle.style.marginBottom = '10px';
          documentTitle.style.paddingBottom = '5px';
          documentTitle.style.position = 'relative';
          documentTitle.innerHTML = 'تفاصيل طلب الشحن';

          // إضافة زخرفة للعنوان لتتناسب مع التصميم في الصورة
          const titleContainer = document.querySelector('.document-title');
          if (titleContainer) {
            titleContainer.style.position = 'relative';
            titleContainer.style.marginBottom = '20px';
            titleContainer.style.textAlign = 'center';

            // إضافة نقطة زرقاء قبل العنوان
            const dotBefore = document.createElement('span');
            dotBefore.style.display = 'inline-block';
            dotBefore.style.width = '8px';
            dotBefore.style.height = '8px';
            dotBefore.style.backgroundColor = '#4338ca';
            dotBefore.style.borderRadius = '50%';
            dotBefore.style.marginRight = '10px';
            dotBefore.style.verticalAlign = 'middle';

            // إضافة نقطة زرقاء بعد العنوان
            const dotAfter = document.createElement('span');
            dotAfter.style.display = 'inline-block';
            dotAfter.style.width = '8px';
            dotAfter.style.height = '8px';
            dotAfter.style.backgroundColor = '#4338ca';
            dotAfter.style.borderRadius = '50%';
            dotAfter.style.marginLeft = '10px';
            dotAfter.style.verticalAlign = 'middle';

            // إضافة خط أفقي تحت العنوان
            const line = document.createElement('div');
            line.style.width = '250px';
            line.style.height = '1px';
            line.style.backgroundColor = '#4338ca';
            line.style.margin = '10px auto 0';

            // إضافة العناصر إلى العنوان
            documentTitle.insertBefore(dotBefore, documentTitle.firstChild);
            documentTitle.appendChild(dotAfter);
            titleContainer.appendChild(line);
          }
        }

        // تحسين عرض تذييل الصفحة
        const pageFooter = document.querySelector('.page-footer');
        if (pageFooter) {
          pageFooter.style.borderTop = '1px solid #4338ca';
          pageFooter.style.marginTop = '30px';
          pageFooter.style.paddingTop = '10px';
          pageFooter.style.textAlign = 'center';
          pageFooter.style.fontSize = '9pt';
          pageFooter.style.color = '#6b7280';
        }

        // تحسين عرض قسم التوقيعات
        const footerSignature = document.querySelector('.footer-signature');
        if (footerSignature) {
          footerSignature.style.marginTop = '40px';
          footerSignature.style.marginBottom = '20px';
          footerSignature.style.display = 'flex';
          footerSignature.style.justifyContent = 'space-around';

          // تحسين عرض صناديق التوقيع
          const signatureBoxes = footerSignature.querySelectorAll('.signature-box');
          signatureBoxes.forEach(box => {
            box.style.textAlign = 'center';
            box.style.padding = '10px';
            box.style.minWidth = '120px';

            // تحسين عرض خط التوقيع
            const signatureLine = box.querySelector('.signature-line');
            if (signatureLine) {
              signatureLine.style.marginTop = '25px';
              signatureLine.style.marginBottom = '5px';
              signatureLine.style.borderTop = '1px solid #4338ca';
              signatureLine.style.width = '100%';
            }
          });
        }

        // طباعة الصفحة
        window.print();
      }

      // زر تحميل PDF
      const printOrderBtn = document.getElementById('printOrderBtn');
      if (printOrderBtn) {
        printOrderBtn.addEventListener('click', function() {
          // تنفيذ دالة إنشاء PDF مباشرة
          try {
            generatePDF();
          } catch (error) {
            console.error('Error calling generatePDF:', error);
            Swal.fire({
              title: 'خطأ!',
              text: 'حدث خطأ أثناء إنشاء ملف PDF: ' + (error.message || 'خطأ غير معروف'),
              icon: 'error',
              confirmButtonColor: '#4f46e5',
              confirmButtonText: 'حسناً'
            });
          }
        });
      }

      // زر حذف الطلب
      const deleteOrderBtn = document.getElementById('deleteOrderBtn');
      if (deleteOrderBtn) {
        deleteOrderBtn.addEventListener('click', function() {
          Swal.fire({
            title: 'هل أنت متأكد؟',
            text: 'سيتم حذف هذا الطلب نهائياً ولا يمكن استرجاعه!',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#ef4444',
            cancelButtonColor: '#64748b',
            confirmButtonText: 'نعم، قم بالحذف!',
            cancelButtonText: 'إلغاء',
            reverseButtons: true
          }).then((result) => {
            if (result.isConfirmed) {
              // إظهار مؤشر التحميل
              Swal.fire({
                title: 'جاري الحذف...',
                text: 'يرجى الانتظار',
                allowOutsideClick: false,
                didOpen: () => {
                  Swal.showLoading();
                }
              });

              // إرسال طلب حذف الطلب
              fetch('../api/delete_order.php', {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify({ id: <?php echo $order['id']; ?> })
              })
              .then(response => response.json())
              .then(data => {
                if (data.status === 'success') {
                  Swal.fire({
                    title: 'تم الحذف!',
                    text: 'تم حذف الطلب بنجاح.',
                    icon: 'success',
                    confirmButtonColor: '#4f46e5',
                    confirmButtonText: 'حسناً'
                  }).then(() => {
                    window.location.href = 'orders.php';
                  });
                } else {
                  Swal.fire({
                    title: 'خطأ!',
                    text: data.message || 'فشل في حذف الطلب',
                    icon: 'error',
                    confirmButtonColor: '#4f46e5',
                    confirmButtonText: 'حسناً'
                  });
                }
              })
              .catch(error => {
                console.error('Error deleting order:', error);
                Swal.fire({
                  title: 'خطأ!',
                  text: 'حدث خطأ أثناء حذف الطلب',
                  icon: 'error',
                  confirmButtonColor: '#3730a3',
                  confirmButtonText: 'حسناً'
                });
              });
            }
          });
        });
      }
    });
  </script>

  <!-- تنسيقات الطباعة -->
  <style>
    /* تنسيقات عامة للطباعة */
    @page {
      size: A4 portrait;
      margin: 0;
    }

    /* تنسيقات خاصة بالطباعة */
    @media print {
      @page {
        size: A4 portrait;
        margin: 0;
      }

      html, body {
        width: 100%;
        height: auto;
        margin: 0;
        padding: 0;
        background-color: white !important;
        font-size: 10pt;
        overflow: visible !important;
        -webkit-print-color-adjust: exact !important;
        print-color-adjust: exact !important;
      }

      body * {
        visibility: visible !important;
        overflow: visible !important;
      }

      .page-container {
        padding: 0 !important;
        margin: 0 !important;
        width: 100% !important;
        display: block !important;
        overflow: visible !important;
        position: static !important;
      }

      .top-navbar, .actions-bar {
        display: none !important;
      }

      .a4-page {
        width: 100% !important;
        height: auto !important;
        min-height: 297mm !important;
        padding: 25mm 20mm 20mm 20mm !important; /* هوامش مناسبة للورقة الرسمية (أعلى، يمين، أسفل، يسار) */
        margin: 0 !important;
        box-shadow: none !important;
        border: 1px solid #e5e7eb !important;
        overflow: visible !important;
        position: static !important;
        display: block !important;
        box-sizing: border-box !important;
        page-break-after: always !important;
        background-color: white !important;
      }

      .page-header {
        margin-bottom: 15mm !important;
        padding-bottom: 10mm !important;
        border-bottom: 2px solid #4338ca !important;
      }

      .company-logo {
        display: block !important;
        max-width: 120px !important;
        height: auto !important;
        margin: 0 auto 10mm !important;
      }

      .company-name {
        color: #000 !important;
        text-shadow: none !important;
        font-size: 16pt !important;
        font-weight: bold !important;
        text-align: center !important;
        margin-bottom: 5mm !important;
      }

      .company-name span {
        font-size: inherit !important;
      }

      .company-name::after {
        display: none !important;
      }

      .company-details {
        font-size: 9pt !important;
        margin: 2mm 0 !important;
        text-align: center !important;
      }

      .document-title {
        margin: 5mm 0 15mm 0 !important;
        text-align: center !important;
      }

      .document-title h1 {
        color: #4338ca !important;
        border-bottom: 2px solid #4338ca !important;
        font-size: 16pt !important;
        padding: 3mm 15mm !important;
        font-weight: bold !important;
        display: inline-block !important;
        margin: 0 auto !important;
      }

      .order-meta-box {
        margin-bottom: 15mm !important;
        page-break-inside: avoid !important;
        padding: 5mm !important;
        background-color: #f9fafb !important;
        border: 1px solid #e5e7eb !important;
        border-radius: 5px !important;
      }

      .meta-label {
        font-size: 8pt !important;
        color: #4b5563 !important;
        margin-bottom: 1mm !important;
      }

      .meta-value {
        font-size: 10pt !important;
        font-weight: bold !important;
        color: #1f2937 !important;
      }

      .order-status {
        font-size: 9pt !important;
        padding: 1mm 3mm !important;
        border-radius: 3mm !important;
      }

      .content-section {
        margin-bottom: 15mm !important;
        page-break-inside: avoid !important;
      }

      .section-title {
        font-size: 14pt !important;
        margin-bottom: 5mm !important;
        padding-bottom: 2mm !important;
        color: #4338ca !important;
        font-weight: bold !important;
        border-bottom: 2px solid #4338ca !important;
      }

      .section-title i {
        color: #4338ca !important;
        margin-left: 3mm !important;
      }

      .info-table {
        width: 100% !important;
        border-collapse: collapse !important;
        page-break-inside: avoid !important;
        display: table !important;
        visibility: visible !important;
        color: #000 !important;
        border: 2px solid #4338ca !important;
        margin-bottom: 10mm !important;
        box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05) !important;
      }

      .info-table th {
        border: 1px solid #4338ca !important;
        padding: 5mm !important;
        color: #ffffff !important;
        background-color: #4338ca !important;
        visibility: visible !important;
        display: table-cell !important;
        font-size: 10pt !important;
        font-weight: bold !important;
        text-align: center !important;
      }

      .info-table td {
        border: 1px solid #d1d5db !important;
        padding: 4mm !important;
        color: #1f2937 !important;
        background-color: white !important;
        visibility: visible !important;
        display: table-cell !important;
        font-size: 9pt !important;
        text-align: center !important;
      }

      .info-table tr:nth-child(even) td {
        background-color: #f9fafb !important;
      }

      .info-table tr {
        display: table-row !important;
        visibility: visible !important;
      }

      .content-section {
        display: block !important;
        visibility: visible !important;
        margin-bottom: 10mm !important;
        page-break-inside: avoid !important;
        overflow: visible !important;
      }

      .page-content {
        display: block !important;
        visibility: visible !important;
        overflow: visible !important;
      }

      .footer-signature {
        margin-top: 20mm !important;
        margin-bottom: 10mm !important;
        page-break-inside: avoid !important;
        display: flex !important;
        justify-content: space-around !important;
      }

      .signature-box {
        max-width: 150px !important;
        text-align: center !important;
      }

      .signature-line {
        margin-top: 15mm !important;
        margin-bottom: 3mm !important;
        border-top: 1px solid #000 !important;
        width: 80% !important;
        margin-left: auto !important;
        margin-right: auto !important;
      }

      .signature-name {
        font-size: 10pt !important;
        font-weight: bold !important;
      }

      .signature-title {
        font-size: 8pt !important;
        color: #4b5563 !important;
      }

      .page-footer {
        position: relative !important;
        bottom: auto !important;
        left: auto !important;
        right: auto !important;
        border-top: 1px solid #4338ca !important;
        padding-top: 5mm !important;
        margin-top: 15mm !important;
        display: block !important;
        width: 100% !important;
        overflow: visible !important;
        font-size: 9pt !important;
        text-align: center !important;
      }

      .watermark {
        display: block !important;
        opacity: 0.08 !important;
        width: 60% !important;
        height: auto !important;
        position: absolute !important;
        top: 50% !important;
        left: 50% !important;
        transform: translate(-50%, -50%) rotate(-30deg) !important;
        z-index: 0 !important;
        max-width: 500px !important;
        pointer-events: none !important;
      }

      /* إصلاح مشكلة عدم ظهور الأيقونات */
      .fas {
        font-family: 'Font Awesome 5 Free' !important;
        font-weight: 900 !important;
        display: inline-block !important;
        visibility: visible !important;
      }
    }
  </style>
</body>

</html>
