/**
 * ملف JavaScript لمعالجة النماذج
 */

document.addEventListener('DOMContentLoaded', function() {
    // تهيئة جميع النماذج
    initForms();
});

/**
 * دالة لتهيئة جميع النماذج في الصفحة
 */
function initForms() {
    // تهيئة نموذج إنشاء/تعديل الطلب
    const orderForm = document.getElementById('shipmentForm');
    if (orderForm) {
        orderForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            // استدعاء الدالة المحسنة لإرسال النموذج
            submitFormWithErrorHandling(this, '../api/save_order.php', function(data) {
                // دالة تنفذ عند نجاح العملية
                Swal.fire({
                    icon: 'success',
                    title: 'تم بنجاح!',
                    text: data.message,
                    confirmButtonText: 'حسناً'
                }).then(() => {
                    // التوجيه إلى صفحة إدارة الطلبات أو عرض الطلب
                    window.location.href = 'orders.php';
                });
            });
        });
    }
}

/**
 * دالة محسنة لإرسال النموذج مع معالجة الأخطاء
 * @param {HTMLFormElement} form - عنصر النموذج
 * @param {string} url - عنوان URL للإرسال
 * @param {Function} successCallback - دالة تنفذ عند نجاح العملية
 */
function submitFormWithErrorHandling(form, url, successCallback) {
    // التحقق من صحة النموذج
    if (!form.checkValidity()) {
        // عرض رسائل التحقق الافتراضية للمتصفح
        form.reportValidity();
        return;
    }
    
    // إنشاء كائن FormData
    const formData = new FormData(form);
    
    // عرض مؤشر التحميل
    const submitBtn = form.querySelector('button[type="submit"]');
    const originalBtnContent = submitBtn ? submitBtn.innerHTML : '';
    
    if (submitBtn) {
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin ml-2"></i> <span>جاري الحفظ...</span>';
    }
    
    // إرسال البيانات
    fetch(url, {
        method: 'POST',
        body: formData
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`فشل في الاتصال بالخادم (${response.status})`);
        }
        return response.json();
    })
    .then(data => {
        // إعادة تفعيل زر الإرسال
        if (submitBtn) {
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnContent;
        }
        
        if (data.status === 'success') {
            // استدعاء دالة النجاح
            if (typeof successCallback === 'function') {
                successCallback(data);
            }
        } else {
            // عرض رسائل الخطأ
            if (data.errors) {
                displayFormErrors(data.errors);
            } else {
                throw new Error(data.message || 'حدث خطأ أثناء معالجة البيانات');
            }
        }
    })
    .catch(error => {
        console.error('خطأ في إرسال النموذج:', error);
        
        // إعادة تفعيل زر الإرسال
        if (submitBtn) {
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalBtnContent;
        }
        
        // عرض رسالة الخطأ
        Swal.fire({
            icon: 'error',
            title: 'خطأ',
            text: error.message || 'حدث خطأ أثناء إرسال النموذج',
            confirmButtonText: 'حسناً'
        });
    });
}

/**
 * دالة لعرض أخطاء النموذج
 * @param {Object} errors - كائن يحتوي على أخطاء النموذج
 */
function displayFormErrors(errors) {
    // إزالة جميع رسائل الخطأ السابقة
    document.querySelectorAll('.error-message').forEach(el => el.remove());
    
    // إعادة تعيين تنسيق جميع الحقول
    document.querySelectorAll('.form-input').forEach(el => {
        el.classList.remove('border-red-500');
    });
    
    // عرض الأخطاء الجديدة
    for (const field in errors) {
        const inputElement = document.getElementById(field);
        if (inputElement) {
            // إضافة تنسيق الخطأ للحقل
            inputElement.classList.add('border-red-500');
            
            // إنشاء عنصر رسالة الخطأ
            const errorElement = document.createElement('div');
            errorElement.className = 'error-message text-red-500 text-sm mt-1';
            errorElement.textContent = errors[field];
            
            // إضافة رسالة الخطأ بعد الحقل
            inputElement.parentNode.appendChild(errorElement);
        }
    }
    
    // التمرير إلى أول حقل به خطأ
    const firstErrorField = document.querySelector('.border-red-500');
    if (firstErrorField) {
        firstErrorField.scrollIntoView({ behavior: 'smooth', block: 'center' });
        firstErrorField.focus();
    }
}

/**
 * دالة لعرض رسالة نجاح
 * @param {string} message - نص الرسالة
 */
function showSuccess(message) {
    Swal.fire({
        icon: 'success',
        title: 'تم بنجاح!',
        text: message,
        confirmButtonText: 'حسناً'
    });
}

/**
 * دالة لعرض رسالة خطأ
 * @param {string} message - نص الرسالة
 */
function showError(message) {
    Swal.fire({
        icon: 'error',
        title: 'خطأ',
        text: message,
        confirmButtonText: 'حسناً'
    });
}
