<?php
/**
 * ملف لإضافة طلبات تجريبية إلى قاعدة البيانات
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/db_connection.php';

// دالة لإنشاء رقم طلب فريد بناءً على رمز الفرع
function generateOrderNumber($branchCode) {
    global $conn;

    // الحصول على السنة الحالية
    $year = date('y');

    // التحقق من وجود تسلسل للفرع
    $stmt = $conn->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
    $stmt->execute([$branchCode]);
    $sequence = $stmt->fetch();

    if ($sequence) {
        // زيادة رقم التسلسل
        $sequenceNumber = $sequence['last_sequence_number'] + 1;

        // تحديث رقم التسلسل في قاعدة البيانات
        $updateStmt = $conn->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
        $updateStmt->execute([$sequenceNumber, $branchCode]);
    } else {
        // إنشاء تسلسل جديد للفرع
        $sequenceNumber = 1;
        $insertStmt = $conn->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, ?)");
        $insertStmt->execute([$branchCode, $sequenceNumber]);
    }

    // تنسيق رقم الطلب: رمز الفرع + السنة + رقم التسلسل (مثال: ADN23-01)
    return $branchCode . $year . '-' . str_pad($sequenceNumber, 2, '0', STR_PAD_LEFT);
}

// دالة لإضافة طلب جديد
function addOrder($orderData) {
    global $conn;

    try {
        // بدء المعاملة
        $conn->beginTransaction();

        // إعداد استعلام الإدراج
        $stmt = $conn->prepare("
            INSERT INTO orders (
                order_number, client_name, service_type, branch_id,
                departure_area, second_destination, delivery_area, goods_name, goods_type,
                country_of_origin, weight, quantity, unit, service_fees,
                currency, payment_type, delivery_duration, amount_paid, remaining_amount,
                order_upload_date, scheduled_delivery_date, recipient_name, phone_number, delivery_company,
                employee_name, notes, amount_agreed_with_customer, amount_agreed_with_agent, other_expenses,
                profit, payment_status, status
            ) VALUES (
                :order_number, :client_name, :service_type, :branch_id,
                :departure_area, :second_destination, :delivery_area, :goods_name, :goods_type,
                :country_of_origin, :weight, :quantity, :unit, :service_fees,
                :currency, :payment_type, :delivery_duration, :amount_paid, :remaining_amount,
                :order_upload_date, :scheduled_delivery_date, :recipient_name, :phone_number, :delivery_company,
                :employee_name, :notes, :amount_agreed_with_customer, :amount_agreed_with_agent, :other_expenses,
                :profit, :payment_status, :status
            )
        ");

        // تنفيذ الاستعلام
        $stmt->execute($orderData);

        // الحصول على معرف الطلب المدرج
        $orderId = $conn->lastInsertId();

        // تأكيد المعاملة
        $conn->commit();

        return [
            'status' => 'success',
            'message' => 'تم إضافة الطلب بنجاح',
            'orderId' => $orderId
        ];
    } catch (PDOException $e) {
        // التراجع عن المعاملة في حالة حدوث خطأ
        $conn->rollBack();

        return [
            'status' => 'error',
            'message' => 'فشل في إضافة الطلب: ' . $e->getMessage()
        ];
    }
}

// الحصول على معرفات الفروع
function getBranchId($branchName) {
    global $conn;

    $stmt = $conn->prepare("SELECT id FROM branches WHERE name = ?");
    $stmt->execute([$branchName]);
    $branch = $stmt->fetch();

    return $branch ? $branch['id'] : null;
}

// إضافة الفروع إذا لم تكن موجودة
function addBranchesIfNotExist() {
    global $conn;

    $branches = [
        ['name' => 'فرع عدن', 'code' => 'ADN', 'address' => 'عدن - المنصورة', 'phone' => '777123456'],
        ['name' => 'فرع صنعاء', 'code' => 'SNA', 'address' => 'صنعاء - شارع الستين', 'phone' => '777654321'],
        ['name' => 'فرع المكلا', 'code' => 'MKL', 'address' => 'المكلا - الشرج', 'phone' => '777987654'],
        ['name' => 'فرع الصين', 'code' => 'CN', 'address' => 'قوانزو - الصين', 'phone' => '777456789']
    ];

    foreach ($branches as $branch) {
        $stmt = $conn->prepare("SELECT id FROM branches WHERE code = ?");
        $stmt->execute([$branch['code']]);

        if (!$stmt->fetch()) {
            $insertStmt = $conn->prepare("INSERT INTO branches (name, code, address, phone) VALUES (?, ?, ?, ?)");
            $insertStmt->execute([$branch['name'], $branch['code'], $branch['address'], $branch['phone']]);
        }
    }
}

// التحقق من وجود جدول الطلبات
function checkOrdersTable() {
    global $conn;

    try {
        $stmt = $conn->query("SELECT 1 FROM orders LIMIT 1");
        return true;
    } catch (PDOException $e) {
        return false;
    }
}

// إنشاء طلبات تجريبية
function createTestOrders() {
    global $conn;

    // إضافة الفروع إذا لم تكن موجودة
    addBranchesIfNotExist();

    // الحصول على معرفات الفروع
    $adenBranchId = getBranchId('فرع عدن');
    $sanaaBranchId = getBranchId('فرع صنعاء');
    $mukallaBranchId = getBranchId('فرع المكلا');
    $chinaBranchId = getBranchId('فرع الصين');

    // بيانات الطلبات التجريبية
    $testOrders = [
        [
            'order_number' => generateOrderNumber('ADN'),
            'client_name' => 'محمد علي',
            'service_type' => 'شحن بحري',
            'branch_id' => $adenBranchId,
            'departure_area' => 'عدن',
            'second_destination' => '',
            'delivery_area' => 'صنعاء',
            'goods_name' => 'أجهزة كهربائية',
            'goods_type' => 'إلكترونيات',
            'country_of_origin' => 'الصين',
            'weight' => 150.00,
            'quantity' => 10,
            'unit' => 'كرتون',
            'service_fees' => 1000.00,
            'currency' => 'دولار',
            'payment_type' => 'كاش',
            'delivery_duration' => '3-5 أيام',
            'amount_paid' => 500.00,
            'remaining_amount' => 500.00,
            'order_upload_date' => date('Y-m-d'),
            'scheduled_delivery_date' => date('Y-m-d', strtotime('+5 days')),
            'recipient_name' => 'أحمد محمد',
            'phone_number' => '777123456',
            'delivery_company' => 'شركة النقل السريع',
            'employee_name' => 'خالد عبدالله',
            'notes' => 'يرجى الاتصال قبل التسليم',
            'amount_agreed_with_customer' => 1200.00,
            'amount_agreed_with_agent' => 800.00,
            'other_expenses' => 100.00,
            'profit' => 300.00,
            'payment_status' => 'جزئي',
            'status' => 'قيد المعالجة'
        ],
        [
            'order_number' => generateOrderNumber('SNA'),
            'client_name' => 'علي محمد',
            'service_type' => 'شحن جوي',
            'branch_id' => $sanaaBranchId,
            'departure_area' => 'صنعاء',
            'second_destination' => 'دبي',
            'delivery_area' => 'المكلا',
            'goods_name' => 'ملابس',
            'goods_type' => 'نسيج',
            'country_of_origin' => 'تركيا',
            'weight' => 100.00,
            'quantity' => 5,
            'unit' => 'كيس',
            'service_fees' => 800.00,
            'currency' => 'دولار',
            'payment_type' => 'تحويل بنكي',
            'delivery_duration' => '1-2 أيام',
            'amount_paid' => 800.00,
            'remaining_amount' => 0.00,
            'order_upload_date' => date('Y-m-d'),
            'scheduled_delivery_date' => date('Y-m-d', strtotime('+2 days')),
            'recipient_name' => 'سعيد عمر',
            'phone_number' => '777654321',
            'delivery_company' => 'شركة النقل الجوي',
            'employee_name' => 'محمد سالم',
            'notes' => 'بضاعة قابلة للكسر',
            'amount_agreed_with_customer' => 1000.00,
            'amount_agreed_with_agent' => 600.00,
            'other_expenses' => 50.00,
            'profit' => 350.00,
            'payment_status' => 'مدفوع',
            'status' => 'في الطريق'
        ],
        // طلبات إضافية من الصورة
        [
            'order_number' => generateOrderNumber('ADN'),
            'client_name' => 'حسن علي',
            'service_type' => 'شحن بري',
            'branch_id' => $adenBranchId,
            'departure_area' => 'عدن',
            'second_destination' => '',
            'delivery_area' => 'المكلا',
            'goods_name' => 'أثاث منزلي',
            'goods_type' => 'أثاث',
            'country_of_origin' => 'الصين',
            'weight' => 150.00,
            'quantity' => 10,
            'unit' => 'قطعة',
            'service_fees' => 1000.00,
            'currency' => 'دولار',
            'payment_type' => 'كاش',
            'delivery_duration' => '5-7 أيام',
            'amount_paid' => 1000.00,
            'remaining_amount' => 0.00,
            'order_upload_date' => date('Y-m-d'),
            'scheduled_delivery_date' => date('Y-m-d', strtotime('+7 days')),
            'recipient_name' => 'عبدالله سالم',
            'phone_number' => '777987654',
            'delivery_company' => 'شركة النقل البري',
            'employee_name' => 'سالم عمر',
            'notes' => 'توصيل إلى باب المنزل',
            'amount_agreed_with_customer' => 1200.00,
            'amount_agreed_with_agent' => 700.00,
            'other_expenses' => 150.00,
            'profit' => 350.00,
            'payment_status' => 'مدفوع',
            'status' => 'قيد المعالجة'
        ],
        [
            'order_number' => generateOrderNumber('MKL'),
            'client_name' => 'عمر سعيد',
            'service_type' => 'شحن بحري',
            'branch_id' => $mukallaBranchId,
            'departure_area' => 'المكلا',
            'second_destination' => '',
            'delivery_area' => 'عدن',
            'goods_name' => 'مواد غذائية',
            'goods_type' => 'غذائيات',
            'country_of_origin' => 'اليمن',
            'weight' => 200.00,
            'quantity' => 20,
            'unit' => 'صندوق',
            'service_fees' => 1500.00,
            'currency' => 'دولار',
            'payment_type' => 'تحويل بنكي',
            'delivery_duration' => '2-3 أيام',
            'amount_paid' => 1000.00,
            'remaining_amount' => 500.00,
            'order_upload_date' => date('Y-m-d'),
            'scheduled_delivery_date' => date('Y-m-d', strtotime('+3 days')),
            'recipient_name' => 'فهد محمد',
            'phone_number' => '777456789',
            'delivery_company' => 'شركة النقل البحري',
            'employee_name' => 'خالد أحمد',
            'notes' => 'بضاعة سريعة التلف',
            'amount_agreed_with_customer' => 1700.00,
            'amount_agreed_with_agent' => 1200.00,
            'other_expenses' => 200.00,
            'profit' => 300.00,
            'payment_status' => 'جزئي',
            'status' => 'في الطريق'
        ],
        [
            'order_number' => generateOrderNumber('CN'),
            'client_name' => 'سالم عبدالله',
            'service_type' => 'شحن جوي',
            'branch_id' => $chinaBranchId,
            'departure_area' => 'قوانزو',
            'second_destination' => 'دبي',
            'delivery_area' => 'صنعاء',
            'goods_name' => 'قطع غيار سيارات',
            'goods_type' => 'قطع غيار',
            'country_of_origin' => 'الصين',
            'weight' => 100.00,
            'quantity' => 5,
            'unit' => 'صندوق',
            'service_fees' => 2000.00,
            'currency' => 'دولار',
            'payment_type' => 'كاش',
            'delivery_duration' => '7-10 أيام',
            'amount_paid' => 1000.00,
            'remaining_amount' => 1000.00,
            'order_upload_date' => date('Y-m-d'),
            'scheduled_delivery_date' => date('Y-m-d', strtotime('+10 days')),
            'recipient_name' => 'محمد عبدالرحمن',
            'phone_number' => '777123789',
            'delivery_company' => 'شركة الشحن الجوي',
            'employee_name' => 'أحمد علي',
            'notes' => 'بضاعة ثقيلة',
            'amount_agreed_with_customer' => 2500.00,
            'amount_agreed_with_agent' => 1800.00,
            'other_expenses' => 300.00,
            'profit' => 400.00,
            'payment_status' => 'جزئي',
            'status' => 'قيد المعالجة'
        ]
    ];

    $results = [];

    // إضافة الطلبات
    foreach ($testOrders as $orderData) {
        $result = addOrder($orderData);
        $results[] = $result;
    }

    return $results;
}

// التحقق من وجود جدول الطلبات
if (!checkOrdersTable()) {
    echo '<div style="color: red; padding: 20px; background-color: #ffe6e6; border: 1px solid #ff8080; margin: 20px; border-radius: 5px;">';
    echo '<h2>خطأ: جدول الطلبات غير موجود</h2>';
    echo '<p>يرجى التأكد من إنشاء جدول الطلبات في قاعدة البيانات أولاً.</p>';
    echo '<p>يمكنك استخدام ملف <code>create_orders_table.sql</code> لإنشاء جدول الطلبات.</p>';
    echo '</div>';
    exit;
}

// إضافة الطلبات التجريبية
$results = createTestOrders();

// عرض النتائج
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إضافة طلبات تجريبية</title>
    <style>
        body {
            font-family: 'Tajawal', sans-serif;
            background-color: #f8f9fa;
            margin: 0;
            padding: 20px;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #4f46e5;
            text-align: center;
            margin-bottom: 30px;
        }
        .result {
            margin-bottom: 15px;
            padding: 15px;
            border-radius: 5px;
        }
        .success {
            background-color: #d1fae5;
            border: 1px solid #10b981;
            color: #047857;
        }
        .error {
            background-color: #fee2e2;
            border: 1px solid #ef4444;
            color: #b91c1c;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background-color: #4f46e5;
            color: #fff;
            text-decoration: none;
            border-radius: 5px;
            margin-top: 20px;
            text-align: center;
        }
        .btn:hover {
            background-color: #4338ca;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>إضافة طلبات تجريبية</h1>

        <?php foreach ($results as $result): ?>
            <div class="result <?php echo $result['status'] === 'success' ? 'success' : 'error'; ?>">
                <strong><?php echo $result['message']; ?></strong>
                <?php if (isset($result['orderId'])): ?>
                    <p>معرف الطلب: <?php echo $result['orderId']; ?></p>
                <?php endif; ?>
            </div>
        <?php endforeach; ?>

        <a href="orders.php" class="btn">العودة إلى صفحة الطلبات</a>
    </div>
</body>
</html>
