<?php
/**
 * إدارة صلاحيات المستخدمين
 * يحتوي هذا الملف على الدوال اللازمة لإدارة صلاحيات المستخدمين
 *
 * @version 1.0
 * @author فريق التطوير
 */

// تضمين ملف الاتصال بقاعدة البيانات
require_once __DIR__ . '/../config/db_connection.php';

/**
 * الحصول على صلاحيات المستخدم
 *
 * @param int $userId معرف المستخدم
 * @return array مصفوفة من الصلاحيات
 */
function getUserPermissions($userId) {
    global $pdo;

    try {
        // الحصول على صلاحيات المستخدم المباشرة
        $stmt = $pdo->prepare("
            SELECT p.*
            FROM permissions p
            JOIN user_permissions up ON p.id = up.permission_id
            WHERE up.user_id = ?
            ORDER BY p.category, p.id
        ");
        $stmt->execute([$userId]);
        $directPermissions = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // الحصول على صلاحيات المستخدم من خلال الدور
        $stmt = $pdo->prepare("
            SELECT p.*
            FROM permissions p
            JOIN role_permissions rp ON p.id = rp.permission_id
            JOIN roles r ON rp.role_id = r.id
            JOIN users u ON r.name = u.role
            WHERE u.id = ?
            ORDER BY p.category, p.id
        ");
        $stmt->execute([$userId]);
        $rolePermissions = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // دمج الصلاحيات وإزالة التكرار
        $allPermissions = array_merge($directPermissions, $rolePermissions);
        $uniquePermissions = [];
        $permissionIds = [];

        foreach ($allPermissions as $permission) {
            if (!in_array($permission['id'], $permissionIds)) {
                $permissionIds[] = $permission['id'];
                $uniquePermissions[] = $permission;
            }
        }

        return $uniquePermissions;
    } catch (PDOException $e) {
        error_log("خطأ في الحصول على صلاحيات المستخدم: " . $e->getMessage());
        return [];
    }
}

/**
 * التحقق من صلاحية المستخدم
 *
 * @param int $userId معرف المستخدم
 * @param string $permissionName اسم الصلاحية
 * @return bool هل يملك المستخدم الصلاحية
 */
function checkUserDirectPermission($userId, $permissionName) {
    global $pdo;

    try {
        // التحقق من صلاحيات المستخدم المباشرة
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as count
            FROM user_permissions up
            JOIN permissions p ON up.permission_id = p.id
            WHERE up.user_id = ? AND p.name = ?
        ");
        $stmt->execute([$userId, $permissionName]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);

        return $result['count'] > 0;
    } catch (PDOException $e) {
        error_log("خطأ في التحقق من صلاحية المستخدم: " . $e->getMessage());
        return false;
    }
}

/**
 * إضافة صلاحية للمستخدم
 *
 * @param int $userId معرف المستخدم
 * @param int $permissionId معرف الصلاحية
 * @return bool نجاح العملية
 */
function addUserPermission($userId, $permissionId) {
    global $pdo;

    try {
        $stmt = $pdo->prepare("
            INSERT INTO user_permissions (user_id, permission_id)
            VALUES (?, ?)
            ON DUPLICATE KEY UPDATE permission_id = VALUES(permission_id)
        ");
        $stmt->execute([$userId, $permissionId]);

        return true;
    } catch (PDOException $e) {
        error_log("خطأ في إضافة صلاحية للمستخدم: " . $e->getMessage());
        return false;
    }
}

/**
 * حذف صلاحية من المستخدم
 *
 * @param int $userId معرف المستخدم
 * @param int $permissionId معرف الصلاحية
 * @return bool نجاح العملية
 */
function removeUserPermission($userId, $permissionId) {
    global $pdo;

    try {
        $stmt = $pdo->prepare("
            DELETE FROM user_permissions
            WHERE user_id = ? AND permission_id = ?
        ");
        $stmt->execute([$userId, $permissionId]);

        return true;
    } catch (PDOException $e) {
        error_log("خطأ في حذف صلاحية من المستخدم: " . $e->getMessage());
        return false;
    }
}

/**
 * تحديث صلاحيات المستخدم
 *
 * @param int $userId معرف المستخدم
 * @param array $permissionIds مصفوفة من معرفات الصلاحيات
 * @return bool نجاح العملية
 */
function updateUserPermissions($userId, $permissionIds) {
    global $pdo;

    try {
        $pdo->beginTransaction();

        // حذف جميع صلاحيات المستخدم الحالية
        $stmt = $pdo->prepare("DELETE FROM user_permissions WHERE user_id = ?");
        $stmt->execute([$userId]);

        // إضافة الصلاحيات الجديدة
        foreach ($permissionIds as $permissionId) {
            $stmt = $pdo->prepare("
                INSERT INTO user_permissions (user_id, permission_id)
                VALUES (?, ?)
            ");
            $stmt->execute([$userId, $permissionId]);
        }

        $pdo->commit();
        return true;
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("خطأ في تحديث صلاحيات المستخدم: " . $e->getMessage());
        return false;
    }
}
