<?php
/**
 * ملف الوظائف المساعدة
 * 
 * هذا الملف يحتوي على وظائف مساعدة متنوعة تستخدم في جميع أنحاء النظام
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// دالة لتنظيف المدخلات
function sanitize($input) {
    if (is_array($input)) {
        foreach ($input as $key => $value) {
            $input[$key] = sanitize($value);
        }
        return $input;
    }
    
    // تنظيف المدخلات النصية
    $input = trim($input);
    $input = stripslashes($input);
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    
    return $input;
}

// دالة للتحقق من صحة البريد الإلكتروني
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// دالة للتحقق من صحة رقم الهاتف
function isValidPhone($phone) {
    // التحقق من أن رقم الهاتف يتكون من أرقام فقط ويبدأ بـ 05 ويتكون من 10 أرقام
    return preg_match('/^05\d{8}$/', $phone);
}

// دالة لتنسيق رقم الهاتف
function formatPhone($phone) {
    // إزالة جميع الأحرف غير الرقمية
    $phone = preg_replace('/[^0-9]/', '', $phone);
    
    // التحقق من أن رقم الهاتف يتكون من 10 أرقام
    if (strlen($phone) === 10) {
        return substr($phone, 0, 2) . ' ' . substr($phone, 2, 4) . ' ' . substr($phone, 6, 4);
    }
    
    return $phone;
}

// دالة لتنسيق التاريخ
function formatDate($date, $format = 'Y-m-d') {
    if (empty($date)) {
        return '';
    }
    
    $timestamp = strtotime($date);
    
    if ($timestamp === false) {
        return $date;
    }
    
    return date($format, $timestamp);
}

// دالة لتنسيق التاريخ والوقت
function formatDateTime($datetime, $format = 'Y-m-d H:i:s') {
    if (empty($datetime)) {
        return '';
    }
    
    $timestamp = strtotime($datetime);
    
    if ($timestamp === false) {
        return $datetime;
    }
    
    return date($format, $timestamp);
}

// دالة لتنسيق المبلغ
function formatAmount($amount, $decimals = 2, $decimalSeparator = '.', $thousandsSeparator = ',') {
    return number_format($amount, $decimals, $decimalSeparator, $thousandsSeparator);
}

// دالة لتحويل المبلغ إلى كلمات
function amountToWords($amount, $currency = 'ريال سعودي', $decimals = 'هللة') {
    $ones = [
        0 => 'صفر',
        1 => 'واحد',
        2 => 'اثنان',
        3 => 'ثلاثة',
        4 => 'أربعة',
        5 => 'خمسة',
        6 => 'ستة',
        7 => 'سبعة',
        8 => 'ثمانية',
        9 => 'تسعة',
        10 => 'عشرة',
        11 => 'أحد عشر',
        12 => 'اثنا عشر',
        13 => 'ثلاثة عشر',
        14 => 'أربعة عشر',
        15 => 'خمسة عشر',
        16 => 'ستة عشر',
        17 => 'سبعة عشر',
        18 => 'ثمانية عشر',
        19 => 'تسعة عشر'
    ];
    
    $tens = [
        2 => 'عشرون',
        3 => 'ثلاثون',
        4 => 'أربعون',
        5 => 'خمسون',
        6 => 'ستون',
        7 => 'سبعون',
        8 => 'ثمانون',
        9 => 'تسعون'
    ];
    
    $hundreds = [
        1 => 'مائة',
        2 => 'مائتان',
        3 => 'ثلاثمائة',
        4 => 'أربعمائة',
        5 => 'خمسمائة',
        6 => 'ستمائة',
        7 => 'سبعمائة',
        8 => 'ثمانمائة',
        9 => 'تسعمائة'
    ];
    
    $thousands = [
        1 => 'ألف',
        2 => 'ألفان',
        3 => 'ثلاثة آلاف',
        4 => 'أربعة آلاف',
        5 => 'خمسة آلاف',
        6 => 'ستة آلاف',
        7 => 'سبعة آلاف',
        8 => 'ثمانية آلاف',
        9 => 'تسعة آلاف'
    ];
    
    $millions = [
        1 => 'مليون',
        2 => 'مليونان',
        3 => 'ثلاثة ملايين',
        4 => 'أربعة ملايين',
        5 => 'خمسة ملايين',
        6 => 'ستة ملايين',
        7 => 'سبعة ملايين',
        8 => 'ثمانية ملايين',
        9 => 'تسعة ملايين'
    ];
    
    $billions = [
        1 => 'مليار',
        2 => 'ملياران',
        3 => 'ثلاثة مليارات',
        4 => 'أربعة مليارات',
        5 => 'خمسة مليارات',
        6 => 'ستة مليارات',
        7 => 'سبعة مليارات',
        8 => 'ثمانية مليارات',
        9 => 'تسعة مليارات'
    ];
    
    // تقسيم المبلغ إلى جزء صحيح وجزء عشري
    $parts = explode('.', (string)$amount);
    $integerPart = (int)$parts[0];
    $decimalPart = isset($parts[1]) ? (int)$parts[1] : 0;
    
    // تحويل الجزء الصحيح إلى كلمات
    $integerWords = '';
    
    if ($integerPart === 0) {
        $integerWords = $ones[0];
    } else {
        // المليارات
        $billions_number = floor($integerPart / 1000000000);
        if ($billions_number > 0) {
            if ($billions_number <= 9) {
                $integerWords .= $billions[$billions_number] . ' ';
            } else {
                $integerWords .= numberToWords($billions_number) . ' مليار ';
            }
            $integerPart %= 1000000000;
        }
        
        // الملايين
        $millions_number = floor($integerPart / 1000000);
        if ($millions_number > 0) {
            if ($millions_number <= 9) {
                $integerWords .= $millions[$millions_number] . ' ';
            } else {
                $integerWords .= numberToWords($millions_number) . ' مليون ';
            }
            $integerPart %= 1000000;
        }
        
        // الآلاف
        $thousands_number = floor($integerPart / 1000);
        if ($thousands_number > 0) {
            if ($thousands_number <= 9) {
                $integerWords .= $thousands[$thousands_number] . ' ';
            } else {
                $integerWords .= numberToWords($thousands_number) . ' ألف ';
            }
            $integerPart %= 1000;
        }
        
        // المئات
        $hundreds_number = floor($integerPart / 100);
        if ($hundreds_number > 0) {
            $integerWords .= $hundreds[$hundreds_number] . ' ';
            $integerPart %= 100;
        }
        
        // العشرات والآحاد
        if ($integerPart > 0) {
            if ($integerPart < 20) {
                $integerWords .= $ones[$integerPart] . ' ';
            } else {
                $ones_number = $integerPart % 10;
                $tens_number = floor($integerPart / 10);
                
                if ($ones_number > 0) {
                    $integerWords .= $ones[$ones_number] . ' و';
                }
                
                $integerWords .= $tens[$tens_number] . ' ';
            }
        }
    }
    
    // تحويل الجزء العشري إلى كلمات
    $decimalWords = '';
    
    if ($decimalPart > 0) {
        if ($decimalPart < 20) {
            $decimalWords = $ones[$decimalPart];
        } else {
            $ones_number = $decimalPart % 10;
            $tens_number = floor($decimalPart / 10);
            
            if ($ones_number > 0) {
                $decimalWords = $ones[$ones_number] . ' و';
            }
            
            $decimalWords .= $tens[$tens_number];
        }
    }
    
    // تكوين النتيجة النهائية
    $result = trim($integerWords) . ' ' . $currency;
    
    if (!empty($decimalWords)) {
        $result .= ' و' . $decimalWords . ' ' . $decimals;
    }
    
    return $result;
}

// دالة لتحويل الرقم إلى كلمات (مساعدة لدالة amountToWords)
function numberToWords($number) {
    // تنفيذ الدالة هنا
    return '';
}

// دالة لإنشاء رمز عشوائي
function generateRandomCode($length = 6, $type = 'numeric') {
    $characters = '';
    
    switch ($type) {
        case 'numeric':
            $characters = '0123456789';
            break;
        case 'alpha':
            $characters = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
            break;
        case 'alphanumeric':
            $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
            break;
        case 'hexadecimal':
            $characters = '0123456789abcdef';
            break;
    }
    
    $code = '';
    $charactersLength = strlen($characters);
    
    for ($i = 0; $i < $length; $i++) {
        $code .= $characters[rand(0, $charactersLength - 1)];
    }
    
    return $code;
}

// دالة لإنشاء رقم طلب فريد
function generateOrderNumber() {
    // إنشاء رقم طلب فريد بناءً على التاريخ والوقت ورقم عشوائي
    $prefix = 'ORD';
    $date = date('Ymd');
    $time = date('His');
    $random = generateRandomCode(4, 'numeric');
    
    return $prefix . $date . $time . $random;
}

// دالة لتحويل النص إلى slug
function slugify($text) {
    // تحويل النص إلى أحرف صغيرة
    $text = mb_strtolower($text, 'UTF-8');
    
    // استبدال الأحرف غير الإنجليزية بأحرف إنجليزية
    $text = str_replace(['أ', 'إ', 'آ'], 'a', $text);
    $text = str_replace(['ب'], 'b', $text);
    $text = str_replace(['ت', 'ة'], 't', $text);
    $text = str_replace(['ث'], 'th', $text);
    $text = str_replace(['ج'], 'j', $text);
    $text = str_replace(['ح'], 'h', $text);
    $text = str_replace(['خ'], 'kh', $text);
    $text = str_replace(['د'], 'd', $text);
    $text = str_replace(['ذ'], 'th', $text);
    $text = str_replace(['ر'], 'r', $text);
    $text = str_replace(['ز'], 'z', $text);
    $text = str_replace(['س'], 's', $text);
    $text = str_replace(['ش'], 'sh', $text);
    $text = str_replace(['ص'], 's', $text);
    $text = str_replace(['ض'], 'd', $text);
    $text = str_replace(['ط'], 't', $text);
    $text = str_replace(['ظ'], 'z', $text);
    $text = str_replace(['ع'], 'a', $text);
    $text = str_replace(['غ'], 'gh', $text);
    $text = str_replace(['ف'], 'f', $text);
    $text = str_replace(['ق'], 'q', $text);
    $text = str_replace(['ك'], 'k', $text);
    $text = str_replace(['ل'], 'l', $text);
    $text = str_replace(['م'], 'm', $text);
    $text = str_replace(['ن'], 'n', $text);
    $text = str_replace(['ه'], 'h', $text);
    $text = str_replace(['و'], 'w', $text);
    $text = str_replace(['ي', 'ى', 'ئ'], 'y', $text);
    
    // استبدال المسافات والأحرف الخاصة بشرطة
    $text = preg_replace('/[^a-z0-9]/', '-', $text);
    
    // إزالة الشرطات المتكررة
    $text = preg_replace('/-+/', '-', $text);
    
    // إزالة الشرطات من بداية ونهاية النص
    $text = trim($text, '-');
    
    return $text;
}

// دالة لتحويل حجم الملف إلى صيغة مقروءة
function formatFileSize($size) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $i = 0;
    
    while ($size >= 1024 && $i < count($units) - 1) {
        $size /= 1024;
        $i++;
    }
    
    return round($size, 2) . ' ' . $units[$i];
}

// دالة للتحقق من نوع الملف
function isAllowedFileType($file, $allowedTypes) {
    $fileType = $file['type'];
    
    return in_array($fileType, $allowedTypes);
}

// دالة لرفع ملف
function uploadFile($file, $destination, $allowedTypes = [], $maxSize = 5242880) {
    // التحقق من وجود خطأ في رفع الملف
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء رفع الملف'
        ];
    }
    
    // التحقق من حجم الملف
    if ($file['size'] > $maxSize) {
        return [
            'success' => false,
            'message' => 'حجم الملف كبير جدًا. الحد الأقصى هو ' . formatFileSize($maxSize)
        ];
    }
    
    // التحقق من نوع الملف
    if (!empty($allowedTypes) && !isAllowedFileType($file, $allowedTypes)) {
        return [
            'success' => false,
            'message' => 'نوع الملف غير مسموح به'
        ];
    }
    
    // إنشاء اسم فريد للملف
    $filename = uniqid() . '_' . basename($file['name']);
    $filepath = $destination . '/' . $filename;
    
    // التأكد من وجود المجلد الوجهة
    if (!is_dir($destination)) {
        mkdir($destination, 0755, true);
    }
    
    // رفع الملف
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return [
            'success' => true,
            'message' => 'تم رفع الملف بنجاح',
            'filename' => $filename,
            'filepath' => $filepath
        ];
    } else {
        return [
            'success' => false,
            'message' => 'فشل رفع الملف'
        ];
    }
}
