<?php
/**
 * ملف المصادقة
 * 
 * هذا الملف يحتوي على وظائف المصادقة وإدارة المستخدمين
 * 
 * @version 1.0
 * @author فريق التطوير
 */

// دالة لتسجيل الدخول
function login($username, $password) {
    global $conn;
    
    try {
        // البحث عن المستخدم في قاعدة البيانات
        $stmt = $conn->prepare("SELECT * FROM users WHERE username = ?");
        $stmt->execute([$username]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // التحقق من وجود المستخدم وصحة كلمة المرور
        if ($user && password_verify($password, $user['password'])) {
            // التحقق من حالة المستخدم
            if ($user['status'] !== 'active') {
                return [
                    'success' => false,
                    'message' => 'الحساب غير نشط. يرجى التواصل مع الإدارة.'
                ];
            }
            
            // تخزين بيانات المستخدم في الجلسة
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['full_name'] = $user['full_name'];
            $_SESSION['email'] = $user['email'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['branch_id'] = $user['branch_id'];
            $_SESSION['logged_in'] = true;
            $_SESSION['expires_at'] = time() + (60 * 60 * 2); // ساعتين
            
            // تسجيل عملية تسجيل الدخول
            logActivity($user['id'], 'تسجيل الدخول', 'تم تسجيل الدخول بنجاح');
            
            return [
                'success' => true,
                'message' => 'تم تسجيل الدخول بنجاح',
                'user' => $user
            ];
        } else {
            // تسجيل محاولة تسجيل دخول فاشلة
            logError("محاولة تسجيل دخول فاشلة: $username", 'SECURITY');
            
            return [
                'success' => false,
                'message' => 'اسم المستخدم أو كلمة المرور غير صحيحة'
            ];
        }
    } catch (PDOException $e) {
        logError("خطأ في تسجيل الدخول: " . $e->getMessage());
        
        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء تسجيل الدخول. يرجى المحاولة مرة أخرى.'
        ];
    }
}

// دالة لتسجيل الخروج
function logout() {
    // تسجيل عملية تسجيل الخروج
    if (isset($_SESSION['user_id'])) {
        logActivity($_SESSION['user_id'], 'تسجيل الخروج', 'تم تسجيل الخروج بنجاح');
    }
    
    // حذف بيانات الجلسة
    session_unset();
    session_destroy();
    
    return [
        'success' => true,
        'message' => 'تم تسجيل الخروج بنجاح'
    ];
}

// دالة لتسجيل مستخدم جديد
function register($userData) {
    global $conn;
    
    try {
        // التحقق من وجود المستخدم
        $stmt = $conn->prepare("SELECT * FROM users WHERE username = ? OR email = ?");
        $stmt->execute([$userData['username'], $userData['email']]);
        $existingUser = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($existingUser) {
            if ($existingUser['username'] === $userData['username']) {
                return [
                    'success' => false,
                    'message' => 'اسم المستخدم موجود بالفعل'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'البريد الإلكتروني موجود بالفعل'
                ];
            }
        }
        
        // تشفير كلمة المرور
        $hashedPassword = password_hash($userData['password'], PASSWORD_DEFAULT);
        
        // إدراج المستخدم الجديد
        $stmt = $conn->prepare("
            INSERT INTO users (username, password, full_name, email, phone, role, branch_id, status, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $userData['username'],
            $hashedPassword,
            $userData['full_name'],
            $userData['email'],
            $userData['phone'] ?? '',
            $userData['role'] ?? 'user',
            $userData['branch_id'] ?? 1,
            $userData['status'] ?? 'active'
        ]);
        
        $userId = $conn->lastInsertId();
        
        // تسجيل عملية التسجيل
        logActivity($userId, 'تسجيل مستخدم جديد', 'تم تسجيل مستخدم جديد بنجاح');
        
        return [
            'success' => true,
            'message' => 'تم تسجيل المستخدم بنجاح',
            'user_id' => $userId
        ];
    } catch (PDOException $e) {
        logError("خطأ في تسجيل مستخدم جديد: " . $e->getMessage());
        
        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء تسجيل المستخدم. يرجى المحاولة مرة أخرى.'
        ];
    }
}

// دالة لتحديث بيانات المستخدم
function updateUser($userId, $userData) {
    global $conn;
    
    try {
        // التحقق من وجود المستخدم
        $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            return [
                'success' => false,
                'message' => 'المستخدم غير موجود'
            ];
        }
        
        // التحقق من وجود اسم مستخدم أو بريد إلكتروني مكرر
        if (isset($userData['username']) || isset($userData['email'])) {
            $stmt = $conn->prepare("
                SELECT * FROM users 
                WHERE (username = ? OR email = ?) 
                AND id != ?
            ");
            
            $stmt->execute([
                $userData['username'] ?? $user['username'],
                $userData['email'] ?? $user['email'],
                $userId
            ]);
            
            $existingUser = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingUser) {
                if (isset($userData['username']) && $existingUser['username'] === $userData['username']) {
                    return [
                        'success' => false,
                        'message' => 'اسم المستخدم موجود بالفعل'
                    ];
                } else if (isset($userData['email']) && $existingUser['email'] === $userData['email']) {
                    return [
                        'success' => false,
                        'message' => 'البريد الإلكتروني موجود بالفعل'
                    ];
                }
            }
        }
        
        // بناء استعلام التحديث
        $updateFields = [];
        $params = [];
        
        foreach ($userData as $field => $value) {
            // تجاهل حقل كلمة المرور (يتم التعامل معه بشكل منفصل)
            if ($field === 'password') {
                continue;
            }
            
            $updateFields[] = "$field = ?";
            $params[] = $value;
        }
        
        // إضافة حقل تاريخ التحديث
        $updateFields[] = "updated_at = NOW()";
        
        // إضافة معرف المستخدم إلى المعلمات
        $params[] = $userId;
        
        // تنفيذ استعلام التحديث
        $stmt = $conn->prepare("
            UPDATE users 
            SET " . implode(', ', $updateFields) . " 
            WHERE id = ?
        ");
        
        $stmt->execute($params);
        
        // تحديث كلمة المرور إذا تم توفيرها
        if (isset($userData['password']) && !empty($userData['password'])) {
            $hashedPassword = password_hash($userData['password'], PASSWORD_DEFAULT);
            
            $stmt = $conn->prepare("
                UPDATE users 
                SET password = ?, updated_at = NOW() 
                WHERE id = ?
            ");
            
            $stmt->execute([$hashedPassword, $userId]);
        }
        
        // تسجيل عملية التحديث
        logActivity(isset($_SESSION['user_id']) ? $_SESSION['user_id'] : $userId, 'تحديث بيانات المستخدم', 'تم تحديث بيانات المستخدم بنجاح');
        
        return [
            'success' => true,
            'message' => 'تم تحديث بيانات المستخدم بنجاح'
        ];
    } catch (PDOException $e) {
        logError("خطأ في تحديث بيانات المستخدم: " . $e->getMessage());
        
        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء تحديث بيانات المستخدم. يرجى المحاولة مرة أخرى.'
        ];
    }
}

// دالة لحذف مستخدم
function deleteUser($userId) {
    global $conn;
    
    try {
        // التحقق من وجود المستخدم
        $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            return [
                'success' => false,
                'message' => 'المستخدم غير موجود'
            ];
        }
        
        // حذف المستخدم
        $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        
        // تسجيل عملية الحذف
        logActivity($_SESSION['user_id'], 'حذف مستخدم', 'تم حذف المستخدم بنجاح');
        
        return [
            'success' => true,
            'message' => 'تم حذف المستخدم بنجاح'
        ];
    } catch (PDOException $e) {
        logError("خطأ في حذف المستخدم: " . $e->getMessage());
        
        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء حذف المستخدم. يرجى المحاولة مرة أخرى.'
        ];
    }
}

// دالة للحصول على بيانات المستخدم
function getUser($userId) {
    global $conn;
    
    try {
        $stmt = $conn->prepare("
            SELECT u.*, b.name as branch_name
            FROM users u
            LEFT JOIN branches b ON u.branch_id = b.id
            WHERE u.id = ?
        ");
        
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            return [
                'success' => false,
                'message' => 'المستخدم غير موجود'
            ];
        }
        
        // إزالة كلمة المرور من البيانات المرجعة
        unset($user['password']);
        
        return [
            'success' => true,
            'user' => $user
        ];
    } catch (PDOException $e) {
        logError("خطأ في الحصول على بيانات المستخدم: " . $e->getMessage());
        
        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء الحصول على بيانات المستخدم. يرجى المحاولة مرة أخرى.'
        ];
    }
}

// دالة للحصول على قائمة المستخدمين
function getUsers($filters = []) {
    global $conn;
    
    try {
        $query = "
            SELECT u.*, b.name as branch_name
            FROM users u
            LEFT JOIN branches b ON u.branch_id = b.id
        ";
        
        $whereConditions = [];
        $params = [];
        
        // إضافة شروط التصفية
        if (isset($filters['role']) && !empty($filters['role'])) {
            $whereConditions[] = "u.role = ?";
            $params[] = $filters['role'];
        }
        
        if (isset($filters['branch_id']) && !empty($filters['branch_id'])) {
            $whereConditions[] = "u.branch_id = ?";
            $params[] = $filters['branch_id'];
        }
        
        if (isset($filters['status']) && !empty($filters['status'])) {
            $whereConditions[] = "u.status = ?";
            $params[] = $filters['status'];
        }
        
        if (isset($filters['search']) && !empty($filters['search'])) {
            $whereConditions[] = "(u.username LIKE ? OR u.full_name LIKE ? OR u.email LIKE ?)";
            $searchTerm = "%{$filters['search']}%";
            $params[] = $searchTerm;
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }
        
        // إضافة شروط التصفية إلى الاستعلام
        if (!empty($whereConditions)) {
            $query .= " WHERE " . implode(" AND ", $whereConditions);
        }
        
        // إضافة الترتيب
        $query .= " ORDER BY u.id DESC";
        
        // تنفيذ الاستعلام
        $stmt = $conn->prepare($query);
        $stmt->execute($params);
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // إزالة كلمات المرور من البيانات المرجعة
        foreach ($users as &$user) {
            unset($user['password']);
        }
        
        return [
            'success' => true,
            'users' => $users,
            'count' => count($users)
        ];
    } catch (PDOException $e) {
        logError("خطأ في الحصول على قائمة المستخدمين: " . $e->getMessage());
        
        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء الحصول على قائمة المستخدمين. يرجى المحاولة مرة أخرى.'
        ];
    }
}

// دالة لتغيير كلمة المرور
function changePassword($userId, $currentPassword, $newPassword) {
    global $conn;
    
    try {
        // التحقق من وجود المستخدم
        $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            return [
                'success' => false,
                'message' => 'المستخدم غير موجود'
            ];
        }
        
        // التحقق من صحة كلمة المرور الحالية
        if (!password_verify($currentPassword, $user['password'])) {
            return [
                'success' => false,
                'message' => 'كلمة المرور الحالية غير صحيحة'
            ];
        }
        
        // تشفير كلمة المرور الجديدة
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        
        // تحديث كلمة المرور
        $stmt = $conn->prepare("
            UPDATE users 
            SET password = ?, updated_at = NOW() 
            WHERE id = ?
        ");
        
        $stmt->execute([$hashedPassword, $userId]);
        
        // تسجيل عملية تغيير كلمة المرور
        logActivity($userId, 'تغيير كلمة المرور', 'تم تغيير كلمة المرور بنجاح');
        
        return [
            'success' => true,
            'message' => 'تم تغيير كلمة المرور بنجاح'
        ];
    } catch (PDOException $e) {
        logError("خطأ في تغيير كلمة المرور: " . $e->getMessage());
        
        return [
            'success' => false,
            'message' => 'حدث خطأ أثناء تغيير كلمة المرور. يرجى المحاولة مرة أخرى.'
        ];
    }
}
