<?php
/**
 * ملف إصلاح مشكلة تسجيل دخول مدير النظام
 * يقوم هذا الملف بتحديث بيانات مدير النظام وربطه بالمركز الرئيسي
 * 
 * @version 1.0
 * @author System Developer
 */

// عرض جميع الأخطاء للمساعدة في التشخيص
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/db_connection.php';

// تعريف متغيرات CSS للتنسيق
$successColor = "#4CAF50";
$errorColor = "#F44336";
$warningColor = "#FF9800";
$infoColor = "#2196F3";
$primaryColor = "#4c0c70";
$secondaryColor = "#6d28d9";

// تعريف متغيرات لبيانات المدير
$adminUsername = "admin";
$adminEmail = "admin@al-ahmadigroup.com";
$adminPassword = "admin123"; // كلمة المرور الجديدة
$adminFullName = "مدير النظام";
$adminPhone = "05318281";
$adminRole = "admin";
$adminStatus = "active";
$adminIsActive = 1;

// تعريف متغير لتخزين الرسائل
$messages = [];
$errors = [];
$debugInfo = [];

// دالة لإضافة رسالة
function addMessage($type, $message) {
    global $messages;
    $messages[] = [
        'type' => $type,
        'message' => $message
    ];
}

// دالة لإضافة معلومات تصحيح
function addDebugInfo($info) {
    global $debugInfo;
    $debugInfo[] = $info;
}

// دالة للتحقق من وجود المركز الرئيسي
function checkHeadquarters($conn) {
    try {
        $stmt = $conn->prepare("SELECT id, name, code FROM branches WHERE name = 'المركز الرئيسي'");
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        addMessage('error', 'خطأ في التحقق من وجود المركز الرئيسي: ' . $e->getMessage());
        return false;
    }
}

// دالة لإنشاء المركز الرئيسي إذا لم يكن موجودًا
function createHeadquarters($conn) {
    try {
        $stmt = $conn->prepare("INSERT INTO branches (name, code, address, is_active) VALUES ('المركز الرئيسي', 'HQ', 'اليمن، حضرموت، المكلا', 1)");
        $stmt->execute();
        addMessage('success', 'تم إنشاء المركز الرئيسي بنجاح.');
        return $conn->lastInsertId();
    } catch (PDOException $e) {
        addMessage('error', 'خطأ في إنشاء المركز الرئيسي: ' . $e->getMessage());
        return false;
    }
}

// دالة للتحقق من وجود مدير النظام
function checkAdmin($conn, $username) {
    try {
        $stmt = $conn->prepare("SELECT * FROM users WHERE username = :username");
        $stmt->bindParam(':username', $username);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        addMessage('error', 'خطأ في التحقق من وجود مدير النظام: ' . $e->getMessage());
        return false;
    }
}

// دالة لتحديث بيانات مدير النظام
function updateAdmin($conn, $adminId, $branchId, $adminData) {
    try {
        // تشفير كلمة المرور
        $hashedPassword = password_hash($adminData['password'], PASSWORD_DEFAULT);
        
        // تحديث بيانات المدير
        $stmt = $conn->prepare("
            UPDATE users 
            SET full_name = :full_name, 
                email = :email, 
                phone = :phone, 
                password = :password, 
                branch_id = :branch_id,
                role = :role,
                status = :status,
                is_active = :is_active
            WHERE id = :id
        ");
        
        $stmt->bindParam(':full_name', $adminData['full_name']);
        $stmt->bindParam(':email', $adminData['email']);
        $stmt->bindParam(':phone', $adminData['phone']);
        $stmt->bindParam(':password', $hashedPassword);
        $stmt->bindParam(':branch_id', $branchId);
        $stmt->bindParam(':role', $adminData['role']);
        $stmt->bindParam(':status', $adminData['status']);
        $stmt->bindParam(':is_active', $adminData['is_active']);
        $stmt->bindParam(':id', $adminId);
        
        $stmt->execute();
        
        addMessage('success', 'تم تحديث بيانات مدير النظام بنجاح.');
        return true;
    } catch (PDOException $e) {
        addMessage('error', 'خطأ في تحديث بيانات مدير النظام: ' . $e->getMessage());
        return false;
    }
}

// دالة لإنشاء مدير النظام إذا لم يكن موجودًا
function createAdmin($conn, $branchId, $adminData) {
    try {
        // تشفير كلمة المرور
        $hashedPassword = password_hash($adminData['password'], PASSWORD_DEFAULT);
        
        // إنشاء المدير
        $stmt = $conn->prepare("
            INSERT INTO users (
                username, 
                password, 
                full_name, 
                email, 
                phone, 
                role, 
                branch_id, 
                status, 
                is_active, 
                registration_date
            ) VALUES (
                :username, 
                :password, 
                :full_name, 
                :email, 
                :phone, 
                :role, 
                :branch_id, 
                :status, 
                :is_active, 
                NOW()
            )
        ");
        
        $stmt->bindParam(':username', $adminData['username']);
        $stmt->bindParam(':password', $hashedPassword);
        $stmt->bindParam(':full_name', $adminData['full_name']);
        $stmt->bindParam(':email', $adminData['email']);
        $stmt->bindParam(':phone', $adminData['phone']);
        $stmt->bindParam(':role', $adminData['role']);
        $stmt->bindParam(':branch_id', $branchId);
        $stmt->bindParam(':status', $adminData['status']);
        $stmt->bindParam(':is_active', $adminData['is_active']);
        
        $stmt->execute();
        
        addMessage('success', 'تم إنشاء مدير النظام بنجاح.');
        return $conn->lastInsertId();
    } catch (PDOException $e) {
        addMessage('error', 'خطأ في إنشاء مدير النظام: ' . $e->getMessage());
        return false;
    }
}

// دالة للتحقق من بنية جدول المستخدمين
function checkUsersTable($conn) {
    try {
        // التحقق من وجود جدول المستخدمين
        $stmt = $conn->query("SHOW TABLES LIKE 'users'");
        $tableExists = $stmt->rowCount() > 0;
        
        if ($tableExists) {
            // الحصول على هيكل الجدول
            $stmt = $conn->query("DESCRIBE users");
            $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // التحقق من وجود الأعمدة المطلوبة
            $requiredColumns = ['id', 'username', 'password', 'full_name', 'email', 'phone', 'role', 'branch_id', 'status', 'is_active'];
            $missingColumns = [];
            
            $columnNames = array_column($columns, 'Field');
            
            foreach ($requiredColumns as $column) {
                if (!in_array($column, $columnNames)) {
                    $missingColumns[] = $column;
                }
            }
            
            return [
                'exists' => true,
                'columns' => $columns,
                'missing_columns' => $missingColumns
            ];
        } else {
            return [
                'exists' => false
            ];
        }
    } catch (PDOException $e) {
        addMessage('error', 'خطأ في التحقق من بنية جدول المستخدمين: ' . $e->getMessage());
        return [
            'exists' => false,
            'error' => $e->getMessage()
        ];
    }
}

// بدء العملية
try {
    // التحقق من بنية جدول المستخدمين
    $usersTable = checkUsersTable($conn);
    
    if (!$usersTable['exists']) {
        throw new Exception('جدول المستخدمين غير موجود. يرجى إنشاء قاعدة البيانات أولاً.');
    }
    
    if (!empty($usersTable['missing_columns'])) {
        addMessage('warning', 'بعض الأعمدة المطلوبة غير موجودة في جدول المستخدمين: ' . implode(', ', $usersTable['missing_columns']));
    }
    
    // التحقق من وجود المركز الرئيسي
    $headquarters = checkHeadquarters($conn);
    
    // إذا لم يكن المركز الرئيسي موجودًا، قم بإنشائه
    if (!$headquarters) {
        $headquartersId = createHeadquarters($conn);
        if (!$headquartersId) {
            throw new Exception('فشل في إنشاء المركز الرئيسي.');
        }
    } else {
        $headquartersId = $headquarters['id'];
        addMessage('info', 'المركز الرئيسي موجود بالفعل (المعرف: ' . $headquartersId . ').');
    }
    
    // تجميع بيانات المدير
    $adminData = [
        'username' => $adminUsername,
        'password' => $adminPassword,
        'full_name' => $adminFullName,
        'email' => $adminEmail,
        'phone' => $adminPhone,
        'role' => $adminRole,
        'status' => $adminStatus,
        'is_active' => $adminIsActive
    ];
    
    // التحقق من وجود مدير النظام
    $admin = checkAdmin($conn, $adminUsername);
    
    // إضافة معلومات تصحيح
    if ($admin) {
        addDebugInfo('تم العثور على مدير النظام (المعرف: ' . $admin['id'] . ')');
        addDebugInfo('البريد الإلكتروني الحالي: ' . $admin['email']);
        addDebugInfo('الفرع الحالي: ' . ($admin['branch_id'] ?? 'غير محدد'));
        addDebugInfo('الحالة الحالية: ' . ($admin['status'] ?? 'غير محدد'));
        addDebugInfo('نشط: ' . ($admin['is_active'] ?? 'غير محدد'));
    } else {
        addDebugInfo('لم يتم العثور على مدير النظام');
    }
    
    // إذا كان مدير النظام موجودًا، قم بتحديث بياناته
    if ($admin) {
        $result = updateAdmin($conn, $admin['id'], $headquartersId, $adminData);
        if (!$result) {
            throw new Exception('فشل في تحديث بيانات مدير النظام.');
        }
    } else {
        // إذا لم يكن مدير النظام موجودًا، قم بإنشائه
        $adminId = createAdmin($conn, $headquartersId, $adminData);
        if (!$adminId) {
            throw new Exception('فشل في إنشاء مدير النظام.');
        }
    }
    
    // إضافة رسالة نجاح نهائية
    addMessage('success', 'تم تحديث بيانات مدير النظام وربطه بالمركز الرئيسي بنجاح.');
    
} catch (Exception $e) {
    addMessage('error', 'حدث خطأ: ' . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>إصلاح مشكلة تسجيل دخول مدير النظام</title>
    <link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Tajawal', sans-serif;
            background-color: #f5f5f5;
            margin: 0;
            padding: 20px;
            color: #333;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
        }
        h1 {
            color: <?php echo $primaryColor; ?>;
            text-align: center;
            margin-bottom: 30px;
            font-size: 28px;
        }
        .message {
            padding: 15px;
            margin-bottom: 15px;
            border-radius: 5px;
            font-size: 16px;
        }
        .success {
            background-color: rgba(76, 175, 80, 0.1);
            border: 1px solid <?php echo $successColor; ?>;
            color: <?php echo $successColor; ?>;
        }
        .error {
            background-color: rgba(244, 67, 54, 0.1);
            border: 1px solid <?php echo $errorColor; ?>;
            color: <?php echo $errorColor; ?>;
        }
        .warning {
            background-color: rgba(255, 152, 0, 0.1);
            border: 1px solid <?php echo $warningColor; ?>;
            color: <?php echo $warningColor; ?>;
        }
        .info {
            background-color: rgba(33, 150, 243, 0.1);
            border: 1px solid <?php echo $infoColor; ?>;
            color: <?php echo $infoColor; ?>;
        }
        .admin-info {
            background-color: rgba(76, 12, 112, 0.05);
            border: 1px solid #ddd;
            padding: 20px;
            border-radius: 5px;
            margin-top: 20px;
        }
        .admin-info h2 {
            color: <?php echo $primaryColor; ?>;
            margin-top: 0;
            font-size: 20px;
        }
        .admin-info p {
            margin: 10px 0;
            font-size: 16px;
        }
        .admin-info strong {
            color: <?php echo $secondaryColor; ?>;
        }
        .back-link {
            display: block;
            text-align: center;
            margin-top: 30px;
            color: <?php echo $primaryColor; ?>;
            text-decoration: none;
            font-weight: bold;
            font-size: 16px;
        }
        .back-link:hover {
            text-decoration: underline;
        }
        .debug-info {
            background-color: #f8f9fa;
            border: 1px solid #ddd;
            padding: 15px;
            border-radius: 5px;
            margin-top: 20px;
            font-family: monospace;
            font-size: 14px;
            white-space: pre-wrap;
        }
        .debug-info h3 {
            margin-top: 0;
            color: #333;
            font-size: 16px;
        }
        .debug-info p {
            margin: 5px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>إصلاح مشكلة تسجيل دخول مدير النظام</h1>
        
        <?php foreach ($messages as $message): ?>
            <div class="message <?php echo $message['type']; ?>">
                <?php echo $message['message']; ?>
            </div>
        <?php endforeach; ?>
        
        <div class="admin-info">
            <h2>معلومات مدير النظام</h2>
            <p><strong>اسم المستخدم:</strong> <?php echo $adminUsername; ?></p>
            <p><strong>الاسم الكامل:</strong> <?php echo $adminFullName; ?></p>
            <p><strong>البريد الإلكتروني:</strong> <?php echo $adminEmail; ?></p>
            <p><strong>رقم الهاتف:</strong> <?php echo $adminPhone; ?></p>
            <p><strong>كلمة المرور:</strong> <?php echo $adminPassword; ?> (كلمة المرور الجديدة)</p>
            <p><strong>الفرع:</strong> المركز الرئيسي</p>
            <p><strong>الدور:</strong> <?php echo $adminRole; ?></p>
            <p><strong>الحالة:</strong> <?php echo $adminStatus; ?></p>
        </div>
        
        <?php if (!empty($debugInfo)): ?>
        <div class="debug-info">
            <h3>معلومات التصحيح</h3>
            <?php foreach ($debugInfo as $info): ?>
                <p><?php echo $info; ?></p>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
        
        <a href="auth/login.php" class="back-link">العودة إلى صفحة تسجيل الدخول</a>
    </div>
</body>
</html>
