<?php
/**
 * ملف إعداد قاعدة البيانات
 * يستخدم هذا الملف لإنشاء قاعدة البيانات وجداولها
 */

// تضمين ملف التكوين
require_once __DIR__ . '/../config/config.php';

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// دالة لتنفيذ ملف SQL
function executeSqlFile($pdo, $filename) {
    try {
        // التحقق من وجود الملف
        if (!file_exists($filename)) {
            return [
                'status' => 'error',
                'message' => "الملف $filename غير موجود"
            ];
        }

        // قراءة محتوى الملف
        $sql = file_get_contents($filename);
        if ($sql === false) {
            return [
                'status' => 'error',
                'message' => "فشل في قراءة محتوى الملف $filename"
            ];
        }

        // تقسيم الاستعلامات
        $queries = explode(';', $sql);

        // تنفيذ كل استعلام
        foreach ($queries as $query) {
            $query = trim($query);
            if (!empty($query)) {
                // استبدال DELIMITER لتجنب الأخطاء
                if (strpos($query, 'DELIMITER') === 0) {
                    continue;
                }

                // تجاهل الأسطر التي تبدأ بـ -- (تعليقات SQL)
                if (strpos($query, '--') === 0) {
                    continue;
                }

                // تجاهل استعلامات USE و CREATE DATABASE في SQLite
                if (!$use_mysql && (strpos($query, 'USE ') === 0 || strpos($query, 'CREATE DATABASE') === 0)) {
                    continue;
                }

                // تعديل استعلامات إنشاء الجداول لـ SQLite
                if (!$use_mysql) {
                    // إزالة ENGINE و CHARSET
                    $query = preg_replace('/ENGINE=.*$/m', '', $query);

                    // تغيير أنواع البيانات
                    $query = str_replace('int(11)', 'INTEGER', $query);
                    $query = str_replace('tinyint(1)', 'INTEGER', $query);

                    // تغيير AUTO_INCREMENT
                    $query = str_replace('AUTO_INCREMENT', 'AUTOINCREMENT', $query);

                    // تغيير ENUM
                    $query = preg_replace('/enum\(([^)]+)\)/', 'TEXT CHECK(\\0 IN (\\1))', $query);
                }

                $pdo->exec($query);
            }
        }

        return [
            'status' => 'success',
            'message' => "تم تنفيذ الملف $filename بنجاح"
        ];
    } catch (PDOException $e) {
        return [
            'status' => 'error',
            'message' => "خطأ في تنفيذ الملف $filename: " . $e->getMessage()
        ];
    }
}

// دالة لإنشاء قاعدة البيانات
function createDatabase($pdo, $dbname, $recreate = false) {
    try {
        global $use_mysql;

        if ($use_mysql) {
            // إنشاء قاعدة البيانات في MySQL
            $pdo->exec("CREATE DATABASE IF NOT EXISTS `$dbname` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            echo "تم إنشاء قاعدة البيانات $dbname بنجاح أو كانت موجودة بالفعل.<br>";

            // استخدام قاعدة البيانات
            $pdo->exec("USE `$dbname`");
        } else {
            // SQLite لا يحتاج إلى إنشاء قاعدة بيانات منفصلة
            echo "تم استخدام قاعدة بيانات SQLite.<br>";
        }

        return true;
    } catch (PDOException $e) {
        echo "خطأ في إنشاء قاعدة البيانات: " . $e->getMessage() . "<br>";
        return false;
    }
}

// دالة لإنشاء جداول قاعدة البيانات
function createTables($pdo, $recreate = false) {
    try {
        global $use_mysql;

        // إذا كان يجب إعادة إنشاء الجداول، قم بحذفها أولاً
        if ($recreate && $use_mysql) {
            // حذف الجداول بترتيب عكسي لتجنب مشاكل المفاتيح الأجنبية
            $pdo->exec("DROP TABLE IF EXISTS `archived_orders`");
            $pdo->exec("DROP TABLE IF EXISTS `orders`");
            $pdo->exec("DROP TABLE IF EXISTS `order_sequences`");
            $pdo->exec("DROP TABLE IF EXISTS `users`");
            $pdo->exec("DROP TABLE IF EXISTS `countries`");
            $pdo->exec("DROP TABLE IF EXISTS `branches`");
            echo "تم حذف الجداول الموجودة بنجاح.<br>";
        }

        // تنفيذ ملف SQL الموحد النهائي
        $result = executeSqlFile($pdo, __DIR__ . '/final_unified_database.sql');

        if ($result['status'] === 'success') {
            echo "تم إنشاء جداول قاعدة البيانات بنجاح.<br>";
            return true;
        } else {
            echo "خطأ في إنشاء جداول قاعدة البيانات: " . $result['message'] . "<br>";
            return false;
        }
    } catch (PDOException $e) {
        echo "خطأ في إنشاء جداول قاعدة البيانات: " . $e->getMessage() . "<br>";
        return false;
    }
}

// التحقق من وجود معلمة إعادة الإنشاء
$recreate = isset($_GET['recreate']) && $_GET['recreate'] === 'true';

// عرض نموذج الإعداد
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    ?>
    <!DOCTYPE html>
    <html lang="ar" dir="rtl">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>إعداد قاعدة البيانات</title>
        <link rel="stylesheet" href="../assets/css/style.css">
    </head>
    <body>
        <div class="container">
            <h1>إعداد قاعدة البيانات</h1>

            <form method="post" action="">
                <div class="form-group">
                    <label for="recreate">إعادة إنشاء الجداول:</label>
                    <input type="checkbox" id="recreate" name="recreate" value="true" <?php echo $recreate ? 'checked' : ''; ?>>
                    <small>تحذير: سيؤدي هذا إلى حذف جميع البيانات الموجودة!</small>
                </div>

                <div class="form-group">
                    <button type="submit" class="btn btn-primary">إعداد قاعدة البيانات</button>
                </div>
            </form>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// معالجة النموذج
$recreate = isset($_POST['recreate']) && $_POST['recreate'] === 'true';

// إنشاء قاعدة البيانات
if (createDatabase($pdo, $db_name, $recreate)) {
    // إنشاء الجداول
    if (createTables($pdo, $recreate)) {
        echo "<p>تم إعداد قاعدة البيانات بنجاح.</p>";
        echo "<p><a href='../index.php'>العودة إلى الصفحة الرئيسية</a></p>";
    }
}
?>
