<?php
/**
 * ملف الدوال المساعدة
 * يحتوي على جميع الدوال المساعدة المستخدمة في النظام
 */

// دالة لتحويل التاريخ من التنسيق العربي إلى تنسيق قاعدة البيانات
function convertDateToMysql($date) {
    if (empty($date)) return null;

    // تحويل التاريخ من التنسيق العربي (DD/MM/YYYY) إلى تنسيق MySQL (YYYY-MM-DD)
    $parts = explode('/', $date);
    if (count($parts) === 3) {
        return $parts[2] . '-' . $parts[1] . '-' . $parts[0];
    }

    return $date;
}

// دالة لتحويل التاريخ من تنسيق قاعدة البيانات إلى التنسيق العربي
function convertDateToArabic($date) {
    if (empty($date)) return '';

    // تحويل التاريخ من تنسيق MySQL (YYYY-MM-DD) إلى التنسيق العربي (DD/MM/YYYY)
    $parts = explode('-', $date);
    if (count($parts) === 3) {
        return $parts[2] . '/' . $parts[1] . '/' . $parts[0];
    }

    return $date;
}

// دالة للحصول على رقم طلب جديد
function getNewOrderNumber($pdo, $branchCode) {
    try {
        // البحث عن آخر رقم تسلسلي للفرع
        $stmt = $pdo->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
        $stmt->execute([$branchCode]);
        $result = $stmt->fetch();

        // الحصول على السنة الحالية (آخر رقمين)
        $year = date('y');

        if ($result) {
            $lastNumber = $result['last_sequence_number'];
            $newNumber = $lastNumber + 1;

            // تحديث الرقم التسلسلي في قاعدة البيانات
            $updateStmt = $pdo->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
            $updateStmt->execute([$newNumber, $branchCode]);

            // تنسيق الرقم التسلسلي
            $sequenceNumber = str_pad($newNumber, 2, '0', STR_PAD_LEFT);

            // إنشاء رقم الطلب الكامل
            $newOrderNumber = $branchCode . $year . '-' . $sequenceNumber;

            return $newOrderNumber;
        } else {
            // إذا لم يتم العثور على رمز الفرع، قم بإنشاء سجل جديد
            $insertStmt = $pdo->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, 1)");
            $insertStmt->execute([$branchCode]);

            return $branchCode . $year . '-01';
        }
    } catch (PDOException $e) {
        // في حالة حدوث خطأ
        error_log('خطأ في إنشاء رقم طلب جديد: ' . $e->getMessage());
        return false;
    }
}

// دالة للتحقق من تسجيل دخول المستخدم
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

// دالة للتحقق من صلاحيات المستخدم
function hasPermission($role) {
    if (!isLoggedIn()) {
        return false;
    }

    // التحقق من دور المستخدم
    $userRole = $_SESSION['user_role'] ?? '';

    // المسؤول لديه جميع الصلاحيات
    if ($userRole === 'admin') {
        return true;
    }

    // التحقق من الصلاحيات حسب الدور
    switch ($role) {
        case 'admin':
            return $userRole === 'admin';
        case 'manager':
            return in_array($userRole, ['admin', 'manager']);
        case 'employee':
            return in_array($userRole, ['admin', 'manager', 'employee']);
        case 'client':
            return in_array($userRole, ['admin', 'manager', 'employee', 'client']);
        default:
            return false;
    }
}

// دالة لتنظيف المدخلات
function sanitizeInput($input) {
    if (is_array($input)) {
        foreach ($input as $key => $value) {
            $input[$key] = sanitizeInput($value);
        }
        return $input;
    }

    return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
}

// دالة لتوليد رمز CSRF - تم تعليقها لتجنب التعارض مع التعريف في config/config.php
// function generateCSRFToken() {
//     if (!isset($_SESSION['csrf_token'])) {
//         $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
//     }
//
//     return $_SESSION['csrf_token'];
// }

// دالة للتحقق من رمز CSRF - تم تعليقها لتجنب التعارض مع التعريف في config/config.php
// function verifyCSRFToken($token) {
//     if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
//         return false;
//     }
//
//     return true;
// }

// دالة لإعادة توجيه المستخدم إلى صفحة تسجيل الدخول إذا لم يكن مسجل الدخول
function redirectIfNotLoggedIn() {
    if (!isLoggedIn()) {
        header('Location: login.php');
        exit;
    }
}

// دالة لإعادة توجيه المستخدم إلى لوحة التحكم إذا كان مسجل الدخول
function redirectIfLoggedIn() {
    if (isLoggedIn()) {
        header('Location: dashboard.php');
        exit;
    }
}

// دالة للحصول على اسم الفرع من معرفه
function getBranchName($pdo, $branchId) {
    if (empty($branchId)) {
        return 'غير محدد';
    }

    try {
        $stmt = $pdo->prepare("SELECT name FROM branches WHERE id = ?");
        $stmt->execute([$branchId]);
        $result = $stmt->fetch();

        if ($result) {
            return $result['name'];
        }

        return 'غير محدد';
    } catch (PDOException $e) {
        error_log('خطأ في الحصول على اسم الفرع: ' . $e->getMessage());
        return 'غير محدد';
    }
}

// دالة للحصول على رمز الفرع من معرفه
function getBranchCode($pdo, $branchId) {
    if (empty($branchId)) {
        return '';
    }

    try {
        $stmt = $pdo->prepare("SELECT code FROM branches WHERE id = ?");
        $stmt->execute([$branchId]);
        $result = $stmt->fetch();

        if ($result) {
            return $result['code'];
        }

        return '';
    } catch (PDOException $e) {
        error_log('خطأ في الحصول على رمز الفرع: ' . $e->getMessage());
        return '';
    }
}

// دالة للتحقق من وجود جدول في قاعدة البيانات
function tableExists($pdo, $tableName) {
    try {
        global $use_mysql;

        if ($use_mysql) {
            // استعلام MySQL
            $stmt = $pdo->prepare("SHOW TABLES LIKE ?");
            $stmt->execute([$tableName]);
        } else {
            // استعلام SQLite
            $stmt = $pdo->prepare("SELECT name FROM sqlite_master WHERE type='table' AND name=?");
            $stmt->execute([$tableName]);
        }

        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log('خطأ في التحقق من وجود الجدول: ' . $e->getMessage());
        return false;
    }
}

// دالة لتسجيل الخروج
function logout() {
    // بدء الجلسة إذا لم تكن قد بدأت
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    // حذف متغيرات الجلسة
    $_SESSION = [];

    // حذف ملف تعريف الارتباط للجلسة
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(
            session_name(),
            '',
            time() - 42000,
            $params["path"],
            $params["domain"],
            $params["secure"],
            $params["httponly"]
        );
    }

    // إنهاء الجلسة
    session_destroy();
}
?>
