<?php
/**
 * ملف الاتصال البديل بقاعدة البيانات باستخدام SQLite
 * يستخدم هذا الملف في حالة عدم توفر خادم MySQL
 */

// مسار ملف قاعدة البيانات SQLite
$db_file = __DIR__ . '/../database/ahmadi.sqlite';

// التأكد من وجود مجلد قاعدة البيانات
$db_dir = dirname($db_file);
if (!file_exists($db_dir)) {
    try {
        mkdir($db_dir, 0777, true);
        echo "<!-- تم إنشاء مجلد قاعدة البيانات بنجاح: $db_dir -->";
    } catch (Exception $e) {
        echo "<!-- فشل إنشاء مجلد قاعدة البيانات: " . $e->getMessage() . " -->";
    }
}

// التحقق من إمكانية الكتابة في المجلد
if (!is_writable($db_dir)) {
    echo "<!-- مجلد قاعدة البيانات غير قابل للكتابة: $db_dir -->";
    error_log("مجلد قاعدة البيانات غير قابل للكتابة: $db_dir");
}

// طباعة رسالة توضيحية للتأكد من المسار الصحيح
echo "<!-- مسار قاعدة البيانات SQLite: $db_file -->";

// إنشاء اتصال PDO باستخدام SQLite
try {
    // إنشاء كائن PDO للاتصال بقاعدة البيانات
    $conn = new PDO("sqlite:$db_file");

    // تعيين وضع الخطأ إلى استثناء
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // تعيين وضع الجلب الافتراضي إلى مصفوفة ترابطية
    $conn->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);

    // إنشاء متغير $pdo كنسخة من $conn للتوافق مع الكود الموجود
    $pdo = $conn;

    // تمكين القيود الأجنبية
    $conn->exec('PRAGMA foreign_keys = ON;');

} catch (PDOException $e) {
    die("
        <div style='font-family: Arial, sans-serif; direction: rtl; text-align: right; margin: 50px auto; max-width: 800px; padding: 20px; border: 1px solid #ddd; border-radius: 5px; background-color: #f9f9f9;'>
            <h2 style='color: #d9534f;'>خطأ في الاتصال بقاعدة البيانات SQLite</h2>
            <p>حدث خطأ أثناء محاولة الاتصال بقاعدة البيانات SQLite.</p>
            <p>رسالة الخطأ: " . $e->getMessage() . "</p>
            <p>رمز الخطأ: " . $e->getCode() . "</p>
        </div>
    ");
}

/**
 * دالة للتحقق من وجود جدول في قاعدة البيانات
 * @param string $tableName - اسم الجدول
 * @return bool - هل الجدول موجود أم لا
 */
function tableExists($tableName) {
    global $conn;

    try {
        $result = $conn->query("SELECT name FROM sqlite_master WHERE type='table' AND name='$tableName'");
        return $result->rowCount() > 0;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * دالة لإنشاء جداول قاعدة البيانات إذا لم تكن موجودة
 */
function createTablesIfNotExist() {
    global $conn;

    // جدول الفروع
    if (!tableExists('branches')) {
        $conn->exec("
            CREATE TABLE branches (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT NOT NULL,
                code TEXT NOT NULL,
                address TEXT,
                phone TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ");

        // إدراج بيانات الفروع الافتراضية
        $conn->exec("
            INSERT INTO branches (name, code) VALUES
            ('المكلا', 'MKL'),
            ('صنعاء', 'SNA'),
            ('عدن', 'ADN'),
            ('شحن', 'SH'),
            ('الصين', 'CN')
        ");
    }

    // جدول تسلسل أرقام الطلبات
    if (!tableExists('order_sequences')) {
        $conn->exec("
            CREATE TABLE order_sequences (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                branch_code TEXT NOT NULL UNIQUE,
                last_sequence_number INTEGER NOT NULL DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        ");

        // إدراج بيانات تسلسل أرقام الطلبات الافتراضية
        $conn->exec("
            INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES
            ('MKL', 0),
            ('SNA', 0),
            ('ADN', 0),
            ('SH', 0),
            ('CN', 0)
        ");
    }

    // جدول المستخدمين
    if (!tableExists('users')) {
        $conn->exec("
            CREATE TABLE users (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                username TEXT NOT NULL UNIQUE,
                full_name TEXT NOT NULL,
                email TEXT NOT NULL UNIQUE,
                password TEXT NOT NULL,
                role TEXT NOT NULL DEFAULT 'user',
                status TEXT NOT NULL DEFAULT 'pending',
                phone TEXT,
                registration_date TIMESTAMP,
                branch_id INTEGER,
                last_login TIMESTAMP,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE SET NULL
            )
        ");

        // إنشاء حساب المدير الافتراضي
        $adminPassword = password_hash('admin123', PASSWORD_DEFAULT);
        $currentDate = date('Y-m-d H:i:s');
        $conn->exec("
            INSERT INTO users (username, full_name, email, password, role, status, registration_date) VALUES
            ('admin', 'مدير النظام', 'admin@example.com', '$adminPassword', 'admin', 'active', '$currentDate')
        ");
    }
}

// إنشاء الجداول إذا لم تكن موجودة
createTablesIfNotExist();

/**
 * دالة لإنشاء رقم طلب جديد
 * @param string $branchName - اسم الفرع
 * @return string - رقم الطلب الجديد
 */
function generateOrderNumber($branchName) {
    global $conn;

    // الحصول على رمز الفرع
    $branchCode = '';
    switch ($branchName) {
        case 'المكلا':
            $branchCode = 'MKL';
            break;
        case 'صنعاء':
            $branchCode = 'SNA';
            break;
        case 'عدن':
            $branchCode = 'ADN';
            break;
        case 'شحن':
            $branchCode = 'SH';
            break;
        case 'الصين':
            $branchCode = 'CN';
            break;
        default:
            $branchCode = 'ORD';
    }

    // الحصول على السنة الحالية
    $year = date('y');

    // استخدام جدول order_sequences للحصول على الرقم التسلسلي
    $stmt = $conn->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
    $stmt->execute([$branchCode]);
    $result = $stmt->fetch();

    if ($result) {
        $lastNumber = $result['last_sequence_number'];
        $newNumber = $lastNumber + 1;

        // تحديث الرقم التسلسلي في قاعدة البيانات
        $updateStmt = $conn->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
        $updateStmt->execute([$newNumber, $branchCode]);

        // تنسيق الرقم التسلسلي
        $sequenceNumber = str_pad($newNumber, 2, '0', STR_PAD_LEFT);

        // إنشاء رقم الطلب الكامل
        $orderNumber = $branchCode . $year . '-' . $sequenceNumber;

        return $orderNumber;
    } else {
        // إذا لم يتم العثور على رمز الفرع، قم بإنشاء سجل جديد
        $insertStmt = $conn->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, 1)");
        $insertStmt->execute([$branchCode]);

        return $branchCode . $year . '-01';
    }
}
?>
