<?php
/**
 * ملف الاتصال الموحد بقاعدة البيانات
 * يستخدم هذا الملف لإنشاء اتصال بقاعدة البيانات واستخدامه في جميع أنحاء التطبيق
 * تم توحيد جميع ملفات الاتصال بقاعدة البيانات في هذا الملف
 */

// معلومات الاتصال بقاعدة البيانات
$host = '127.0.0.1';      // اسم المضيف (استخدام عنوان IP بدلاً من localhost)
$dbname = 'shipping_db';  // اسم قاعدة البيانات
$username = 'root';       // اسم المستخدم الافتراضي في XAMPP
$password = '';           // كلمة المرور الافتراضية في XAMPP فارغة

// التحقق من وجود قاعدة البيانات وإنشائها إذا لم تكن موجودة
try {
    // إنشاء اتصال PDO بدون تحديد قاعدة البيانات
    $pdo_init = new PDO("mysql:host=$host;charset=utf8mb4", $username, $password);
    $pdo_init->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // إنشاء قاعدة البيانات إذا لم تكن موجودة
    $pdo_init->exec("CREATE DATABASE IF NOT EXISTS `$dbname` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
} catch (PDOException $e) {
    $error_code = $e->getCode();
    $error_message = $e->getMessage();

    // تحقق من نوع الخطأ وتقديم رسالة مناسبة
    if ($error_code == 2002) {
        die("
            <div style='font-family: Arial, sans-serif; direction: rtl; text-align: right; margin: 50px auto; max-width: 800px; padding: 20px; border: 1px solid #ddd; border-radius: 5px; background-color: #f9f9f9;'>
                <h2 style='color: #d9534f;'>خطأ في الاتصال بقاعدة البيانات</h2>
                <p>يبدو أن خادم MySQL غير متاح أو لا يعمل. يرجى التأكد من تشغيل خادم MySQL في XAMPP قبل محاولة الوصول إلى النظام.</p>
                <h3>خطوات الحل:</h3>
                <ol>
                    <li>افتح لوحة تحكم XAMPP</li>
                    <li>انقر على زر Start بجانب خدمة MySQL</li>
                    <li>انتظر حتى تصبح الخدمة نشطة (يتحول لونها إلى الأخضر)</li>
                    <li>أعد تحميل هذه الصفحة</li>
                </ol>
                <p>رسالة الخطأ الأصلية: $error_message</p>
            </div>
        ");
    } else {
        die("
            <div style='font-family: Arial, sans-serif; direction: rtl; text-align: right; margin: 50px auto; max-width: 800px; padding: 20px; border: 1px solid #ddd; border-radius: 5px; background-color: #f9f9f9;'>
                <h2 style='color: #d9534f;'>خطأ في إنشاء قاعدة البيانات</h2>
                <p>حدث خطأ أثناء محاولة إنشاء قاعدة البيانات. يرجى التحقق من إعدادات الاتصال بقاعدة البيانات.</p>
                <p>رسالة الخطأ: $error_message</p>
                <p>رمز الخطأ: $error_code</p>
            </div>
        ");
    }
}

// إنشاء اتصال PDO
try {
    // إنشاء كائن PDO للاتصال بقاعدة البيانات
    $conn = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8mb4", $username, $password);

    // تعيين وضع الخطأ إلى استثناء
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // تعيين وضع الجلب الافتراضي إلى مصفوفة ترابطية
    $conn->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);

    // تعطيل محاكاة الاستعلامات المعدة
    $conn->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);

    // تعيين الاتصال لاستخدام UTF-8
    $conn->exec("SET NAMES utf8mb4");

    // إنشاء متغير $pdo كنسخة من $conn للتوافق مع الكود الموجود
    $pdo = $conn;

} catch (PDOException $e) {
    $error_code = $e->getCode();
    $error_message = $e->getMessage();

    // تحقق من نوع الخطأ وتقديم رسالة مناسبة
    if ($error_code == 1049) {
        die("
            <div style='font-family: Arial, sans-serif; direction: rtl; text-align: right; margin: 50px auto; max-width: 800px; padding: 20px; border: 1px solid #ddd; border-radius: 5px; background-color: #f9f9f9;'>
                <h2 style='color: #d9534f;'>خطأ في الاتصال بقاعدة البيانات</h2>
                <p>قاعدة البيانات '$dbname' غير موجودة. يرجى التأكد من إنشاء قاعدة البيانات أولاً.</p>
                <p>رسالة الخطأ: $error_message</p>
            </div>
        ");
    } else {
        die("
            <div style='font-family: Arial, sans-serif; direction: rtl; text-align: right; margin: 50px auto; max-width: 800px; padding: 20px; border: 1px solid #ddd; border-radius: 5px; background-color: #f9f9f9;'>
                <h2 style='color: #d9534f;'>خطأ في الاتصال بقاعدة البيانات</h2>
                <p>حدث خطأ أثناء محاولة الاتصال بقاعدة البيانات. يرجى التحقق من إعدادات الاتصال.</p>
                <p>رسالة الخطأ: $error_message</p>
                <p>رمز الخطأ: $error_code</p>
            </div>
        ");
    }
}

/**
 * دالة لإنشاء رقم طلب جديد
 * @param string $branchName - اسم الفرع
 * @return string - رقم الطلب الجديد
 */
function generateOrderNumber($branchName) {
    global $conn;

    // الحصول على رمز الفرع
    $branchCode = '';
    switch ($branchName) {
        case 'المكلا':
            $branchCode = 'MKL';
            break;
        case 'صنعاء':
            $branchCode = 'SNA';
            break;
        case 'عدن':
            $branchCode = 'ADN';
            break;
        case 'شحن':
            $branchCode = 'SH';
            break;
        case 'الصين':
            $branchCode = 'CN';
            break;
        default:
            $branchCode = 'ORD';
    }

    // الحصول على السنة الحالية
    $year = date('y');

    // استخدام جدول order_sequences للحصول على الرقم التسلسلي
    $stmt = $conn->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
    $stmt->execute([$branchCode]);
    $result = $stmt->fetch();

    if ($result) {
        $lastNumber = $result['last_sequence_number'];
        $newNumber = $lastNumber + 1;

        // تحديث الرقم التسلسلي في قاعدة البيانات
        $updateStmt = $conn->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
        $updateStmt->execute([$newNumber, $branchCode]);

        // تنسيق الرقم التسلسلي
        $sequenceNumber = str_pad($newNumber, 2, '0', STR_PAD_LEFT);

        // إنشاء رقم الطلب الكامل
        $orderNumber = $branchCode . $year . '-' . $sequenceNumber;

        return $orderNumber;
    } else {
        // إذا لم يتم العثور على رمز الفرع، قم بإنشاء سجل جديد
        $insertStmt = $conn->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, 1)");
        $insertStmt->execute([$branchCode]);

        return $branchCode . $year . '-01';
    }
}

/**
 * دالة للتحقق من وجود جدول في قاعدة البيانات
 * @param string $tableName - اسم الجدول
 * @return bool - هل الجدول موجود أم لا
 */
function tableExists($tableName) {
    global $conn;

    try {
        $result = $conn->query("SHOW TABLES LIKE '$tableName'");
        return $result->rowCount() > 0;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * دالة لإنشاء جداول قاعدة البيانات إذا لم تكن موجودة
 */
function createTablesIfNotExist() {
    global $conn;

    // جدول الفروع
    if (!tableExists('branches')) {
        $conn->exec("
            CREATE TABLE branches (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) NOT NULL,
                code VARCHAR(10) NOT NULL,
                address TEXT,
                phone VARCHAR(20),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // إدراج بيانات الفروع الافتراضية
        $conn->exec("
            INSERT INTO branches (name, code) VALUES
            ('المكلا', 'MKL'),
            ('صنعاء', 'SNA'),
            ('عدن', 'ADN'),
            ('شحن', 'SH'),
            ('الصين', 'CN')
        ");
    }

    // جدول تسلسل أرقام الطلبات
    if (!tableExists('order_sequences')) {
        $conn->exec("
            CREATE TABLE order_sequences (
                id INT AUTO_INCREMENT PRIMARY KEY,
                branch_code VARCHAR(10) NOT NULL UNIQUE,
                last_sequence_number INT NOT NULL DEFAULT 0,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // إدراج بيانات تسلسل أرقام الطلبات الافتراضية
        $conn->exec("
            INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES
            ('MKL', 0),
            ('SNA', 0),
            ('ADN', 0),
            ('SH', 0),
            ('CN', 0)
        ");
    }

    // جدول المستخدمين
    if (!tableExists('users')) {
        $conn->exec("
            CREATE TABLE users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                username VARCHAR(50) NOT NULL UNIQUE,
                password VARCHAR(255) NOT NULL,
                full_name VARCHAR(100) NOT NULL,
                email VARCHAR(100) NOT NULL UNIQUE,
                phone VARCHAR(20),
                role ENUM('admin', 'manager', 'employee', 'client') NOT NULL DEFAULT 'client',
                status ENUM('pending', 'active', 'inactive') NOT NULL DEFAULT 'pending',
                registration_date TIMESTAMP NULL DEFAULT NULL,
                branch_id INT,
                last_login DATETIME,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (branch_id) REFERENCES branches(id) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // إنشاء حساب المدير الافتراضي
        $adminPassword = password_hash('admin123', PASSWORD_DEFAULT);
        $conn->exec("
            INSERT INTO users (username, password, full_name, email, phone, role, status, registration_date) VALUES
            ('admin', '$adminPassword', 'مدير النظام', 'admin@example.com', '777123456', 'admin', 'active', NOW())
        ");
    }

    // جدول الطلبات والجداول الأخرى سيتم إنشاؤها من خلال ملفات SQL منفصلة
}

// إنشاء الجداول إذا لم تكن موجودة
createTablesIfNotExist();
?>
