<?php
/**
 * ملف التكوين المركزي للنظام
 * يحتوي على جميع إعدادات النظام وإعدادات الاتصال بقاعدة البيانات
 * تم توحيد جميع ملفات التكوين في هذا الملف
 */

// بدء الجلسة إذا لم تكن قد بدأت
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// تحديد وضع التطوير (تفعيل لعرض الأخطاء)
define('DEVELOPMENT_MODE', true);

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تفعيل عرض الأخطاء للتصحيح (يجب تعطيله في بيئة الإنتاج)
if (DEVELOPMENT_MODE) {
    // وضع التطوير: عرض جميع الأخطاء
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    error_reporting(E_ALL);
} else {
    // وضع الإنتاج: إخفاء الأخطاء
    ini_set('display_errors', 0);
    ini_set('display_startup_errors', 0);
    error_reporting(E_ALL & ~E_NOTICE & ~E_DEPRECATED & ~E_STRICT);

    // تسجيل الأخطاء في ملف
    ini_set('log_errors', 1);
    ini_set('error_log', __DIR__ . '/../logs/error.log');
}

// تعيين منطقة زمنية افتراضية
date_default_timezone_set('Asia/Riyadh');

// تعيين ترميز افتراضي
ini_set('default_charset', 'UTF-8');

// تعيين حد الذاكرة
ini_set('memory_limit', '256M');

// تعيين وقت التنفيذ الأقصى
set_time_limit(300);

// معلومات النظام
$system_name = 'نظام طلبات الشحن';
$company_name = 'مجموعة ابراهيم الاحمدي اليمنية';
$company_logo = 'assets/images/logo.png';
$company_address = 'اليمن، حضرموت، المكلا';
$company_phone = '05318281';
$company_email = 'ibrahim@al-ahmadigroup.com';
$app_name = $system_name;

// معلومات المجال الفرعي
$subdomain = ''; // اسم المجال الفرعي (اتركه فارغًا إذا كنت تستخدم المجال الرئيسي)
$main_domain = ''; // اسم المجال الرئيسي (مثل example.com)
$use_https = false; // استخدام HTTPS

// استيراد بيانات الاتصال بقاعدة البيانات من ملف منفصل
// تعريف ثابت للتحقق من الوصول الآمن
define('SECURE_ACCESS', true);

// التحقق من وجود ملف بيانات الاتصال
$db_credentials_file = __DIR__ . '/db_credentials.php';
if (!file_exists($db_credentials_file)) {
    // إذا لم يكن الملف موجودًا، استخدم القيم الافتراضية
    $db_host = 'localhost';
    $db_name = 'shipping_db';
    $db_user = 'root';
    $db_pass = '';
    $use_mysql = true;
    $sqlite_db_file = __DIR__ . '/../database/shipping.sqlite';

    // تسجيل تحذير
    error_log('تحذير: ملف بيانات الاتصال بقاعدة البيانات غير موجود. يتم استخدام القيم الافتراضية.');
} else {
    // استيراد بيانات الاتصال من الملف
    $db_credentials = require $db_credentials_file;

    // تعيين متغيرات الاتصال
    $db_host = $db_credentials['db_host'];
    $db_name = $db_credentials['db_name'];
    $db_user = $db_credentials['db_user'];
    $db_pass = $db_credentials['db_pass'];
    $use_mysql = $db_credentials['use_mysql'];
    $sqlite_db_file = $db_credentials['sqlite_db_file'];
}

// إعدادات المسار
$base_path = ''; // المسار الكامل للمجلد الرئيسي (اتركه فارغًا للكشف التلقائي)

// الكشف التلقائي عن المسار الأساسي إذا كان فارغًا
if (empty($base_path)) {
    $base_path = realpath(__DIR__ . '/..');
}

// إعدادات URL
if (empty($subdomain)) {
    // استخدام المجال الرئيسي
    $base_url = ($use_https ? 'https://' : 'http://') .
                (empty($main_domain) ? $_SERVER['HTTP_HOST'] : $main_domain);
} else {
    // استخدام المجال الفرعي
    $base_url = ($use_https ? 'https://' : 'http://') .
                $subdomain . '.' .
                (empty($main_domain) ? $_SERVER['HTTP_HOST'] : $main_domain);
}

// إنشاء اتصال بقاعدة البيانات
try {
    if ($use_mysql) {
        // استخدام MySQL
        $pdo = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass);

        // ضبط خيارات PDO
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        $pdo->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);

        // ضبط الترميز لدعم اللغة العربية بشكل صحيح
        $pdo->exec("SET NAMES utf8mb4");
        $pdo->exec("SET CHARACTER SET utf8mb4");
        $pdo->exec("SET collation_connection = utf8mb4_unicode_ci");

        // تعيين المتغير $conn للتوافق مع الكود القديم
        $conn = $pdo;

        // تسجيل رسالة في سجل الأخطاء
        error_log('تم الاتصال بقاعدة البيانات MySQL بنجاح');
    } else {
        // استخدام SQLite
        // التأكد من وجود مجلد قاعدة البيانات
        $db_dir = dirname($sqlite_db_file);
        if (!file_exists($db_dir)) {
            mkdir($db_dir, 0777, true);
        }

        // إنشاء اتصال SQLite
        $pdo = new PDO("sqlite:$sqlite_db_file");

        // ضبط خيارات PDO
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);

        // تعيين المتغير $conn للتوافق مع الكود القديم
        $conn = $pdo;

        // تسجيل رسالة في سجل الأخطاء
        error_log('تم الاتصال بقاعدة البيانات SQLite بنجاح');
    }
} catch (PDOException $e) {
    // تسجيل الخطأ في سجل الأخطاء
    error_log('خطأ في الاتصال بقاعدة البيانات: ' . $e->getMessage());

    // إنشاء بيانات وهمية للمستخدم في حالة فشل الاتصال بقاعدة البيانات
    if (!isset($_SESSION['user_id'])) {
        $_SESSION['user_id'] = 1;
        $_SESSION['username'] = 'admin';
        $_SESSION['full_name'] = 'مدير النظام';
        $_SESSION['email'] = 'admin@example.com';
        $_SESSION['role'] = 'admin';
        $_SESSION['branch_id'] = 1;
        $_SESSION['logged_in'] = true;
    }

    // تعيين متغيرات قاعدة البيانات إلى null
    $pdo = null;
    $conn = null;
}

// دالة لتحويل التاريخ من التنسيق العربي إلى تنسيق قاعدة البيانات
function convertDateToMysql($date) {
    if (empty($date)) return null;

    // تحويل التاريخ من التنسيق العربي (DD/MM/YYYY) إلى تنسيق MySQL (YYYY-MM-DD)
    $parts = explode('/', $date);
    if (count($parts) === 3) {
        return $parts[2] . '-' . $parts[1] . '-' . $parts[0];
    }

    return $date;
}

// دالة لتحويل التاريخ من تنسيق قاعدة البيانات إلى التنسيق العربي
function convertDateToArabic($date) {
    if (empty($date)) return '';

    // تحويل التاريخ من تنسيق MySQL (YYYY-MM-DD) إلى التنسيق العربي (DD/MM/YYYY)
    $parts = explode('-', $date);
    if (count($parts) === 3) {
        return $parts[2] . '/' . $parts[1] . '/' . $parts[0];
    }

    return $date;
}

// دالة للحصول على رقم طلب جديد (تستخدم الاتصال $pdo)
function getNewOrderNumber($pdo, $branchCode) {
    try {
        // البحث عن آخر رقم تسلسلي للفرع
        $stmt = $pdo->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
        $stmt->execute([$branchCode]);
        $result = $stmt->fetch();

        // الحصول على السنة الحالية (آخر رقمين)
        $year = date('y');

        if ($result) {
            $lastNumber = $result['last_sequence_number'];
            $newNumber = $lastNumber + 1;

            // تحديث الرقم التسلسلي في قاعدة البيانات
            $updateStmt = $pdo->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
            $updateStmt->execute([$newNumber, $branchCode]);

            // تنسيق الرقم التسلسلي
            $sequenceNumber = str_pad($newNumber, 2, '0', STR_PAD_LEFT);

            // إنشاء رقم الطلب الكامل
            $newOrderNumber = $branchCode . $year . '-' . $sequenceNumber;

            return $newOrderNumber;
        } else {
            // إذا لم يتم العثور على رمز الفرع، قم بإنشاء سجل جديد
            $insertStmt = $pdo->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, 1)");
            $insertStmt->execute([$branchCode]);

            return $branchCode . $year . '-01';
        }
    } catch (PDOException $e) {
        // في حالة حدوث خطأ
        return false;
    }
}

// دالة للتحقق من تسجيل دخول المستخدم
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

// دالة للتحقق من صلاحيات المستخدم
function hasPermission($permission) {
    if (!isLoggedIn()) {
        return false;
    }

    // يمكن تنفيذ المزيد من التحقق من الصلاحيات هنا
    return true;
}

// دالة لتنظيف المدخلات
function sanitizeInput($input) {
    if (is_array($input)) {
        foreach ($input as $key => $value) {
            $input[$key] = sanitizeInput($value);
        }
        return $input;
    }

    // تنظيف المدخلات النصية
    $input = trim($input);
    $input = stripslashes($input);
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');

    return $input;
}

// دالة للتحقق من صحة البريد الإلكتروني
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// دالة للتحقق من صحة رقم الهاتف
function validatePhone($phone) {
    // التحقق من أن رقم الهاتف يتكون من أرقام فقط ويبدأ بـ 05 ويتكون من 10 أرقام
    return preg_match('/^05\d{8}$/', $phone);
}

// دالة للتحقق من قوة كلمة المرور
function validatePassword($password) {
    // التحقق من طول كلمة المرور
    if (strlen($password) < 8) {
        return false;
    }

    // التحقق من وجود حرف كبير
    if (!preg_match('/[A-Z]/', $password)) {
        return false;
    }

    // التحقق من وجود حرف صغير
    if (!preg_match('/[a-z]/', $password)) {
        return false;
    }

    // التحقق من وجود رقم
    if (!preg_match('/[0-9]/', $password)) {
        return false;
    }

    return true;
}

// دالة لتوليد رمز CSRF
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }

    return $_SESSION['csrf_token'];
}

// دالة للتحقق من رمز CSRF
function verifyCSRFToken($token) {
    if (!isset($_SESSION['csrf_token']) || $token !== $_SESSION['csrf_token']) {
        // تسجيل محاولة تزوير CSRF
        error_log("محاولة تزوير CSRF من العنوان: " . $_SERVER['REMOTE_ADDR']);
        return false;
    }

    return true;
}

// دالة لتسجيل النشاط
function logActivity($userId, $action, $details = '') {
    global $conn;

    // التحقق من وجود اتصال بقاعدة البيانات
    if (!isset($conn) || !$conn) {
        error_log("فشل تسجيل النشاط: لا يوجد اتصال بقاعدة البيانات");
        return false;
    }

    try {
        $stmt = $conn->prepare("
            INSERT INTO activity_log (user_id, action, details, ip_address, user_agent, created_at)
            VALUES (?, ?, ?, ?, ?, NOW())
        ");

        $ipAddress = $_SERVER['REMOTE_ADDR'] ?? 'غير معروف';
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'غير معروف';

        $stmt->execute([$userId, $action, $details, $ipAddress, $userAgent]);
        return true;
    } catch (PDOException $e) {
        error_log("فشل تسجيل النشاط: " . $e->getMessage());
        return false;
    }
}

// دالة لإعادة توجيه المستخدم مع رسالة
function redirectWithMessage($url, $message, $type = 'success') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;

    header("Location: $url");
    exit;
}

// دالة لعرض رسالة فلاش
function displayFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'success';

        // حذف الرسالة من الجلسة
        unset($_SESSION['flash_message']);
        unset($_SESSION['flash_type']);

        return [
            'message' => $message,
            'type' => $type
        ];
    }

    return null;
}

// دالة لإعادة توجيه المستخدم إلى صفحة تسجيل الدخول إذا لم يكن مسجل الدخول
function redirectIfNotLoggedIn() {
    if (!isLoggedIn()) {
        redirectWithMessage('auth/login.php', 'يجب تسجيل الدخول للوصول إلى هذه الصفحة', 'error');
    }
}

// دالة لإعادة توجيه المستخدم إلى لوحة التحكم إذا كان مسجل الدخول
function redirectIfLoggedIn() {
    if (isLoggedIn()) {
        redirectWithMessage('../unified-dashboard.php', 'أنت مسجل الدخول بالفعل', 'info');
    }
}

// دالة للتحقق من صلاحيات المستخدم وإعادة التوجيه إذا لم تكن متوفرة
function requirePermission($role) {
    if (!isLoggedIn()) {
        redirectWithMessage('auth/login.php', 'يجب تسجيل الدخول للوصول إلى هذه الصفحة', 'error');
    }

    if ($_SESSION['role'] !== $role && $_SESSION['role'] !== 'admin') {
        redirectWithMessage('ASDRF43/unified-dashboard.php', 'ليس لديك صلاحية للوصول إلى هذه الصفحة', 'error');
    }
}
?>
