<?php
/**
 * ملف التحقق من متطلبات النظام
 * يستخدم هذا الملف للتحقق من توافق الاستضافة مع متطلبات النظام
 */

// تعيين ترميز الصفحة
header('Content-Type: text/html; charset=utf-8');

// تعطيل عرض الأخطاء
ini_set('display_errors', 0);

// قائمة المتطلبات
$requirements = [
    'php_version' => [
        'name' => 'إصدار PHP',
        'required' => '7.4.0',
        'current' => PHP_VERSION,
        'status' => version_compare(PHP_VERSION, '7.4.0', '>='),
        'critical' => true
    ],
    'pdo_mysql' => [
        'name' => 'PDO MySQL',
        'required' => 'مفعل',
        'current' => extension_loaded('pdo_mysql') ? 'مفعل' : 'غير مفعل',
        'status' => extension_loaded('pdo_mysql'),
        'critical' => true
    ],
    'mbstring' => [
        'name' => 'Mbstring',
        'required' => 'مفعل',
        'current' => extension_loaded('mbstring') ? 'مفعل' : 'غير مفعل',
        'status' => extension_loaded('mbstring'),
        'critical' => true
    ],
    'json' => [
        'name' => 'JSON',
        'required' => 'مفعل',
        'current' => extension_loaded('json') ? 'مفعل' : 'غير مفعل',
        'status' => extension_loaded('json'),
        'critical' => true
    ],
    'gd' => [
        'name' => 'GD Library',
        'required' => 'مفعل',
        'current' => extension_loaded('gd') ? 'مفعل' : 'غير مفعل',
        'status' => extension_loaded('gd'),
        'critical' => false
    ],
    'file_uploads' => [
        'name' => 'تحميل الملفات',
        'required' => 'مفعل',
        'current' => ini_get('file_uploads') ? 'مفعل' : 'غير مفعل',
        'status' => ini_get('file_uploads'),
        'critical' => true
    ],
    'max_upload_size' => [
        'name' => 'الحد الأقصى لحجم التحميل',
        'required' => '8M',
        'current' => ini_get('upload_max_filesize'),
        'status' => (int)ini_get('upload_max_filesize') >= 8,
        'critical' => false
    ],
    'max_post_size' => [
        'name' => 'الحد الأقصى لحجم POST',
        'required' => '8M',
        'current' => ini_get('post_max_size'),
        'status' => (int)ini_get('post_max_size') >= 8,
        'critical' => false
    ],
    'memory_limit' => [
        'name' => 'حد الذاكرة',
        'required' => '128M',
        'current' => ini_get('memory_limit'),
        'status' => (int)ini_get('memory_limit') >= 128,
        'critical' => false
    ],
    'config_writable' => [
        'name' => 'إمكانية الكتابة في مجلد config',
        'required' => 'قابل للكتابة',
        'current' => is_writable(__DIR__ . '/config') ? 'قابل للكتابة' : 'غير قابل للكتابة',
        'status' => is_writable(__DIR__ . '/config'),
        'critical' => true
    ],
    'uploads_writable' => [
        'name' => 'إمكانية الكتابة في مجلد uploads',
        'required' => 'قابل للكتابة',
        'current' => is_writable(__DIR__ . '/uploads') ? 'قابل للكتابة' : 'غير قابل للكتابة',
        'status' => is_writable(__DIR__ . '/uploads'),
        'critical' => true
    ],
    'logs_writable' => [
        'name' => 'إمكانية الكتابة في مجلد logs',
        'required' => 'قابل للكتابة',
        'current' => is_writable(__DIR__ . '/logs') ? 'قابل للكتابة' : 'غير قابل للكتابة',
        'status' => is_writable(__DIR__ . '/logs'),
        'critical' => true
    ]
];

// التحقق من وجود المجلدات المطلوبة
$required_dirs = ['config', 'uploads', 'logs'];
foreach ($required_dirs as $dir) {
    if (!file_exists(__DIR__ . '/' . $dir)) {
        mkdir(__DIR__ . '/' . $dir, 0755, true);
    }
}

// حساب عدد المتطلبات المستوفاة
$passed = 0;
$critical_failed = 0;
foreach ($requirements as $req) {
    if ($req['status']) {
        $passed++;
    } elseif ($req['critical']) {
        $critical_failed++;
    }
}

// حساب النسبة المئوية للمتطلبات المستوفاة
$total = count($requirements);
$percentage = round(($passed / $total) * 100);

// تحديد حالة النظام
$system_status = 'جاهز للتثبيت';
if ($critical_failed > 0) {
    $system_status = 'غير جاهز للتثبيت - هناك متطلبات أساسية غير مستوفاة';
} elseif ($passed < $total) {
    $system_status = 'جاهز للتثبيت مع تحذيرات';
}

// عرض النتائج
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>التحقق من متطلبات النظام</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f8f9fa;
            color: #333;
            margin: 0;
            padding: 20px;
            direction: rtl;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            padding: 20px;
        }
        h1 {
            color: #2c3e50;
            text-align: center;
            margin-bottom: 30px;
        }
        .progress-container {
            margin: 20px 0;
            background-color: #e9ecef;
            border-radius: 4px;
            height: 20px;
            overflow: hidden;
        }
        .progress-bar {
            height: 100%;
            background-color: #4caf50;
            text-align: center;
            color: white;
            line-height: 20px;
            transition: width 0.5s;
        }
        .status {
            text-align: center;
            margin: 20px 0;
            padding: 10px;
            border-radius: 4px;
            font-weight: bold;
        }
        .status.success {
            background-color: #d4edda;
            color: #155724;
        }
        .status.warning {
            background-color: #fff3cd;
            color: #856404;
        }
        .status.danger {
            background-color: #f8d7da;
            color: #721c24;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        th, td {
            padding: 12px 15px;
            text-align: right;
            border-bottom: 1px solid #e9ecef;
        }
        th {
            background-color: #f8f9fa;
            font-weight: bold;
        }
        .success-icon {
            color: #28a745;
        }
        .warning-icon {
            color: #ffc107;
        }
        .danger-icon {
            color: #dc3545;
        }
        .btn {
            display: inline-block;
            padding: 10px 20px;
            background-color: #007bff;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            font-weight: bold;
            margin-top: 20px;
        }
        .btn:hover {
            background-color: #0069d9;
        }
        .text-center {
            text-align: center;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>التحقق من متطلبات النظام</h1>
        
        <div class="progress-container">
            <div class="progress-bar" style="width: <?php echo $percentage; ?>%">
                <?php echo $percentage; ?>%
            </div>
        </div>
        
        <div class="status <?php echo $critical_failed > 0 ? 'danger' : ($passed < $total ? 'warning' : 'success'); ?>">
            <?php echo $system_status; ?>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>المتطلب</th>
                    <th>القيمة المطلوبة</th>
                    <th>القيمة الحالية</th>
                    <th>الحالة</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($requirements as $key => $req): ?>
                <tr>
                    <td><?php echo $req['name']; ?> <?php echo $req['critical'] ? '<strong>(أساسي)</strong>' : ''; ?></td>
                    <td><?php echo $req['required']; ?></td>
                    <td><?php echo $req['current']; ?></td>
                    <td>
                        <?php if ($req['status']): ?>
                            <span class="success-icon">✓</span>
                        <?php else: ?>
                            <span class="<?php echo $req['critical'] ? 'danger-icon' : 'warning-icon'; ?>">✗</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <div class="text-center">
            <?php if ($critical_failed == 0): ?>
                <a href="index.php" class="btn">الانتقال إلى الصفحة الرئيسية</a>
            <?php else: ?>
                <p>يرجى معالجة المتطلبات الأساسية قبل تثبيت النظام.</p>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
