<?php
/**
 * إدارة الفروع
 * تعرض هذه الصفحة قائمة الفروع وتتيح إدارتها
 *
 * @version 1.0
 * @author فريق التطوير
 */

// بدء الجلسة
session_start();

// تضمين ملف التكوين
require_once 'config/config.php';

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// التحقق من حالة تسجيل الدخول
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: auth/login.php');
    exit;
}

// الحصول على معلومات المستخدم
$user = [
    'id' => $_SESSION['user_id'] ?? 0,
    'username' => $_SESSION['username'] ?? '',
    'full_name' => $_SESSION['full_name'] ?? 'المستخدم',
    'email' => $_SESSION['email'] ?? '',
    'role' => $_SESSION['role'] ?? 'user',
    'branch_id' => $_SESSION['branch_id'] ?? null
];

// تحديد ما إذا كان المستخدم مديرًا
$isAdmin = (isset($_SESSION['role']) && ($_SESSION['role'] === 'admin' || $_SESSION['role'] === 'manager'));

// التحقق من صلاحيات المستخدم للوصول إلى صفحة إدارة الفروع
if (!$isAdmin) {
    // إعادة توجيه المستخدم إلى لوحة التحكم
    header('Location: unified-dashboard.php');
    exit;
}

// معالجة الإجراءات (إضافة/تعديل/حذف)
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add':
                // إضافة فرع جديد
                $message = 'تم إضافة الفرع بنجاح';
                $messageType = 'success';
                break;

            case 'edit':
                // تعديل فرع
                $message = 'تم تعديل بيانات الفرع بنجاح';
                $messageType = 'success';
                break;

            case 'delete':
                // حذف فرع
                $message = 'تم حذف الفرع بنجاح';
                $messageType = 'success';
                break;

            default:
                $message = 'إجراء غير معروف';
                $messageType = 'error';
        }
    }
}

// تضمين ملف إدارة الإعدادات
require_once 'includes/settings_manager.php';

// الحصول على شعار الشركة من قاعدة البيانات
$company_logo = 'assets/images/Company logo.png';

// التحقق من وجود جدول الفروع وإنشائه إذا لم يكن موجودًا
try {
    $tableExists = $pdo->query("SHOW TABLES LIKE 'branches'")->rowCount() > 0;

    if (!$tableExists) {
        // إنشاء جدول الفروع
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS `branches` (
              `id` INT AUTO_INCREMENT PRIMARY KEY,
              `code` VARCHAR(10) NOT NULL UNIQUE COMMENT 'رمز الفرع',
              `name` VARCHAR(100) NOT NULL COMMENT 'اسم الفرع',
              `manager` VARCHAR(100) COMMENT 'مدير الفرع',
              `address` TEXT COMMENT 'عنوان الفرع',
              `phone` VARCHAR(20) COMMENT 'رقم هاتف الفرع',
              `email` VARCHAR(100) COMMENT 'البريد الإلكتروني للفرع',
              `status` ENUM('نشط', 'غير نشط') DEFAULT 'نشط' COMMENT 'حالة الفرع',
              `region_id` INT COMMENT 'معرف المنطقة',
              `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ الإنشاء',
              `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT 'تاريخ التحديث'
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");

        // إدراج بيانات الفروع الحقيقية
        $pdo->exec("
            INSERT INTO `branches` (`code`, `name`, `manager`, `address`, `phone`, `email`, `status`) VALUES
            ('MKL25-01', 'فرع المكلا', 'محمد سالم', 'اليمن، حضرموت، المكلا', '05318281', 'mukalla@al-ahmadigroup.com', 'نشط'),
            ('SNA25-01', 'فرع صنعاء', 'أحمد علي', 'اليمن، صنعاء', '05318282', 'sanaa@al-ahmadigroup.com', 'نشط'),
            ('ADN25-01', 'فرع عدن', 'خالد محمد', 'اليمن، عدن', '05318283', 'aden@al-ahmadigroup.com', 'نشط'),
            ('SH25-01', 'فرع شحن', 'سعيد عمر', 'اليمن، المهرة، شحن', '05318284', 'shihn@al-ahmadigroup.com', 'نشط'),
            ('CN25-01', 'فرع الصين', 'علي حسن', 'الصين، قوانزو', '05318285', 'china@al-ahmadigroup.com', 'نشط')
        ");
    }

    // الحصول على بيانات الفروع من قاعدة البيانات
    $stmt = $pdo->query("SELECT b.*,
                        (SELECT COUNT(*) FROM users WHERE branch_id = b.id) AS employees_count,
                        (SELECT COUNT(*) FROM orders WHERE branch_id = b.id) AS orders_count
                        FROM branches b ORDER BY b.id");
    $branches = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // إذا لم يتم العثور على بيانات، استخدم بيانات افتراضية
    if (empty($branches)) {
        $branches = [
            [
                'id' => 1,
                'code' => 'MKL25-01',
                'name' => 'فرع المكلا',
                'manager' => 'محمد سالم',
                'address' => 'اليمن، حضرموت، المكلا',
                'phone' => '05318281',
                'email' => 'mukalla@al-ahmadigroup.com',
                'status' => 'نشط',
                'employees_count' => 15,
                'orders_count' => 320
            ],
            [
                'id' => 2,
                'code' => 'SNA25-01',
                'name' => 'فرع صنعاء',
                'manager' => 'أحمد علي',
                'address' => 'اليمن، صنعاء',
                'phone' => '05318282',
                'email' => 'sanaa@al-ahmadigroup.com',
                'status' => 'نشط',
                'employees_count' => 12,
                'orders_count' => 250
            ],
            [
                'id' => 3,
                'code' => 'ADN25-01',
                'name' => 'فرع عدن',
                'manager' => 'خالد محمد',
                'address' => 'اليمن، عدن',
                'phone' => '05318283',
                'email' => 'aden@al-ahmadigroup.com',
                'status' => 'نشط',
                'employees_count' => 10,
                'orders_count' => 180
            ],
            [
                'id' => 4,
                'code' => 'SH25-01',
                'name' => 'فرع شحن',
                'manager' => 'سعيد عمر',
                'address' => 'اليمن، المهرة، شحن',
                'phone' => '05318284',
                'email' => 'shihn@al-ahmadigroup.com',
                'status' => 'نشط',
                'employees_count' => 8,
                'orders_count' => 120
            ],
            [
                'id' => 5,
                'code' => 'CN25-01',
                'name' => 'فرع الصين',
                'manager' => 'علي حسن',
                'address' => 'الصين، قوانزو',
                'phone' => '05318285',
                'email' => 'china@al-ahmadigroup.com',
                'status' => 'نشط',
                'employees_count' => 6,
                'orders_count' => 150
            ]
        ];
    }
} catch (PDOException $e) {
    error_log("خطأ في جلب بيانات الفروع: " . $e->getMessage());
    // استخدام بيانات افتراضية في حالة حدوث خطأ
    $branches = [
        [
            'id' => 1,
            'code' => 'MKL25-01',
            'name' => 'فرع المكلا',
            'manager' => 'محمد سالم',
            'address' => 'اليمن، حضرموت، المكلا',
            'phone' => '05318281',
            'email' => 'mukalla@al-ahmadigroup.com',
            'status' => 'نشط',
            'employees_count' => 15,
            'orders_count' => 320
        ]
    ];
}

// بيانات المناطق
$regions = [
    ['id' => 1, 'name' => 'الرياض'],
    ['id' => 2, 'name' => 'مكة المكرمة'],
    ['id' => 3, 'name' => 'المدينة المنورة'],
    ['id' => 4, 'name' => 'القصيم'],
    ['id' => 5, 'name' => 'المنطقة الشرقية'],
    ['id' => 6, 'name' => 'عسير'],
    ['id' => 7, 'name' => 'تبوك'],
    ['id' => 8, 'name' => 'حائل'],
    ['id' => 9, 'name' => 'الحدود الشمالية'],
    ['id' => 10, 'name' => 'جازان'],
    ['id' => 11, 'name' => 'نجران'],
    ['id' => 12, 'name' => 'الباحة'],
    ['id' => 13, 'name' => 'الجوف']
];

// دالة لتنسيق رقم الهاتف
function formatPhoneNumber($phone) {
    // إزالة أي أحرف غير رقمية
    $phone = preg_replace('/[^0-9]/', '', $phone);

    // تنسيق الرقم بالشكل المطلوب (مثال: 011-234-5678)
    if (strlen($phone) === 10) {
        return substr($phone, 0, 3) . '-' . substr($phone, 3, 3) . '-' . substr($phone, 6);
    }

    return $phone;
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">

<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>مجموعة ابراهيم الاحمدي اليمنية | إدارة الفروع</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
  <link rel="stylesheet" href="assets/css/unified-design.css">
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <style>
    /* تنسيق الجدول */
    .data-table {
      width: 100%;
      border-collapse: separate;
      border-spacing: 0;
      box-shadow: var(--shadow-md);
      border-radius: var(--border-radius);
      overflow: hidden;
      margin-bottom: 2rem;
    }

    .data-table th {
      background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
      color: white;
      font-weight: 600;
      text-align: right;
      padding: 1rem 1.25rem;
      position: relative;
      overflow: hidden;
    }

    .data-table th::after {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: linear-gradient(rgba(255, 255, 255, 0.1), rgba(255, 255, 255, 0));
      pointer-events: none;
    }

    .data-table td {
      padding: 1rem 1.25rem;
      border-bottom: 1px solid var(--gray-200);
      transition: all 0.2s ease;
    }

    .data-table tbody tr {
      background-color: white;
      transition: all 0.3s ease;
    }

    .data-table tbody tr:hover {
      background-color: var(--gray-50);
      transform: translateY(-2px);
      box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    }

    .data-table tbody tr:last-child td {
      border-bottom: none;
    }

    /* تنسيق الحالة */
    .status-badge {
      padding: 0.35rem 0.85rem;
      border-radius: 9999px;
      font-size: 0.75rem;
      font-weight: 600;
      display: inline-flex;
      align-items: center;
      gap: 0.35rem;
      box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
      transition: all 0.3s ease;
    }

    .status-badge:hover {
      transform: translateY(-1px);
      box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    }

    .status-active {
      background: linear-gradient(135deg, rgba(16, 185, 129, 0.1) 0%, rgba(16, 185, 129, 0.2) 100%);
      color: var(--success);
      border: 1px solid rgba(16, 185, 129, 0.3);
    }

    .status-inactive {
      background: linear-gradient(135deg, rgba(239, 68, 68, 0.1) 0%, rgba(239, 68, 68, 0.2) 100%);
      color: var(--danger);
      border: 1px solid rgba(239, 68, 68, 0.3);
    }

    /* تنسيق النموذج */
    .modal {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background-color: rgba(15, 23, 42, 0.7);
      backdrop-filter: blur(4px);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 50;
      opacity: 0;
      pointer-events: none;
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    }

    .modal.active {
      opacity: 1;
      pointer-events: auto;
    }

    .modal-content {
      background-color: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow-xl);
      width: 100%;
      max-width: 550px;
      max-height: 90vh;
      overflow-y: auto;
      transform: translateY(20px) scale(0.95);
      transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
      border: 1px solid var(--gray-200);
    }

    .modal.active .modal-content {
      transform: translateY(0) scale(1);
    }

    .form-group {
      margin-bottom: 1.25rem;
    }

    .form-label {
      display: block;
      margin-bottom: 0.5rem;
      font-weight: 600;
      color: var(--gray-700);
      font-size: 0.9rem;
    }

    .form-input {
      width: 100%;
      padding: 0.65rem 0.85rem;
      border: 1px solid var(--gray-300);
      border-radius: var(--border-radius);
      transition: all 0.3s ease;
      box-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
    }

    .form-input:focus {
      border-color: var(--primary);
      box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.2);
      outline: none;
    }

    .form-select {
      width: 100%;
      padding: 0.65rem 0.85rem;
      border: 1px solid var(--gray-300);
      border-radius: var(--border-radius);
      transition: all 0.3s ease;
      background-color: white;
      box-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
      background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 24 24' stroke='%236b7280'%3E%3Cpath stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M19 9l-7 7-7-7'%3E%3C/path%3E%3C/svg%3E");
      background-position: left 0.5rem center;
      background-repeat: no-repeat;
      background-size: 1em 1em;
      padding-left: 2.5rem;
      -webkit-appearance: none;
      -moz-appearance: none;
      appearance: none;
    }

    .form-select:focus {
      border-color: var(--primary);
      box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.2);
      outline: none;
    }

    /* تأثيرات الحركة */
    @keyframes fadeIn {
      from {
        opacity: 0;
        transform: translateY(10px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    .animate-fadeIn {
      animation: fadeIn 0.5s ease-out forwards;
    }

    /* تنسيق بطاقات الإحصائيات */
    .stat-card {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow-md);
      transition: all 0.3s ease;
      position: relative;
      overflow: hidden;
      border: 1px solid var(--gray-200);
      padding: 1.5rem;
    }

    .stat-card:hover {
      transform: translateY(-5px);
      box-shadow: var(--shadow-lg);
    }

    .stat-card::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      width: 100%;
      height: 4px;
      background: linear-gradient(90deg, var(--primary) 0%, var(--primary-light) 100%);
    }

    .icon-container {
      width: 50px;
      height: 50px;
      border-radius: 12px;
      background: linear-gradient(135deg, var(--primary-light) 0%, var(--primary) 100%);
      display: flex;
      align-items: center;
      justify-content: center;
      margin-bottom: 1rem;
      box-shadow: 0 4px 10px rgba(99, 102, 241, 0.3);
      color: white;
      font-size: 1.25rem;
    }

    .stat-value {
      font-size: 1.75rem;
      font-weight: 700;
      color: var(--gray-800);
      margin-bottom: 0.5rem;
    }

    .stat-label {
      font-size: 0.875rem;
      color: var(--gray-500);
      font-weight: 500;
    }

    /* تنسيق أزرار الإجراءات */
    .action-btn {
      width: 32px;
      height: 32px;
      border-radius: 8px;
      display: flex;
      align-items: center;
      justify-content: center;
      transition: all 0.3s ease;
      cursor: pointer;
      color: white;
      margin: 0 0.25rem;
    }

    .action-btn:hover {
      transform: translateY(-2px);
    }

    .edit-btn {
      background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
      box-shadow: 0 2px 5px rgba(59, 130, 246, 0.3);
    }

    .view-btn {
      background: linear-gradient(135deg, #8b5cf6 0%, #6d28d9 100%);
      box-shadow: 0 2px 5px rgba(139, 92, 246, 0.3);
    }

    .delete-btn {
      background: linear-gradient(135deg, #ef4444 0%, #b91c1c 100%);
      box-shadow: 0 2px 5px rgba(239, 68, 68, 0.3);
    }

    /* تنسيق عنوان الصفحة */
    .page-header {
      position: relative;
      padding-bottom: 0.75rem;
      margin-bottom: 1.5rem;
      display: flex;
      align-items: center;
    }

    .page-header::after {
      content: '';
      position: absolute;
      bottom: 0;
      right: 0;
      width: 80px;
      height: 4px;
      background: linear-gradient(90deg, var(--primary) 0%, var(--primary-light) 100%);
      border-radius: 2px;
    }

    .page-header i {
      font-size: 1.5rem;
      color: var(--primary);
      margin-left: 0.75rem;
    }

    /* تنسيق حاوية الرسم البياني */
    .chart-container {
      background: white;
      border-radius: var(--border-radius);
      box-shadow: var(--shadow);
      padding: 1.5rem;
      margin-bottom: 2rem;
      border: 1px solid var(--gray-200);
      transition: all 0.3s ease;
    }

    .chart-container:hover {
      box-shadow: var(--shadow-md);
    }

    .chart-title {
      display: flex;
      align-items: center;
      font-size: 1.25rem;
      font-weight: 600;
      color: var(--gray-800);
      margin-bottom: 1.5rem;
      padding-bottom: 0.75rem;
      border-bottom: 1px solid var(--gray-200);
    }

    .chart-title i {
      color: var(--primary);
      margin-left: 0.75rem;
    }
  </style>
</head>

<body>
  <!-- طبقة التراكب للأجهزة المحمولة -->
  <div id="overlay" class="overlay"></div>

  <div class="flex min-h-screen">
    <!-- القائمة الجانبية -->
    <aside class="sidebar sidebar-expanded hidden md:block">
      <div class="p-6">
        <div class="logo-container flex items-center justify-center mb-8 p-2">
          <img src="<?php echo $company_logo; ?>" alt="مجموعة ابراهيم الاحمدي اليمنية" class="h-40 w-auto object-contain logo-animation">
        </div>

        <div class="user-info mb-8">
          <div class="flex items-center space-x-4 space-x-reverse mb-3">
            <div class="user-avatar">
              <?php echo mb_substr($user['full_name'], 0, 1, 'UTF-8'); ?>
            </div>
            <div>
              <h3 class="text-white font-semibold"><?php echo htmlspecialchars($user['full_name']); ?></h3>
              <p class="text-indigo-200 text-sm"><?php echo $isAdmin ? 'مدير النظام' : 'مستخدم'; ?></p>
            </div>
          </div>
          <div class="bg-white/10 h-0.5 rounded-full mb-3"></div>
        </div>

        <nav>
          <ul class="space-y-3">
            <!-- لوحة التحكم -->
            <li class="nav-item">
              <a href="unified-dashboard.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white transition-all">
                <i class="fas fa-tachometer-alt ml-3 text-lg"></i>
                <span>لوحة التحكم</span>
              </a>
            </li>

            <!-- إنشاء طلب جديد -->
            <li class="nav-item">
              <a href="create-order-unified.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white transition-all">
                <i class="fas fa-plus-circle ml-3 text-lg"></i>
                <span>إنشاء طلب جديد</span>
              </a>
            </li>

            <!-- قائمة الطلبات -->
            <li class="nav-item">
              <a href="orders.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white transition-all">
                <i class="fas fa-box ml-3 text-lg"></i>
                <span>قائمة الطلبات</span>
              </a>
            </li>

            <?php if ($isAdmin): ?>
            <!-- قائمة التقارير المنسدلة -->
            <li class="nav-item">
              <a href="#" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white dropdown-toggle transition-all" id="reportsDropdown">
                <i class="fas fa-chart-bar ml-3 text-lg"></i>
                <span>التقارير</span>
                <i class="fas fa-chevron-down mr-auto text-xs transition-transform duration-300"></i>
              </a>
              <ul class="dropdown-menu rounded-lg bg-indigo-900/50 mt-2 p-2 space-y-1 hidden" id="reportsMenu">
                <li>
                  <a href="weekly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white hover:bg-indigo-700/50 transition-all">
                    <i class="fas fa-calendar-week ml-2 text-sm"></i>
                    <span>التقارير الأسبوعية</span>
                  </a>
                </li>
                <li>
                  <a href="monthly-reports.php" class="flex items-center p-2 rounded-lg text-white/80 hover:text-white hover:bg-indigo-700/50 transition-all">
                    <i class="fas fa-calendar-alt ml-2 text-sm"></i>
                    <span>التقارير الشهرية</span>
                  </a>
                </li>
              </ul>
            </li>

            <!-- إدارة المستخدمين -->
            <li class="nav-item">
              <a href="users.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white transition-all">
                <i class="fas fa-users ml-3 text-lg"></i>
                <span>إدارة المستخدمين</span>
              </a>
            </li>

            <!-- الفروع -->
            <li class="nav-item active">
              <a href="branches.php" class="flex items-center p-3 rounded-lg text-white transition-all">
                <i class="fas fa-code-branch ml-3 text-lg"></i>
                <span>الفروع</span>
              </a>
            </li>

            <!-- إعدادات النظام -->
            <li class="nav-item">
              <a href="settings.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white transition-all">
                <i class="fas fa-cog ml-3 text-lg"></i>
                <span>إعدادات النظام</span>
              </a>
            </li>
            <?php endif; ?>
          </ul>

          <div class="pt-6 mt-6 border-t border-white/10">
            <a href="auth/logout.php" class="flex items-center p-3 rounded-lg text-white/80 hover:text-white transition-all">
              <i class="fas fa-sign-out-alt ml-3 text-lg"></i>
              <span>تسجيل الخروج</span>
            </a>
          </div>
        </nav>
      </div>
    </aside>

    <!-- المحتوى الرئيسي -->
    <main class="flex-1 overflow-x-hidden overflow-y-auto bg-gray-50">
      <!-- شريط التنقل العلوي -->
      <header class="bg-white shadow-md">
        <div class="flex items-center justify-between p-4">
          <!-- زر القائمة للأجهزة المحمولة -->
          <button id="menuBtn" class="md:hidden text-gray-500 hover:text-gray-600 focus:outline-none">
            <i class="fas fa-bars text-xl"></i>
          </button>

          <!-- عنوان الصفحة -->
          <div class="flex items-center">
            <h1 class="text-xl font-bold bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">
              إدارة الفروع
            </h1>
          </div>

          <!-- زر إضافة فرع جديد -->
          <button id="addBranchBtn" class="btn btn-primary">
            <i class="fas fa-plus-circle ml-2"></i>
            <span>إضافة فرع</span>
          </button>
        </div>
      </header>

      <!-- محتوى الصفحة -->
      <div class="p-6">
        <?php if (!empty($message)): ?>
        <div class="mb-6 p-4 rounded-lg shadow-sm border-r-4 <?php echo $messageType === 'success' ? 'bg-green-50 border-green-500 text-green-800' : 'bg-red-50 border-red-500 text-red-800'; ?> animate-fadeIn">
          <div class="flex items-center">
            <i class="fas <?php echo $messageType === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'; ?> ml-2 text-xl"></i>
            <span class="font-medium"><?php echo $message; ?></span>
          </div>
        </div>
        <?php endif; ?>

        <!-- عنوان القسم -->
        <div class="page-header mb-6 animate-fadeIn">
          <i class="fas fa-chart-pie text-2xl text-primary"></i>
          <h2 class="text-2xl font-bold bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">لوحة إحصائيات الفروع</h2>
        </div>

        <!-- وصف اللوحة -->
        <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-200 mb-6 animate-fadeIn" style="animation-delay: 0.05s;">
          <div class="flex items-start">
            <div class="flex-shrink-0 bg-primary-50 p-2 rounded-lg ml-3">
              <i class="fas fa-info-circle text-primary text-xl"></i>
            </div>
            <div>
              <h3 class="font-bold text-gray-800 mb-1">نظرة عامة على الفروع</h3>
              <p class="text-gray-600 text-sm">تعرض هذه اللوحة إحصائيات مهمة عن فروع مجموعة ابراهيم الاحمدي اليمنية، بما في ذلك عدد الفروع النشطة وإجمالي الموظفين والطلبات.</p>
            </div>
          </div>
        </div>

        <!-- بطاقات إحصائيات الفروع -->
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <!-- إجمالي الفروع -->
          <div class="stat-card relative overflow-hidden animate-fadeIn group" style="animation-delay: 0.1s;">
            <!-- زخرفة خلفية -->
            <div class="absolute -left-10 -top-10 w-40 h-40 bg-primary opacity-5 rounded-full group-hover:scale-150 transition-all duration-500"></div>

            <div class="relative z-10">
              <!-- رأس البطاقة -->
              <div class="flex items-center justify-between mb-4">
                <div class="icon-container">
                  <i class="fas fa-code-branch"></i>
                </div>
                <div class="text-xs font-medium text-gray-500 bg-gray-100 px-2 py-1 rounded-full">
                  إجمالي
                </div>
              </div>

              <!-- محتوى البطاقة -->
              <div class="stat-value flex items-baseline">
                <?php echo count($branches); ?>
                <span class="text-sm font-medium text-gray-500 mr-1">فرع</span>
              </div>
              <div class="stat-label">إجمالي الفروع</div>

              <!-- معلومات إضافية -->
              <div class="mt-4 pt-3 border-t border-gray-100 text-xs text-gray-500">
                <span class="flex items-center">
                  <i class="fas fa-info-circle ml-1 text-primary"></i>
                  العدد الكلي للفروع في النظام
                </span>
              </div>
            </div>
          </div>

          <!-- الفروع النشطة -->
          <div class="stat-card relative overflow-hidden animate-fadeIn group" style="animation-delay: 0.2s;">
            <!-- زخرفة خلفية -->
            <div class="absolute -left-10 -top-10 w-40 h-40 bg-green-500 opacity-5 rounded-full group-hover:scale-150 transition-all duration-500"></div>

            <div class="relative z-10">
              <!-- رأس البطاقة -->
              <div class="flex items-center justify-between mb-4">
                <div class="icon-container" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                  <i class="fas fa-check-circle"></i>
                </div>
                <div class="text-xs font-medium text-green-700 bg-green-100 px-2 py-1 rounded-full">
                  نشط
                </div>
              </div>

              <!-- محتوى البطاقة -->
              <div class="stat-value flex items-baseline text-green-600">
                <?php echo count(array_filter($branches, function($branch) { return $branch['status'] === 'نشط'; })); ?>
                <span class="text-sm font-medium text-gray-500 mr-1">فرع</span>
              </div>
              <div class="stat-label">الفروع النشطة</div>

              <!-- معلومات إضافية -->
              <div class="mt-4 pt-3 border-t border-gray-100 text-xs text-gray-500">
                <span class="flex items-center">
                  <i class="fas fa-info-circle ml-1 text-green-500"></i>
                  الفروع التي تعمل حالياً
                </span>
              </div>
            </div>
          </div>

          <!-- إجمالي الموظفين -->
          <div class="stat-card relative overflow-hidden animate-fadeIn group" style="animation-delay: 0.3s;">
            <!-- زخرفة خلفية -->
            <div class="absolute -left-10 -top-10 w-40 h-40 bg-blue-500 opacity-5 rounded-full group-hover:scale-150 transition-all duration-500"></div>

            <div class="relative z-10">
              <!-- رأس البطاقة -->
              <div class="flex items-center justify-between mb-4">
                <div class="icon-container" style="background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);">
                  <i class="fas fa-users"></i>
                </div>
                <div class="text-xs font-medium text-blue-700 bg-blue-100 px-2 py-1 rounded-full">
                  الموظفين
                </div>
              </div>

              <!-- محتوى البطاقة -->
              <div class="stat-value flex items-baseline text-blue-600">
                <?php echo array_sum(array_column($branches, 'employees_count')); ?>
                <span class="text-sm font-medium text-gray-500 mr-1">موظف</span>
              </div>
              <div class="stat-label">إجمالي الموظفين</div>

              <!-- معلومات إضافية -->
              <div class="mt-4 pt-3 border-t border-gray-100 text-xs text-gray-500">
                <span class="flex items-center">
                  <i class="fas fa-info-circle ml-1 text-blue-500"></i>
                  عدد الموظفين في جميع الفروع
                </span>
              </div>
            </div>
          </div>

          <!-- إجمالي الطلبات -->
          <div class="stat-card relative overflow-hidden animate-fadeIn group" style="animation-delay: 0.4s;">
            <!-- زخرفة خلفية -->
            <div class="absolute -left-10 -top-10 w-40 h-40 bg-amber-500 opacity-5 rounded-full group-hover:scale-150 transition-all duration-500"></div>

            <div class="relative z-10">
              <!-- رأس البطاقة -->
              <div class="flex items-center justify-between mb-4">
                <div class="icon-container" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
                  <i class="fas fa-box"></i>
                </div>
                <div class="text-xs font-medium text-amber-700 bg-amber-100 px-2 py-1 rounded-full">
                  الطلبات
                </div>
              </div>

              <!-- محتوى البطاقة -->
              <div class="stat-value flex items-baseline text-amber-600">
                <?php echo array_sum(array_column($branches, 'orders_count')); ?>
                <span class="text-sm font-medium text-gray-500 mr-1">طلب</span>
              </div>
              <div class="stat-label">إجمالي الطلبات</div>

              <!-- معلومات إضافية -->
              <div class="mt-4 pt-3 border-t border-gray-100 text-xs text-gray-500">
                <span class="flex items-center">
                  <i class="fas fa-info-circle ml-1 text-amber-500"></i>
                  عدد الطلبات المسجلة من جميع الفروع
                </span>
              </div>
            </div>
          </div>
        </div>

        <!-- شريط التقدم -->
        <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-200 mb-8 animate-fadeIn" style="animation-delay: 0.5s;">
          <div class="flex items-center justify-between mb-2">
            <h3 class="font-bold text-gray-800">نسبة الفروع النشطة</h3>
            <span class="text-sm font-medium text-primary">
              <?php
                $activeCount = count(array_filter($branches, function($branch) { return isset($branch['status']) && $branch['status'] === 'نشط'; }));
                $totalCount = count($branches);
                $percentage = $totalCount > 0 ? round(($activeCount / $totalCount) * 100) : 0;
                echo $percentage . '%';
              ?>
            </span>
          </div>
          <div class="w-full bg-gray-200 rounded-full h-2.5">
            <div class="bg-primary h-2.5 rounded-full" style="width: <?php echo $percentage; ?>%"></div>
          </div>
          <div class="flex justify-between mt-1 text-xs text-gray-500">
            <span>0%</span>
            <span>50%</span>
            <span>100%</span>
          </div>
        </div>

        <!-- مخططات توزيع الفروع -->
        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
          <!-- مخطط توزيع الفروع حسب الحالة -->
          <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-200 animate-fadeIn" style="animation-delay: 0.6s;">
            <h3 class="font-bold text-gray-800 mb-4 flex items-center">
              <i class="fas fa-chart-pie text-primary ml-2"></i>
              <span>توزيع الفروع حسب الحالة</span>
            </h3>
            <div class="flex items-center">
              <div class="w-1/2">
                <div id="statusChart" class="h-48"></div>
              </div>
              <div class="w-1/2 pr-4">
                <div class="space-y-3">
                  <div class="flex items-center">
                    <span class="w-3 h-3 rounded-full bg-primary ml-2"></span>
                    <span class="text-sm text-gray-700">نشط:
                      <?php echo count(array_filter($branches, function($branch) { return isset($branch['status']) && $branch['status'] === 'نشط'; })); ?> فرع
                    </span>
                  </div>
                  <div class="flex items-center">
                    <span class="w-3 h-3 rounded-full bg-red-500 ml-2"></span>
                    <span class="text-sm text-gray-700">غير نشط:
                      <?php echo count(array_filter($branches, function($branch) { return isset($branch['status']) && $branch['status'] === 'غير نشط'; })); ?> فرع
                    </span>
                  </div>
                </div>
              </div>
            </div>
          </div>

          <!-- توزيع الموظفين حسب الفروع -->
          <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-200 animate-fadeIn" style="animation-delay: 0.7s;">
            <h3 class="font-bold text-gray-800 mb-4 flex items-center">
              <i class="fas fa-users text-blue-500 ml-2"></i>
              <span>أكثر الفروع من حيث عدد الموظفين</span>
            </h3>
            <div class="space-y-3">
              <?php
                // ترتيب الفروع حسب عدد الموظفين
                $sortedBranches = $branches;
                usort($sortedBranches, function($a, $b) {
                  return $b['employees_count'] - $a['employees_count'];
                });

                // عرض أعلى 4 فروع
                $topBranches = array_slice($sortedBranches, 0, 4);
                $maxEmployees = count($topBranches) > 0 ? $topBranches[0]['employees_count'] : 0;

                foreach($topBranches as $branch):
                  $percentage = $maxEmployees > 0 ? ($branch['employees_count'] / $maxEmployees) * 100 : 0;
              ?>
              <div>
                <div class="flex items-center justify-between mb-1">
                  <span class="text-sm font-medium text-gray-700"><?php echo $branch['name']; ?></span>
                  <span class="text-sm font-medium text-blue-600"><?php echo $branch['employees_count']; ?> موظف</span>
                </div>
                <div class="w-full bg-gray-200 rounded-full h-2">
                  <div class="bg-blue-500 h-2 rounded-full" style="width: <?php echo $percentage; ?>%"></div>
                </div>
              </div>
              <?php endforeach; ?>
            </div>
          </div>
        </div>

        <!-- عنوان القسم -->
        <div class="page-header mb-6 animate-fadeIn" style="animation-delay: 0.8s;">
          <i class="fas fa-list"></i>
          <h2 class="text-2xl font-bold text-gray-800">قائمة الفروع</h2>
        </div>

        <!-- قائمة الفروع -->
        <div class="chart-container animate-fadeIn" style="animation-delay: 0.6s;">
          <!-- فلترة الفروع -->
          <div class="bg-white p-4 rounded-lg mb-6 border border-gray-200 shadow-sm">
            <div class="flex flex-wrap items-center justify-between gap-4">
              <div class="flex items-center">
                <i class="fas fa-filter text-primary ml-2"></i>
                <span class="font-medium text-gray-700">تصفية الفروع:</span>
              </div>

              <div class="flex flex-wrap gap-4">
                <!-- فلتر المنطقة -->
                <div class="flex items-center">
                  <label for="regionFilter" class="ml-2 text-sm font-medium text-gray-600">المنطقة:</label>
                  <select id="regionFilter" class="form-select">
                    <option value="">كل المناطق</option>
                    <?php foreach ($regions as $region): ?>
                    <option value="<?php echo $region['name']; ?>"><?php echo $region['name']; ?></option>
                    <?php endforeach; ?>
                  </select>
                </div>

                <!-- فلتر الحالة -->
                <div class="flex items-center">
                  <label for="statusFilter" class="ml-2 text-sm font-medium text-gray-600">الحالة:</label>
                  <select id="statusFilter" class="form-select">
                    <option value="">كل الحالات</option>
                    <option value="نشط">نشط</option>
                    <option value="غير نشط">غير نشط</option>
                  </select>
                </div>

                <!-- زر التصفية -->
                <button id="filterBtn" class="btn btn-primary">
                  <i class="fas fa-search ml-1"></i>
                  <span>تصفية</span>
                </button>
              </div>
            </div>
          </div>

          <!-- جدول الفروع -->
          <div class="overflow-x-auto">
            <table class="data-table">
              <thead>
                <tr>
                  <th>المعرف</th>
                  <th>الرمز</th>
                  <th>اسم الفرع</th>
                  <th>المدير</th>
                  <th>العنوان</th>
                  <th>رقم الهاتف</th>
                  <th>الموظفين</th>
                  <th>الطلبات</th>
                  <th>الحالة</th>
                  <th>الإجراءات</th>
                </tr>
              </thead>
              <tbody>
                <?php if (empty($branches)): ?>
                <tr>
                  <td colspan="10" class="text-center py-12">
                    <div class="flex flex-col items-center justify-center text-gray-500">
                      <div class="w-20 h-20 rounded-full bg-gray-100 flex items-center justify-center mb-4">
                        <i class="fas fa-code-branch text-4xl text-gray-300"></i>
                      </div>
                      <p class="text-lg font-medium">لا يوجد فروع</p>
                      <p class="text-sm text-gray-400 mt-1">قم بإضافة فرع جديد للبدء</p>
                    </div>
                  </td>
                </tr>
                <?php else: ?>
                  <?php foreach ($branches as $branch): ?>
                  <tr>
                    <td class="font-medium"><?php echo $branch['id']; ?></td>
                    <td class="font-mono font-bold text-primary"><?php echo $branch['code']; ?></td>
                    <td>
                      <div class="flex items-center">
                        <div class="w-10 h-10 rounded-lg bg-primary-100 flex items-center justify-center text-primary font-semibold ml-3 shadow-sm">
                          <i class="fas fa-store-alt"></i>
                        </div>
                        <span class="font-medium"><?php echo $branch['name']; ?></span>
                      </div>
                    </td>
                    <td><?php echo $branch['manager']; ?></td>
                    <td><?php echo $branch['address']; ?></td>
                    <td dir="ltr" class="text-right font-medium"><?php echo formatPhoneNumber($branch['phone']); ?></td>
                    <td>
                      <div class="flex items-center">
                        <span class="bg-blue-100 text-blue-800 text-xs font-medium px-2.5 py-0.5 rounded-full">
                          <?php echo $branch['employees_count']; ?> موظف
                        </span>
                      </div>
                    </td>
                    <td>
                      <div class="flex items-center">
                        <span class="bg-amber-100 text-amber-800 text-xs font-medium px-2.5 py-0.5 rounded-full">
                          <?php echo $branch['orders_count']; ?> طلب
                        </span>
                      </div>
                    </td>
                    <td>
                      <span class="status-badge <?php echo isset($branch['status']) && $branch['status'] === 'نشط' ? 'status-active' : 'status-inactive'; ?>">
                        <i class="fas fa-<?php echo isset($branch['status']) && $branch['status'] === 'نشط' ? 'check-circle' : 'times-circle'; ?> ml-1"></i>
                        <?php echo isset($branch['status']) ? $branch['status'] : 'غير محدد'; ?>
                      </span>
                    </td>
                    <td>
                      <div class="flex items-center">
                        <button class="edit-branch-btn action-btn edit-btn" title="تعديل الفرع" data-id="<?php echo $branch['id']; ?>">
                          <i class="fas fa-edit"></i>
                        </button>
                        <button class="view-branch-btn action-btn view-btn" title="عرض تفاصيل الفرع" data-id="<?php echo $branch['id']; ?>">
                          <i class="fas fa-eye"></i>
                        </button>
                        <button class="delete-branch-btn action-btn delete-btn" title="حذف الفرع" data-id="<?php echo $branch['id']; ?>">
                          <i class="fas fa-trash-alt"></i>
                        </button>
                      </div>
                    </td>
                  </tr>
                  <?php endforeach; ?>
                <?php endif; ?>
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </main>
  </div>

  <!-- نموذج إضافة/تعديل فرع -->
  <div id="branchModal" class="modal">
    <div class="modal-content">
      <div class="p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-xl font-bold text-gray-800 flex items-center" id="modalTitle">
            <i class="fas fa-plus-circle text-primary ml-2"></i>
            <span>إضافة فرع جديد</span>
          </h3>
          <button id="closeModal" class="text-gray-400 hover:text-gray-600 transition-all hover:rotate-90">
            <i class="fas fa-times text-xl"></i>
          </button>
        </div>

        <form id="branchForm" method="post">
          <input type="hidden" name="action" id="formAction" value="add">
          <input type="hidden" name="branch_id" id="branchId" value="">

          <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
            <div class="form-group">
              <label for="branchCode" class="form-label">
                <i class="fas fa-hashtag text-primary ml-1"></i>
                رمز الفرع
              </label>
              <input type="text" id="branchCode" name="code" class="form-input" placeholder="مثال: HQ" required>
              <p class="text-xs text-gray-500 mt-1">رمز فريد للفرع (3-5 أحرف)</p>
            </div>

            <div class="form-group">
              <label for="branchName" class="form-label">
                <i class="fas fa-store-alt text-primary ml-1"></i>
                اسم الفرع
              </label>
              <input type="text" id="branchName" name="name" class="form-input" placeholder="مثال: الفرع الرئيسي" required>
            </div>
          </div>

          <div class="form-group">
            <label for="manager" class="form-label">
              <i class="fas fa-user-tie text-primary ml-1"></i>
              مدير الفرع
            </label>
            <input type="text" id="manager" name="manager" class="form-input" placeholder="مثال: أحمد محمد" required>
          </div>

          <div class="form-group">
            <label for="region" class="form-label">
              <i class="fas fa-map-marker-alt text-primary ml-1"></i>
              المنطقة
            </label>
            <select id="region" name="region" class="form-select" required>
              <?php foreach ($regions as $region): ?>
              <option value="<?php echo $region['id']; ?>"><?php echo $region['name']; ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="form-group">
            <label for="address" class="form-label">
              <i class="fas fa-map-signs text-primary ml-1"></i>
              العنوان
            </label>
            <input type="text" id="address" name="address" class="form-input" placeholder="مثال: شارع الملك فهد، الرياض" required>
          </div>

          <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div class="form-group">
              <label for="phone" class="form-label">
                <i class="fas fa-phone-alt text-primary ml-1"></i>
                رقم الهاتف
              </label>
              <input type="tel" id="phone" name="phone" class="form-input" placeholder="مثال: 0112345678" required>
            </div>

            <div class="form-group">
              <label for="email" class="form-label">
                <i class="fas fa-envelope text-primary ml-1"></i>
                البريد الإلكتروني
              </label>
              <input type="email" id="email" name="email" class="form-input" placeholder="مثال: branch@example.com" required>
            </div>
          </div>

          <div class="form-group">
            <label for="status" class="form-label">
              <i class="fas fa-toggle-on text-primary ml-1"></i>
              الحالة
            </label>
            <select id="status" name="status" class="form-select" required>
              <option value="نشط">نشط</option>
              <option value="غير نشط">غير نشط</option>
            </select>
          </div>

          <div class="flex justify-end mt-8 border-t pt-4">
            <button type="button" id="cancelBtn" class="btn btn-secondary ml-3">
              <i class="fas fa-times ml-1"></i>
              <span>إلغاء</span>
            </button>
            <button type="submit" class="btn btn-primary">
              <i class="fas fa-save ml-1"></i>
              <span>حفظ</span>
            </button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- نموذج عرض تفاصيل الفرع -->
  <div id="viewBranchModal" class="modal">
    <div class="modal-content">
      <div class="p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-xl font-bold text-gray-800 flex items-center">
            <i class="fas fa-info-circle text-primary ml-2"></i>
            <span>تفاصيل الفرع</span>
          </h3>
          <button id="closeViewModal" class="text-gray-400 hover:text-gray-600 transition-all hover:rotate-90">
            <i class="fas fa-times text-xl"></i>
          </button>
        </div>

        <div id="branchDetails" class="space-y-4">
          <!-- سيتم ملء هذا القسم بالبيانات عبر JavaScript -->
        </div>

        <div class="flex justify-end mt-8 border-t pt-4">
          <button type="button" id="closeViewBtn" class="btn btn-primary">
            <i class="fas fa-check ml-1"></i>
            <span>إغلاق</span>
          </button>
        </div>
      </div>
    </div>
  </div>

  <!-- نموذج حذف فرع -->
  <div id="deleteModal" class="modal">
    <div class="modal-content">
      <div class="p-6">
        <div class="flex items-center justify-between mb-6">
          <h3 class="text-xl font-bold text-gray-800 flex items-center">
            <i class="fas fa-exclamation-triangle text-danger ml-2"></i>
            <span>حذف فرع</span>
          </h3>
          <button id="closeDeleteModal" class="text-gray-400 hover:text-gray-600 transition-all hover:rotate-90">
            <i class="fas fa-times text-xl"></i>
          </button>
        </div>

        <div class="bg-red-50 p-4 rounded-lg border-r-4 border-red-500 mb-6">
          <div class="flex">
            <div class="flex-shrink-0">
              <i class="fas fa-exclamation-circle text-red-500 text-xl"></i>
            </div>
            <div class="mr-3">
              <p class="text-red-800 font-medium">تحذير!</p>
              <p class="text-red-700 text-sm mt-1">هل أنت متأكد من رغبتك في حذف هذا الفرع؟ هذا الإجراء لا يمكن التراجع عنه.</p>
            </div>
          </div>
        </div>

        <form id="deleteForm" method="post">
          <input type="hidden" name="action" value="delete">
          <input type="hidden" name="branch_id" id="deleteBranchId" value="">

          <div class="flex justify-end mt-6">
            <button type="button" id="cancelDeleteBtn" class="btn btn-secondary ml-3">
              <i class="fas fa-times ml-1"></i>
              <span>إلغاء</span>
            </button>
            <button type="submit" class="btn btn-danger">
              <i class="fas fa-trash-alt ml-1"></i>
              <span>حذف</span>
            </button>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- سكريبت القائمة الجانبية للأجهزة المحمولة -->
  <script>
    // تبديل القائمة الجانبية للأجهزة المحمولة
    const menuBtn = document.getElementById('menuBtn');
    const sidebar = document.querySelector('aside');
    const overlay = document.getElementById('overlay');

    if (menuBtn && sidebar && overlay) {
      menuBtn.addEventListener('click', () => {
        sidebar.classList.toggle('hidden');
        sidebar.classList.toggle('fixed');
        sidebar.classList.toggle('inset-0');
        sidebar.classList.toggle('z-40');
        sidebar.classList.toggle('w-64');
        overlay.classList.toggle('hidden');
      });

      overlay.addEventListener('click', () => {
        sidebar.classList.add('hidden');
        sidebar.classList.remove('fixed', 'inset-0', 'z-40', 'w-64');
        overlay.classList.add('hidden');
      });
    }

    // تبديل القوائم المنسدلة
    const dropdownToggles = document.querySelectorAll('.dropdown-toggle');

    dropdownToggles.forEach(toggle => {
      toggle.addEventListener('click', (e) => {
        e.preventDefault();
        const menuId = toggle.getAttribute('id');
        const menu = document.getElementById(menuId.replace('Dropdown', 'Menu'));

        if (menu) {
          menu.classList.toggle('active');
        }
      });
    });

    // فلترة الفروع
    const filterBtn = document.getElementById('filterBtn');
    const regionFilter = document.getElementById('regionFilter');
    const statusFilter = document.getElementById('statusFilter');
    const branchRows = document.querySelectorAll('.data-table tbody tr');

    if (filterBtn && regionFilter && statusFilter) {
      filterBtn.addEventListener('click', () => {
        const selectedRegion = regionFilter.value;
        const selectedStatus = statusFilter.value;

        branchRows.forEach(row => {
          const address = row.querySelector('td:nth-child(5)').textContent;
          const status = row.querySelector('td:nth-child(9) span').textContent.trim();

          const regionMatch = !selectedRegion || address.includes(selectedRegion);
          const statusMatch = !selectedStatus || status === selectedStatus;

          if (regionMatch && statusMatch) {
            row.style.display = '';
          } else {
            row.style.display = 'none';
          }
        });
      });
    }

    // إدارة النماذج
    const branchModal = document.getElementById('branchModal');
    const viewBranchModal = document.getElementById('viewBranchModal');
    const deleteModal = document.getElementById('deleteModal');
    const addBranchBtn = document.getElementById('addBranchBtn');
    const closeModal = document.getElementById('closeModal');
    const cancelBtn = document.getElementById('cancelBtn');
    const closeViewModal = document.getElementById('closeViewModal');
    const closeViewBtn = document.getElementById('closeViewBtn');
    const closeDeleteModal = document.getElementById('closeDeleteModal');
    const cancelDeleteBtn = document.getElementById('cancelDeleteBtn');
    const editBranchBtns = document.querySelectorAll('.edit-branch-btn');
    const viewBranchBtns = document.querySelectorAll('.view-branch-btn');
    const deleteBranchBtns = document.querySelectorAll('.delete-branch-btn');
    const modalTitle = document.getElementById('modalTitle');
    const formAction = document.getElementById('formAction');
    const branchId = document.getElementById('branchId');
    const deleteBranchId = document.getElementById('deleteBranchId');
    const branchDetails = document.getElementById('branchDetails');

    // فتح نموذج إضافة فرع
    if (addBranchBtn && branchModal) {
      addBranchBtn.addEventListener('click', () => {
        modalTitle.textContent = 'إضافة فرع جديد';
        formAction.value = 'add';
        branchId.value = '';
        document.getElementById('branchForm').reset();
        branchModal.classList.add('active');
      });
    }

    // إغلاق نموذج الفرع
    if ((closeModal && branchModal) || (cancelBtn && branchModal)) {
      closeModal.addEventListener('click', () => {
        branchModal.classList.remove('active');
      });

      cancelBtn.addEventListener('click', () => {
        branchModal.classList.remove('active');
      });
    }

    // إغلاق نموذج عرض تفاصيل الفرع
    if ((closeViewModal && viewBranchModal) || (closeViewBtn && viewBranchModal)) {
      closeViewModal.addEventListener('click', () => {
        viewBranchModal.classList.remove('active');
      });

      closeViewBtn.addEventListener('click', () => {
        viewBranchModal.classList.remove('active');
      });
    }

    // إغلاق نموذج حذف الفرع
    if ((closeDeleteModal && deleteModal) || (cancelDeleteBtn && deleteModal)) {
      closeDeleteModal.addEventListener('click', () => {
        deleteModal.classList.remove('active');
      });

      cancelDeleteBtn.addEventListener('click', () => {
        deleteModal.classList.remove('active');
      });
    }

    // فتح نموذج تعديل فرع
    if (editBranchBtns.length > 0 && branchModal) {
      editBranchBtns.forEach(btn => {
        btn.addEventListener('click', () => {
          const id = btn.getAttribute('data-id');
          modalTitle.textContent = 'تعديل بيانات الفرع';
          formAction.value = 'edit';
          branchId.value = id;

          // هنا يمكن إضافة كود لجلب بيانات الفرع وملء النموذج

          branchModal.classList.add('active');
        });
      });
    }

    // فتح نموذج عرض تفاصيل الفرع
    if (viewBranchBtns.length > 0 && viewBranchModal && branchDetails) {
      viewBranchBtns.forEach(btn => {
        btn.addEventListener('click', () => {
          const id = btn.getAttribute('data-id');

          // البحث عن الفرع بالمعرف
          const branch = <?php echo json_encode($branches); ?>.find(b => b.id == id);

          if (branch) {
            // ملء تفاصيل الفرع
            branchDetails.innerHTML = `
              <div class="bg-gray-50 p-4 rounded-lg">
                <div class="flex items-center mb-4">
                  <div class="w-12 h-12 rounded-full bg-info-100 flex items-center justify-center text-info font-semibold ml-4">
                    <i class="fas fa-store-alt"></i>
                  </div>
                  <div>
                    <h4 class="text-lg font-semibold">${branch.name}</h4>
                    <p class="text-gray-500">رمز الفرع: ${branch.code}</p>
                  </div>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                  <div>
                    <p class="text-gray-500 mb-1">المدير</p>
                    <p class="font-medium">${branch.manager}</p>
                  </div>
                  <div>
                    <p class="text-gray-500 mb-1">الحالة</p>
                    <span class="status-badge ${branch.status === 'نشط' ? 'status-active' : 'status-inactive'}">
                      <i class="fas fa-${branch.status === 'نشط' ? 'check-circle' : 'times-circle'} ml-1"></i>
                      ${branch.status}
                    </span>
                  </div>
                </div>

                <div class="mb-4">
                  <p class="text-gray-500 mb-1">العنوان</p>
                  <p class="font-medium">${branch.address}</p>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                  <div>
                    <p class="text-gray-500 mb-1">رقم الهاتف</p>
                    <p class="font-medium" dir="ltr">${formatPhoneNumber(branch.phone)}</p>
                  </div>
                  <div>
                    <p class="text-gray-500 mb-1">البريد الإلكتروني</p>
                    <p class="font-medium">${branch.email}</p>
                  </div>
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <p class="text-gray-500 mb-1">عدد الموظفين</p>
                    <p class="font-medium">${branch.employees_count} موظف</p>
                  </div>
                  <div>
                    <p class="text-gray-500 mb-1">عدد الطلبات</p>
                    <p class="font-medium">${branch.orders_count} طلب</p>
                  </div>
                </div>
              </div>
            `;
          }

          viewBranchModal.classList.add('active');
        });
      });
    }

    // فتح نموذج حذف فرع
    if (deleteBranchBtns.length > 0 && deleteModal) {
      deleteBranchBtns.forEach(btn => {
        btn.addEventListener('click', () => {
          const id = btn.getAttribute('data-id');
          deleteBranchId.value = id;
          deleteModal.classList.add('active');
        });
      });
    }

    // دالة مساعدة لتنسيق رقم الهاتف
    function formatPhoneNumber(phone) {
      // إزالة أي أحرف غير رقمية
      phone = phone.replace(/[^0-9]/g, '');

      // تنسيق الرقم بالشكل المطلوب (مثال: 011-234-5678)
      if (phone.length === 10) {
        return phone.substring(0, 3) + '-' + phone.substring(3, 6) + '-' + phone.substring(6);
      }

      return phone;
    }
  </script>

  <!-- سكريبت المخططات البيانية -->
  <script>
    document.addEventListener('DOMContentLoaded', function() {
      // بيانات المخطط الدائري لتوزيع الفروع حسب الحالة
      const activeCount = <?php echo count(array_filter($branches, function($branch) { return isset($branch['status']) && $branch['status'] === 'نشط'; })); ?>;
      const inactiveCount = <?php echo count(array_filter($branches, function($branch) { return isset($branch['status']) && $branch['status'] === 'غير نشط'; })); ?>;

      // إنشاء المخطط الدائري
      const statusChartEl = document.getElementById('statusChart');
      if (statusChartEl) {
        new Chart(statusChartEl, {
          type: 'doughnut',
          data: {
            labels: ['نشط', 'غير نشط'],
            datasets: [{
              data: [activeCount, inactiveCount],
              backgroundColor: [
                '#4F46E5', // لون الفروع النشطة
                '#EF4444'  // لون الفروع غير النشطة
              ],
              borderColor: [
                '#ffffff',
                '#ffffff'
              ],
              borderWidth: 2,
              hoverOffset: 4
            }]
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            cutout: '65%',
            plugins: {
              legend: {
                display: false
              },
              tooltip: {
                rtl: true,
                titleFont: {
                  family: 'Tajawal, sans-serif'
                },
                bodyFont: {
                  family: 'Tajawal, sans-serif'
                },
                callbacks: {
                  label: function(context) {
                    const total = activeCount + inactiveCount;
                    const percentage = Math.round((context.raw / total) * 100);
                    return `${context.label}: ${context.raw} (${percentage}%)`;
                  }
                }
              }
            }
          }
        });
      }

      // تأثيرات حركية للبطاقات الإحصائية
      const statCards = document.querySelectorAll('.stat-card');
      statCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
          this.querySelector('.icon-container').classList.add('animate-bounce');
          setTimeout(() => {
            this.querySelector('.icon-container').classList.remove('animate-bounce');
          }, 1000);
        });
      });
    });
  </script>
</body>
</html>
