<?php
/**
 * صفحة تعديل بيانات المستخدم
 * تستخدم هذه الصفحة لتعديل بيانات المستخدم
 */

// بدء الجلسة
session_start();

// تضمين ملف التكوين الجديد
require_once 'config/config_new.php';

// تضمين ملف الدوال المساعدة
require_once 'includes/functions.php';

// تضمين ملف الاتصال بقاعدة البيانات
require_once 'config/db_config.php';

// التحقق من حالة تسجيل الدخول
if (!isLoggedIn()) {
    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: login.php');
    exit;
}

// التحقق من صلاحيات المستخدم (يجب أن يكون مسؤولاً أو يقوم بتعديل بياناته الشخصية)
if ($_SESSION['user_role'] !== 'admin' && (!isset($_GET['id']) || $_GET['id'] != $_SESSION['user_id'])) {
    // إعادة توجيه المستخدم إلى لوحة التحكم
    header('Location: dashboard.php');
    exit;
}

// التحقق من وجود معرف المستخدم
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $_SESSION['user_message'] = 'معرف المستخدم غير صحيح.';
    $_SESSION['user_message_type'] = 'error';
    header('Location: manage_users.php');
    exit;
}

$userId = $_GET['id'];

// الحصول على بيانات المستخدم
try {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    if (!$user) {
        $_SESSION['user_message'] = 'المستخدم غير موجود.';
        $_SESSION['user_message_type'] = 'error';
        header('Location: manage_users.php');
        exit;
    }
} catch (PDOException $e) {
    $_SESSION['user_message'] = 'حدث خطأ أثناء استرداد بيانات المستخدم: ' . $e->getMessage();
    $_SESSION['user_message_type'] = 'error';
    header('Location: manage_users.php');
    exit;
}

// معالجة النموذج عند الإرسال
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // التحقق من رمز CSRF
    if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
        $_SESSION['user_message'] = 'خطأ في التحقق من صحة النموذج. يرجى المحاولة مرة أخرى.';
        $_SESSION['user_message_type'] = 'error';
        header('Location: edit_user.php?id=' . $userId);
        exit;
    }
    
    // الحصول على بيانات النموذج
    $fullname = sanitizeInput($_POST['fullname'] ?? '');
    $email = sanitizeInput($_POST['email'] ?? '');
    $phone = sanitizeInput($_POST['phone'] ?? '');
    $role = sanitizeInput($_POST['role'] ?? $user['role']);
    $status = sanitizeInput($_POST['status'] ?? $user['status']);
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    // التحقق من البيانات
    $errors = [];
    
    if (empty($fullname)) {
        $errors[] = 'الاسم الكامل مطلوب.';
    }
    
    if (empty($email)) {
        $errors[] = 'البريد الإلكتروني مطلوب.';
    } elseif (!isValidEmail($email)) {
        $errors[] = 'البريد الإلكتروني غير صحيح.';
    }
    
    // التحقق من تفرد البريد الإلكتروني
    if ($email !== $user['email']) {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
        $stmt->execute([$email, $userId]);
        if ($stmt->rowCount() > 0) {
            $errors[] = 'البريد الإلكتروني مستخدم بالفعل.';
        }
    }
    
    // التحقق من كلمة المرور إذا تم تغييرها
    if (!empty($password)) {
        if (strlen($password) < 6) {
            $errors[] = 'يجب أن تكون كلمة المرور 6 أحرف على الأقل.';
        } elseif ($password !== $confirm_password) {
            $errors[] = 'كلمة المرور وتأكيد كلمة المرور غير متطابقين.';
        }
    }
    
    // إذا لم تكن هناك أخطاء، قم بتحديث بيانات المستخدم
    if (empty($errors)) {
        try {
            // إعداد استعلام التحديث
            $sql = "UPDATE users SET fullname = ?, email = ?, phone = ?";
            $params = [$fullname, $email, $phone];
            
            // إضافة حقول إضافية إذا كان المستخدم مسؤولاً
            if ($_SESSION['user_role'] === 'admin') {
                $sql .= ", role = ?, status = ?";
                $params[] = $role;
                $params[] = $status;
            }
            
            // إضافة كلمة المرور إذا تم تغييرها
            if (!empty($password)) {
                $sql .= ", password = ?";
                $params[] = password_hash($password, PASSWORD_DEFAULT);
            }
            
            // إضافة شرط معرف المستخدم
            $sql .= " WHERE id = ?";
            $params[] = $userId;
            
            // تنفيذ الاستعلام
            $stmt = $pdo->prepare($sql);
            $result = $stmt->execute($params);
            
            if ($result) {
                $_SESSION['user_message'] = 'تم تحديث بيانات المستخدم بنجاح.';
                $_SESSION['user_message_type'] = 'success';
                
                // إذا كان المستخدم يقوم بتعديل بياناته الشخصية، قم بتحديث بيانات الجلسة
                if ($userId == $_SESSION['user_id']) {
                    $_SESSION['user_name'] = $fullname;
                    $_SESSION['user_email'] = $email;
                }
                
                // إعادة توجيه المستخدم
                if ($_SESSION['user_role'] === 'admin') {
                    header('Location: manage_users.php');
                } else {
                    header('Location: dashboard.php');
                }
                exit;
            } else {
                $errors[] = 'فشل تحديث بيانات المستخدم.';
            }
        } catch (PDOException $e) {
            $errors[] = 'حدث خطأ أثناء تحديث بيانات المستخدم: ' . $e->getMessage();
        }
    }
}

// عنوان الصفحة
$pageTitle = 'تعديل بيانات المستخدم';

// تضمين ملف الرأس
include 'includes/header.php';
?>

<div class="container mt-4 rtl">
    <div class="row">
        <div class="col-md-8 mx-auto">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">تعديل بيانات المستخدم</h5>
                </div>
                <div class="card-body">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo $error; ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>
                    
                    <form method="post" action="edit_user.php?id=<?php echo $userId; ?>">
                        <!-- رمز CSRF للحماية من هجمات CSRF -->
                        <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
                        
                        <div class="form-group">
                            <label for="username">اسم المستخدم</label>
                            <input type="text" class="form-control" id="username" value="<?php echo htmlspecialchars($user['username']); ?>" readonly>
                            <small class="form-text text-muted">لا يمكن تغيير اسم المستخدم.</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="fullname">الاسم الكامل</label>
                            <input type="text" class="form-control" id="fullname" name="fullname" value="<?php echo htmlspecialchars($user['fullname']); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="email">البريد الإلكتروني</label>
                            <input type="email" class="form-control" id="email" name="email" value="<?php echo htmlspecialchars($user['email']); ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="phone">رقم الهاتف</label>
                            <input type="text" class="form-control" id="phone" name="phone" value="<?php echo htmlspecialchars($user['phone'] ?? ''); ?>">
                        </div>
                        
                        <?php if ($_SESSION['user_role'] === 'admin'): ?>
                            <div class="form-group">
                                <label for="role">الدور</label>
                                <select class="form-control" id="role" name="role">
                                    <option value="admin" <?php echo ($user['role'] === 'admin') ? 'selected' : ''; ?>>مسؤول</option>
                                    <option value="manager" <?php echo ($user['role'] === 'manager') ? 'selected' : ''; ?>>مدير</option>
                                    <option value="user" <?php echo ($user['role'] === 'user') ? 'selected' : ''; ?>>مستخدم</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="status">الحالة</label>
                                <select class="form-control" id="status" name="status">
                                    <option value="active" <?php echo ($user['status'] === 'active') ? 'selected' : ''; ?>>نشط</option>
                                    <option value="pending" <?php echo ($user['status'] === 'pending') ? 'selected' : ''; ?>>معلق</option>
                                </select>
                            </div>
                        <?php endif; ?>
                        
                        <div class="form-group">
                            <label for="password">كلمة المرور الجديدة</label>
                            <input type="password" class="form-control" id="password" name="password">
                            <small class="form-text text-muted">اتركها فارغة إذا كنت لا ترغب في تغيير كلمة المرور.</small>
                        </div>
                        
                        <div class="form-group">
                            <label for="confirm_password">تأكيد كلمة المرور الجديدة</label>
                            <input type="password" class="form-control" id="confirm_password" name="confirm_password">
                        </div>
                        
                        <div class="form-group">
                            <button type="submit" class="btn btn-primary">حفظ التغييرات</button>
                            <?php if ($_SESSION['user_role'] === 'admin'): ?>
                                <a href="manage_users.php" class="btn btn-secondary">إلغاء</a>
                            <?php else: ?>
                                <a href="dashboard.php" class="btn btn-secondary">إلغاء</a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
// تضمين ملف التذييل
include 'includes/footer.php';
?>
