<?php
/**
 * ملف التحقق من حالة تسجيل الدخول
 * يستخدم هذا الملف للتحقق من حالة تسجيل الدخول وتوجيه المستخدمين غير المسجلين إلى صفحة تسجيل الدخول
 */

// بدء الجلسة إذا لم تكن قد بدأت بالفعل
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * التحقق من حالة تسجيل الدخول
 * @return bool حالة تسجيل الدخول
 */
function isLoggedIn() {
    return isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
}

/**
 * التحقق من دور المستخدم
 * @param string $role الدور المطلوب
 * @return bool ما إذا كان المستخدم لديه الدور المطلوب
 */
function hasRole($role) {
    return isLoggedIn() && $_SESSION['role'] === $role;
}

/**
 * التحقق من أن المستخدم لديه أحد الأدوار المحددة
 * @param array $roles مصفوفة الأدوار المسموح بها
 * @return bool ما إذا كان المستخدم لديه أحد الأدوار المطلوبة
 */
function hasAnyRole($roles) {
    if (!isLoggedIn()) {
        return false;
    }

    return in_array($_SESSION['role'], $roles);
}

/**
 * توجيه المستخدم إلى صفحة تسجيل الدخول إذا لم يكن مسجل الدخول
 */
function requireLogin() {
    if (!isLoggedIn()) {
        // تخزين الصفحة الحالية للعودة إليها بعد تسجيل الدخول
        $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];

        // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
        header('Location: login.php');
        exit;
    }
}

/**
 * توجيه المستخدم إذا لم يكن لديه الدور المطلوب
 * @param string $role الدور المطلوب
 */
function requireRole($role) {
    requireLogin();

    if (!hasRole($role)) {
        // إعادة توجيه المستخدم إلى صفحة رفض الوصول
        header('Location: access_denied.html');
        exit;
    }
}

/**
 * توجيه المستخدم إذا لم يكن لديه أحد الأدوار المطلوبة
 * @param array $roles مصفوفة الأدوار المسموح بها
 */
function requireAnyRole($roles) {
    requireLogin();

    if (!hasAnyRole($roles)) {
        // إعادة توجيه المستخدم إلى صفحة رفض الوصول
        header('Location: access_denied.html');
        exit;
    }
}

/**
 * تسجيل الخروج
 */
function logout() {
    // حذف جميع متغيرات الجلسة
    $_SESSION = array();

    // حذف ملف تعريف الجلسة إذا كان موجود
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }

    // إنهاء الجلسة
    session_destroy();

    // إعادة توجيه المستخدم إلى صفحة تسجيل الدخول
    header('Location: login.php');
    exit;
}
?>
