/**
 * تأثيرات متقدمة لصفحة تسجيل الدخول
 * تأثيرات حديثة وتفاعلية لتحسين تجربة المستخدم
 */

document.addEventListener('DOMContentLoaded', function() {
    // إنشاء عناصر الخلفية المتحركة
    createBackgroundEffects();
    
    // تهيئة تأثيرات الحقول
    initializeInputEffects();
    
    // تهيئة تأثيرات الأزرار
    initializeButtonEffects();
    
    // تهيئة تأثيرات الحركة
    initializeMotionEffects();
    
    // تهيئة التحقق من صحة النموذج
    initializeFormValidation();
    
    // إضافة تأثير الماوس
    initializeMouseEffect();
});

/**
 * إنشاء تأثيرات الخلفية المتحركة
 */
function createBackgroundEffects() {
    // إنشاء حاوية الخلفية إذا لم تكن موجودة
    if (!document.querySelector('.background-animation')) {
        const backgroundAnimation = document.createElement('div');
        backgroundAnimation.classList.add('background-animation');
        
        const animatedGradient = document.createElement('div');
        animatedGradient.classList.add('animated-gradient');
        
        backgroundAnimation.appendChild(animatedGradient);
        document.body.insertBefore(backgroundAnimation, document.body.firstChild);
    }
    
    // إنشاء تأثير الدوائر المتحركة
    createFloatingCircles();
}

/**
 * إنشاء الدوائر العائمة في الخلفية
 */
function createFloatingCircles() {
    const backgroundAnimation = document.querySelector('.background-animation');
    if (!backgroundAnimation) return;
    
    // عدد الدوائر
    const circleCount = 6;
    
    // إنشاء الدوائر
    for (let i = 0; i < circleCount; i++) {
        const circle = document.createElement('div');
        circle.classList.add('floating-circle');
        
        // تعيين حجم عشوائي
        const size = Math.random() * 200 + 50;
        circle.style.width = `${size}px`;
        circle.style.height = `${size}px`;
        
        // تعيين موقع عشوائي
        const posX = Math.random() * 100;
        const posY = Math.random() * 100;
        circle.style.left = `${posX}%`;
        circle.style.top = `${posY}%`;
        
        // تعيين لون عشوائي
        const colors = [
            'rgba(99, 102, 241, 0.03)',
            'rgba(245, 158, 11, 0.03)',
            'rgba(16, 185, 129, 0.03)'
        ];
        const colorIndex = Math.floor(Math.random() * colors.length);
        circle.style.backgroundColor = colors[colorIndex];
        
        // تعيين مدة وتأخير عشوائي للحركة
        const duration = Math.random() * 20 + 10;
        const delay = Math.random() * 5;
        circle.style.animationDuration = `${duration}s`;
        circle.style.animationDelay = `${delay}s`;
        
        // إضافة الدائرة إلى الحاوية
        backgroundAnimation.appendChild(circle);
    }
    
    // إضافة أنماط CSS للدوائر العائمة
    if (!document.getElementById('floating-circles-style')) {
        const style = document.createElement('style');
        style.id = 'floating-circles-style';
        style.textContent = `
            .floating-circle {
                position: absolute;
                border-radius: 50%;
                opacity: 0.5;
                filter: blur(40px);
                animation: floatAnimation 20s infinite ease-in-out;
            }
            
            @keyframes floatAnimation {
                0%, 100% {
                    transform: translate(0, 0) scale(1);
                }
                25% {
                    transform: translate(-50px, 50px) scale(1.1);
                }
                50% {
                    transform: translate(50px, 100px) scale(0.9);
                }
                75% {
                    transform: translate(100px, -50px) scale(1.05);
                }
            }
        `;
        document.head.appendChild(style);
    }
}

/**
 * تهيئة تأثيرات الحقول
 */
function initializeInputEffects() {
    const inputs = document.querySelectorAll('.auth-input');
    
    inputs.forEach(input => {
        // تأثير التركيز
        input.addEventListener('focus', function() {
            this.parentElement.classList.add('input-focused');
            
            // تحريك الأيقونة
            const icon = this.nextElementSibling;
            if (icon && icon.classList.contains('input-icon')) {
                icon.style.transform = 'translateY(-50%) scale(1.2)';
                icon.style.color = 'var(--primary-light)';
            }
        });
        
        // إزالة تأثير التركيز
        input.addEventListener('blur', function() {
            this.parentElement.classList.remove('input-focused');
            
            // إعادة الأيقونة
            const icon = this.nextElementSibling;
            if (icon && icon.classList.contains('input-icon')) {
                icon.style.transform = 'translateY(-50%) scale(1)';
                icon.style.color = 'rgba(255, 255, 255, 0.5)';
            }
        });
        
        // تأثير الكتابة
        input.addEventListener('input', function() {
            if (this.value !== '') {
                this.parentElement.classList.add('has-text');
            } else {
                this.parentElement.classList.remove('has-text');
            }
        });
    });
}

/**
 * تهيئة تأثيرات الأزرار
 */
function initializeButtonEffects() {
    const buttons = document.querySelectorAll('.auth-button');
    
    buttons.forEach(button => {
        // تأثير التموج عند النقر
        button.addEventListener('click', function(e) {
            // إنشاء عنصر التموج إذا لم يكن موجودًا
            if (!document.querySelector('.ripple-style')) {
                const style = document.createElement('style');
                style.classList.add('ripple-style');
                style.textContent = `
                    .ripple {
                        position: absolute;
                        background: rgba(255, 255, 255, 0.3);
                        border-radius: 50%;
                        transform: scale(0);
                        animation: ripple 0.6s linear;
                        pointer-events: none;
                    }
                    
                    @keyframes ripple {
                        to {
                            transform: scale(4);
                            opacity: 0;
                        }
                    }
                `;
                document.head.appendChild(style);
            }
            
            // إنشاء عنصر التموج
            const ripple = document.createElement('span');
            ripple.classList.add('ripple');
            this.appendChild(ripple);
            
            // تحديد موقع النقر
            const rect = button.getBoundingClientRect();
            const x = e.clientX - rect.left;
            const y = e.clientY - rect.top;
            
            // تعيين موقع وحجم التموج
            ripple.style.left = `${x}px`;
            ripple.style.top = `${y}px`;
            
            // إزالة عنصر التموج بعد انتهاء التأثير
            setTimeout(() => {
                ripple.remove();
            }, 600);
        });
    });
}

/**
 * تهيئة تأثيرات الحركة
 */
function initializeMotionEffects() {
    // تأثير تحريك العناصر عند تحميل الصفحة
    const authContainer = document.querySelector('.auth-container');
    
    if (authContainer) {
        // إضافة تأثير الظهور التدريجي
        authContainer.style.opacity = '0';
        authContainer.style.transform = 'translateY(20px)';
        
        setTimeout(() => {
            authContainer.style.transition = 'opacity 0.8s ease, transform 0.8s ease';
            authContainer.style.opacity = '1';
            authContainer.style.transform = 'translateY(0)';
        }, 100);
    }
}

/**
 * تهيئة التحقق من صحة النموذج
 */
function initializeFormValidation() {
    const form = document.getElementById('loginForm');
    if (!form) return;
    
    form.addEventListener('submit', function(e) {
        const emailInput = form.querySelector('input[name="email"]');
        const passwordInput = form.querySelector('input[name="password"]');
        let isValid = true;
        
        // التحقق من البريد الإلكتروني
        if (emailInput && !validateEmail(emailInput.value)) {
            showInputError(emailInput, 'يرجى إدخال بريد إلكتروني صحيح');
            isValid = false;
        } else if (emailInput) {
            removeInputError(emailInput);
        }
        
        // التحقق من كلمة المرور
        if (passwordInput && passwordInput.value.length < 6) {
            showInputError(passwordInput, 'يجب أن تتكون كلمة المرور من 6 أحرف على الأقل');
            isValid = false;
        } else if (passwordInput) {
            removeInputError(passwordInput);
        }
        
        if (!isValid) {
            e.preventDefault();
        }
    });
}

/**
 * التحقق من صحة البريد الإلكتروني
 */
function validateEmail(email) {
    const re = /^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
    return re.test(String(email).toLowerCase());
}

/**
 * إظهار رسالة خطأ للحقل
 */
function showInputError(input, message) {
    const parent = input.parentElement;
    
    // إزالة رسالة خطأ سابقة إن وجدت
    removeInputError(input);
    
    // إضافة فئة الخطأ
    parent.classList.add('input-error');
    
    // إنشاء رسالة الخطأ
    const errorMessage = document.createElement('div');
    errorMessage.classList.add('input-error-message');
    errorMessage.textContent = message;
    
    // إضافة رسالة الخطأ
    parent.appendChild(errorMessage);
    
    // إضافة أنماط CSS لرسالة الخطأ إذا لم تكن موجودة
    if (!document.getElementById('error-message-style')) {
        const style = document.createElement('style');
        style.id = 'error-message-style';
        style.textContent = `
            .input-error .auth-input {
                border-color: var(--error-color);
                box-shadow: 0 0 0 1px var(--error-color);
            }
            
            .input-error-message {
                color: var(--error-color);
                font-size: 12px;
                margin-top: 5px;
                text-align: right;
            }
        `;
        document.head.appendChild(style);
    }
}

/**
 * إزالة رسالة خطأ من الحقل
 */
function removeInputError(input) {
    const parent = input.parentElement;
    
    // إزالة فئة الخطأ
    parent.classList.remove('input-error');
    
    // إزالة رسالة الخطأ إن وجدت
    const errorMessage = parent.querySelector('.input-error-message');
    if (errorMessage) {
        errorMessage.remove();
    }
}

/**
 * تهيئة تأثير الماوس
 */
function initializeMouseEffect() {
    const authContainer = document.querySelector('.auth-container');
    if (!authContainer) return;
    
    // تأثير حركة الحاوية مع حركة الماوس
    document.addEventListener('mousemove', function(e) {
        const x = e.clientX / window.innerWidth - 0.5;
        const y = e.clientY / window.innerHeight - 0.5;
        
        authContainer.style.transform = `perspective(1000px) rotateY(${x * 2}deg) rotateX(${y * -2}deg) translateZ(0)`;
    });
    
    // إعادة الحاوية إلى وضعها الطبيعي عند مغادرة الماوس
    document.addEventListener('mouseleave', function() {
        authContainer.style.transform = 'perspective(1000px) rotateY(0) rotateX(0) translateZ(0)';
    });
}
