/**
 * ملف JavaScript للتعامل مع طلبات التسجيل المعلقة
 */

document.addEventListener('DOMContentLoaded', function() {
    // تحميل طلبات التسجيل المعلقة عند تحميل الصفحة
    loadPendingUsers();

    // إضافة مستمع حدث لزر التحديث
    const refreshBtn = document.getElementById('refreshBtn');
    if (refreshBtn) {
        refreshBtn.addEventListener('click', loadPendingUsers);
    }

    // إضافة مستمع حدث للبحث
    const searchInput = document.getElementById('searchInput');
    if (searchInput) {
        searchInput.addEventListener('input', function() {
            filterUsers(this.value);
        });
    }
});

/**
 * تحميل طلبات التسجيل المعلقة من الخادم
 */
function loadPendingUsers() {
    // عرض مؤشر التحميل
    const pendingUsersTable = document.getElementById('pendingUsersTable');
    if (pendingUsersTable) {
        pendingUsersTable.innerHTML = `
            <tr>
                <td colspan="7" class="text-center py-8 text-gray-500">
                    <i class="fas fa-spinner fa-spin ml-2"></i>
                    جاري تحميل طلبات التسجيل...
                </td>
            </tr>
        `;
    }

    // إخفاء رسالة عدم وجود طلبات
    const noRequestsMessage = document.getElementById('noRequestsMessage');
    if (noRequestsMessage) {
        noRequestsMessage.classList.add('hidden');
    }

    // طلب البيانات من الخادم
    fetch('api/get_pending_users.php')
        .then(response => {
            if (!response.ok) {
                throw new Error(`خطأ في استجابة الخادم: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            if (data.status === 'success') {
                displayPendingUsers(data.users);
                updatePendingCount(data.users.length);
            } else {
                showStatusMessage('error', data.message || 'حدث خطأ أثناء تحميل طلبات التسجيل');
                pendingUsersTable.innerHTML = `
                    <tr>
                        <td colspan="7" class="text-center py-8 text-red-500">
                            <i class="fas fa-exclamation-circle ml-2"></i>
                            فشل في تحميل طلبات التسجيل
                        </td>
                    </tr>
                `;
            }
        })
        .catch(error => {
            console.error('خطأ في تحميل طلبات التسجيل:', error);
            showStatusMessage('error', 'حدث خطأ أثناء الاتصال بالخادم. يرجى المحاولة مرة أخرى.');
            if (pendingUsersTable) {
                pendingUsersTable.innerHTML = `
                    <tr>
                        <td colspan="7" class="text-center py-8 text-red-500">
                            <i class="fas fa-exclamation-circle ml-2"></i>
                            فشل في تحميل طلبات التسجيل: ${error.message}
                        </td>
                    </tr>
                `;
            }
        });
}

/**
 * عرض طلبات التسجيل المعلقة في الجدول
 * @param {Array} users - مصفوفة المستخدمين المعلقين
 */
function displayPendingUsers(users) {
    const pendingUsersTable = document.getElementById('pendingUsersTable');
    const noRequestsMessage = document.getElementById('noRequestsMessage');
    
    if (!pendingUsersTable) return;

    if (!users || users.length === 0) {
        // عرض رسالة عدم وجود طلبات
        pendingUsersTable.innerHTML = '';
        if (noRequestsMessage) {
            noRequestsMessage.classList.remove('hidden');
        }
        return;
    }

    // إخفاء رسالة عدم وجود طلبات
    if (noRequestsMessage) {
        noRequestsMessage.classList.add('hidden');
    }

    // إنشاء صفوف الجدول
    let tableHTML = '';
    users.forEach((user, index) => {
        const registrationDate = new Date(user.created_at).toLocaleDateString('ar-SA');
        
        tableHTML += `
            <tr data-user-id="${user.id}" class="user-row">
                <td class="text-center">${index + 1}</td>
                <td>${escapeHTML(user.full_name)}</td>
                <td>${escapeHTML(user.username)}</td>
                <td>${escapeHTML(user.email)}</td>
                <td>${escapeHTML(user.phone || '-')}</td>
                <td>${registrationDate}</td>
                <td>
                    <div class="flex space-x-2 space-x-reverse">
                        <button class="btn btn-approve" onclick="approveUser(${user.id})">
                            <i class="fas fa-check ml-1"></i>
                            قبول
                        </button>
                        <button class="btn btn-reject" onclick="rejectUser(${user.id})">
                            <i class="fas fa-times ml-1"></i>
                            رفض
                        </button>
                    </div>
                </td>
            </tr>
        `;
    });

    pendingUsersTable.innerHTML = tableHTML;
}

/**
 * تحديث عدد الطلبات المعلقة
 * @param {number} count - عدد الطلبات المعلقة
 */
function updatePendingCount(count) {
    const pendingCount = document.getElementById('pendingCount');
    if (pendingCount) {
        pendingCount.textContent = count + (count === 1 ? ' طلب معلق' : ' طلبات معلقة');
        
        // تغيير لون العداد حسب العدد
        if (count > 0) {
            pendingCount.classList.remove('bg-green-100', 'text-green-800');
            pendingCount.classList.add('bg-yellow-100', 'text-yellow-800');
        } else {
            pendingCount.classList.remove('bg-yellow-100', 'text-yellow-800');
            pendingCount.classList.add('bg-green-100', 'text-green-800');
        }
    }
}

/**
 * الموافقة على طلب تسجيل مستخدم
 * @param {number} userId - معرف المستخدم
 */
function approveUser(userId) {
    // تأكيد الإجراء
    Swal.fire({
        title: 'تأكيد الموافقة',
        text: 'هل أنت متأكد من الموافقة على طلب التسجيل هذا؟',
        icon: 'question',
        showCancelButton: true,
        confirmButtonColor: '#10b981',
        cancelButtonColor: '#6b7280',
        confirmButtonText: 'نعم، وافق',
        cancelButtonText: 'إلغاء'
    }).then((result) => {
        if (result.isConfirmed) {
            // إرسال طلب الموافقة إلى الخادم
            updateUserStatus(userId, 'approve');
        }
    });
}

/**
 * رفض طلب تسجيل مستخدم
 * @param {number} userId - معرف المستخدم
 */
function rejectUser(userId) {
    // تأكيد الإجراء
    Swal.fire({
        title: 'تأكيد الرفض',
        text: 'هل أنت متأكد من رفض طلب التسجيل هذا؟',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#ef4444',
        cancelButtonColor: '#6b7280',
        confirmButtonText: 'نعم، ارفض',
        cancelButtonText: 'إلغاء'
    }).then((result) => {
        if (result.isConfirmed) {
            // إرسال طلب الرفض إلى الخادم
            updateUserStatus(userId, 'reject');
        }
    });
}

/**
 * تحديث حالة المستخدم (قبول أو رفض)
 * @param {number} userId - معرف المستخدم
 * @param {string} action - الإجراء (approve أو reject)
 */
function updateUserStatus(userId, action) {
    // إرسال الطلب إلى الخادم
    fetch('api/update_user_status.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            user_id: userId,
            action: action
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`خطأ في استجابة الخادم: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        if (data.status === 'success') {
            // إزالة صف المستخدم من الجدول
            const userRow = document.querySelector(`tr[data-user-id="${userId}"]`);
            if (userRow) {
                userRow.classList.add('bg-green-50');
                setTimeout(() => {
                    userRow.remove();
                    
                    // تحديث عدد الطلبات المعلقة
                    const remainingRows = document.querySelectorAll('#pendingUsersTable tr.user-row').length;
                    updatePendingCount(remainingRows);
                    
                    // عرض رسالة عدم وجود طلبات إذا لم يتبق أي طلبات
                    if (remainingRows === 0) {
                        const noRequestsMessage = document.getElementById('noRequestsMessage');
                        if (noRequestsMessage) {
                            noRequestsMessage.classList.remove('hidden');
                        }
                    }
                }, 500);
            }
            
            // عرض رسالة النجاح
            const message = action === 'approve' ? 'تمت الموافقة على طلب التسجيل بنجاح' : 'تم رفض طلب التسجيل بنجاح';
            showStatusMessage('success', message);
            
            // عرض إشعار نجاح
            Swal.fire({
                title: 'تم بنجاح!',
                text: message,
                icon: 'success',
                confirmButtonText: 'حسناً'
            });
        } else {
            // عرض رسالة الخطأ
            showStatusMessage('error', data.message || 'حدث خطأ أثناء تحديث حالة المستخدم');
            
            // عرض إشعار خطأ
            Swal.fire({
                title: 'خطأ!',
                text: data.message || 'حدث خطأ أثناء تحديث حالة المستخدم',
                icon: 'error',
                confirmButtonText: 'حسناً'
            });
        }
    })
    .catch(error => {
        console.error('خطأ في تحديث حالة المستخدم:', error);
        
        // عرض رسالة الخطأ
        showStatusMessage('error', 'حدث خطأ أثناء الاتصال بالخادم. يرجى المحاولة مرة أخرى.');
        
        // عرض إشعار خطأ
        Swal.fire({
            title: 'خطأ!',
            text: 'حدث خطأ أثناء الاتصال بالخادم. يرجى المحاولة مرة أخرى.',
            icon: 'error',
            confirmButtonText: 'حسناً'
        });
    });
}

/**
 * تصفية المستخدمين حسب نص البحث
 * @param {string} searchText - نص البحث
 */
function filterUsers(searchText) {
    const userRows = document.querySelectorAll('#pendingUsersTable tr.user-row');
    const searchLower = searchText.toLowerCase();
    
    userRows.forEach(row => {
        const fullName = row.children[1].textContent.toLowerCase();
        const username = row.children[2].textContent.toLowerCase();
        const email = row.children[3].textContent.toLowerCase();
        
        if (fullName.includes(searchLower) || username.includes(searchLower) || email.includes(searchLower)) {
            row.style.display = '';
        } else {
            row.style.display = 'none';
        }
    });
}

/**
 * عرض رسالة حالة
 * @param {string} type - نوع الرسالة (success أو error)
 * @param {string} message - نص الرسالة
 */
function showStatusMessage(type, message) {
    const statusMessage = document.getElementById('statusMessage');
    if (!statusMessage) return;
    
    // تعيين نوع الرسالة
    statusMessage.className = type === 'success' 
        ? 'bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative mb-6'
        : 'bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-6';
    
    // تعيين نص الرسالة
    statusMessage.innerHTML = `
        <strong class="font-bold">${type === 'success' ? 'نجاح!' : 'خطأ!'}</strong>
        <span class="block sm:inline mr-2">${message}</span>
        <button type="button" class="absolute top-0 bottom-0 left-0 px-4 py-3" onclick="this.parentElement.classList.add('hidden')">
            <i class="fas fa-times"></i>
        </button>
    `;
    
    // عرض الرسالة
    statusMessage.classList.remove('hidden');
    
    // إخفاء الرسالة بعد 5 ثوانٍ
    setTimeout(() => {
        statusMessage.classList.add('hidden');
    }, 5000);
}

/**
 * تهرب من النص لمنع هجمات XSS
 * @param {string} text - النص المراد تهربه
 * @returns {string} - النص المهرب
 */
function escapeHTML(text) {
    if (!text) return '';
    return text
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#039;');
}
