/**
 * ملف JavaScript الخاص بصفحة إدارة الطلبات
 * يستخدم هذا الملف لعرض الطلبات وتصفيتها وإدارتها
 */

// متغيرات عامة
let allOrders = [];
let filteredOrders = [];
let currentPage = 1;
const ordersPerPage = 10;
let userRole = 'employee';

// عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    // تحميل الطلبات
    loadOrders();

    // تحميل الفروع
    loadBranches();

    // إضافة مستمعي الأحداث
    document.getElementById('refreshOrdersBtn').addEventListener('click', loadOrders);

    // إضافة مستمعي الأحداث للأزرار الجديدة
    document.getElementById('pdfExportBtn').addEventListener('click', showPdfExportModal);
    document.getElementById('excelExportBtn').addEventListener('click', exportToExcel);
    document.getElementById('printBtn').addEventListener('click', printOrders);
    document.getElementById('addSampleOrdersBtn').addEventListener('click', addSampleOrders);

    // إضافة مستمعي الأحداث لفلاتر التاريخ
    document.getElementById('dateFrom').addEventListener('change', applyDateFilter);
    document.getElementById('dateTo').addEventListener('change', applyDateFilter);

    // إضافة مستمع الحدث لفلتر الفروع
    document.getElementById('branchFilter').addEventListener('change', applyFilters);

    // إضافة مستمعي الأحداث لمودال تصدير PDF
    document.getElementById('closePdfModal').addEventListener('click', hidePdfExportModal);
    document.getElementById('closePdfModalBtn').addEventListener('click', hidePdfExportModal);
    document.getElementById('pdfExportForm').addEventListener('submit', handlePdfExport);

    // تعبئة قائمة الفروع في مودال تصدير PDF
    populatePdfBranchSelect();

    // تبديل حالة القائمة الجانبية
    const toggleSidebar = document.getElementById('toggleSidebar');
    const sidebar = document.querySelector('.sidebar');
    const mainContent = document.querySelector('.main-content');

    toggleSidebar.addEventListener('click', function() {
        const isExpanded = sidebar.classList.contains('sidebar-expanded');

        if (isExpanded) {
            sidebar.classList.remove('sidebar-expanded');
            sidebar.classList.add('sidebar-collapsed');
            mainContent.style.marginRight = '80px';

            // تغيير اتجاه أيقونة السهم
            toggleSidebar.querySelector('i').classList.remove('fa-chevron-right');
            toggleSidebar.querySelector('i').classList.add('fa-chevron-left');
        } else {
            sidebar.classList.remove('sidebar-collapsed');
            sidebar.classList.add('sidebar-expanded');
            mainContent.style.marginRight = '280px';

            // تغيير اتجاه أيقونة السهم
            toggleSidebar.querySelector('i').classList.remove('fa-chevron-left');
            toggleSidebar.querySelector('i').classList.add('fa-chevron-right');
        }
    });

    // إغلاق القائمة الجانبية عند النقر على الخلفية في الشاشات الصغيرة
    const overlay = document.getElementById('overlay');
    overlay.addEventListener('click', function() {
        if (window.innerWidth < 768) {
            sidebar.classList.remove('sidebar-open');
            overlay.classList.remove('overlay-open');
        }
    });

    // فتح القائمة الجانبية في الشاشات الصغيرة
    const mobileMenuButton = document.getElementById('mobileMenuButton');
    mobileMenuButton.addEventListener('click', function() {
        sidebar.classList.add('sidebar-open');
        overlay.classList.add('overlay-open');
    });
});

/**
 * تحميل الطلبات من الخادم
 */
function loadOrders() {
    console.log('بدء تحميل الطلبات...');

    // عرض مؤشر التحميل
    document.getElementById('ordersTableBody').innerHTML = `
        <tr>
            <td colspan="28" class="px-6 py-4 text-center">
                <div class="flex justify-center items-center">
                    <i class="fas fa-spinner fa-spin text-indigo-600 text-2xl"></i>
                    <span class="mr-2">جاري تحميل الطلبات...</span>
                </div>
            </td>
        </tr>
    `;

    // جلب الطلبات من الخادم
    fetch('api/get_orders.php')
        .then(response => {
            console.log('استجابة الخادم:', response.status, response.statusText);

            // التحقق من حالة الاستجابة
            if (!response.ok) {
                console.error('خطأ في استجابة الخادم:', response.status, response.statusText);
                throw new Error(`فشل في جلب الطلبات (${response.status}: ${response.statusText})`);
            }

            return response.json().catch(err => {
                console.error('خطأ في تحليل JSON:', err);
                throw new Error('فشل في تحليل استجابة الخادم. يرجى التحقق من سجلات الخطأ.');
            });
        })
        .then(data => {
            console.log('البيانات المستلمة:', data);

            if (data.status === 'success') {
                console.log('تم استلام البيانات بنجاح. عدد الطلبات:', data.orders.length);

                // تخزين الطلبات
                allOrders = data.orders;
                userRole = data.user_role;

                // تحديث عدد الطلبات في الفلتر
                const filterElement = document.querySelector('.text-indigo-700 span');
                if (filterElement) {
                    filterElement.textContent = `الكل (${allOrders.length})`;
                } else {
                    console.warn('لم يتم العثور على عنصر الفلتر');
                }

                // تطبيق الفلتر الحالي
                applyFilters();

                // إذا كانت المصفوفة فارغة، عرض رسالة مناسبة
                if (allOrders.length === 0) {
                    console.log('لا توجد طلبات متاحة');
                    document.getElementById('ordersTableBody').innerHTML = `
                        <tr>
                            <td colspan="28" class="px-6 py-4 text-center text-gray-500">
                                <i class="fas fa-info-circle ml-2"></i>
                                لا توجد طلبات متاحة حالياً
                            </td>
                        </tr>
                    `;
                }
            } else {
                console.error('خطأ في البيانات المستلمة:', data.message);
                throw new Error(data.message || 'فشل في جلب الطلبات');
            }
        })
        .catch(error => {
            console.error('خطأ في تحميل الطلبات:', error);

            // عرض رسالة الخطأ مع زر لإعادة المحاولة
            document.getElementById('ordersTableBody').innerHTML = `
                <tr>
                    <td colspan="28" class="px-6 py-4 text-center">
                        <div class="flex flex-col items-center">
                            <div class="text-red-600 mb-2">
                                <i class="fas fa-exclamation-triangle ml-2"></i>
                                ${error.message}
                            </div>
                            <button onclick="loadOrders()" class="bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-lg mt-2">
                                <i class="fas fa-sync-alt ml-1"></i>
                                إعادة المحاولة
                            </button>
                        </div>
                    </td>
                </tr>
            `;

            // إذا كان الخطأ متعلقًا بتسجيل الدخول، عرض زر لتسجيل الدخول
            if (error.message.includes('تسجيل الدخول')) {
                document.getElementById('ordersTableBody').innerHTML += `
                    <tr>
                        <td colspan="28" class="px-6 py-4 text-center">
                            <a href="login.php" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded-lg">
                                <i class="fas fa-sign-in-alt ml-1"></i>
                                تسجيل الدخول
                            </a>
                        </td>
                    </tr>
                `;
            }
        });
}

/**
 * تحميل الفروع من الخادم
 */
function loadBranches() {
    fetch('api/get_branches.php')
        .then(response => {
            if (!response.ok) {
                throw new Error('فشل في جلب الفروع');
            }
            return response.json();
        })
        .then(data => {
            if (data.status === 'success') {
                // إضافة الفروع إلى قائمة الفلتر
                const branchFilter = document.getElementById('branchFilter');

                data.branches.forEach(branch => {
                    const option = document.createElement('option');
                    option.value = branch.id;
                    option.textContent = branch.name;
                    branchFilter.appendChild(option);
                });
            }
        })
        .catch(error => {
            console.error('خطأ في تحميل الفروع:', error);
        });
}

/**
 * تطبيق الفلتر على الطلبات
 */
function applyFilters() {
    const statusFilter = document.getElementById('statusFilter').value;
    const branchFilter = document.getElementById('branchFilter').value;
    const searchInput = document.getElementById('searchInput').value.trim().toLowerCase();

    // تطبيق الفلتر
    filteredOrders = allOrders.filter(order => {
        // فلتر الحالة
        if (statusFilter && order.status !== statusFilter) {
            return false;
        }

        // فلتر الفرع
        if (branchFilter && order.branch_id != branchFilter) {
            return false;
        }

        // فلتر البحث
        if (searchInput) {
            const searchFields = [
                order.order_number,
                order.client_name,
                order.service_type,
                order.goods_name,
                order.goods_type,
                order.country_of_origin,
                order.departure_area,
                order.second_destination,
                order.delivery_area,
                order.recipient_name,
                order.phone_number,
                order.notes
            ];

            return searchFields.some(field => field && field.toLowerCase().includes(searchInput));
        }

        return true;
    });

    // إعادة تعيين الصفحة الحالية
    currentPage = 1;

    // عرض الطلبات
    displayOrders();
}

/**
 * عرض الطلبات في الجدول
 */
function displayOrders() {
    const ordersTableBody = document.getElementById('ordersTableBody');

    // حساب الطلبات التي سيتم عرضها في الصفحة الحالية
    const startIndex = (currentPage - 1) * ordersPerPage;
    const endIndex = Math.min(startIndex + ordersPerPage, filteredOrders.length);
    const ordersToDisplay = filteredOrders.slice(startIndex, endIndex);

    // التحقق من وجود طلبات
    if (ordersToDisplay.length === 0) {
        ordersTableBody.innerHTML = `
            <tr>
                <td colspan="28" class="px-6 py-4 text-center text-gray-500">
                    <i class="fas fa-search ml-2"></i>
                    لا توجد طلبات متطابقة مع معايير البحث
                </td>
            </tr>
        `;
        return;
    }

    // إنشاء صفوف الجدول
    let tableRows = '';

    ordersToDisplay.forEach((order, index) => {
        const rowNumber = startIndex + index + 1;

        // تحديد لون خلفية الصف بناءً على حالة الطلب
        let statusClass = '';
        let statusBadgeClass = '';

        switch (order.status) {
            case 'قيد المعالجة':
                statusClass = 'bg-blue-50';
                statusBadgeClass = 'bg-blue-100 text-blue-800';
                break;
            case 'في الطريق':
                statusClass = 'bg-yellow-50';
                statusBadgeClass = 'bg-yellow-100 text-yellow-800';
                break;
            case 'تم التسليم':
                statusClass = 'bg-green-50';
                statusBadgeClass = 'bg-green-100 text-green-800';
                break;
            case 'ملغى':
                statusClass = 'bg-red-50';
                statusBadgeClass = 'bg-red-100 text-red-800';
                break;
        }

        // تحديد لون خلفية حالة الدفع
        let paymentStatusBadgeClass = '';

        switch (order.payment_status) {
            case 'مدفوع':
                paymentStatusBadgeClass = 'bg-green-100 text-green-800';
                break;
            case 'جزئي':
                paymentStatusBadgeClass = 'bg-yellow-100 text-yellow-800';
                break;
            case 'غير مدفوع':
                paymentStatusBadgeClass = 'bg-red-100 text-red-800';
                break;
        }

        tableRows += `
            <tr class="${statusClass}">
                <td class="px-3 py-2 whitespace-nowrap text-center text-sm text-gray-500">${rowNumber}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm font-medium">${order.order_number}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-900">${order.client_name}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.service_type || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.goods_type || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.goods_name || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.delivery_duration || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.payment_type || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.amount_paid || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.remaining_amount || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.second_destination || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.delivery_area || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.order_upload_date || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm text-gray-500">${order.scheduled_delivery_date || '-'}</td>
                <td class="px-3 py-2 whitespace-nowrap text-right text-sm">
                    <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${statusBadgeClass}">
                        ${order.status || 'قيد المعالجة'}
                    </span>
                </td>
                <td class="px-3 py-2 whitespace-nowrap text-center text-sm font-medium">
                    <div class="flex justify-center space-x-2 space-x-reverse">
                        <button onclick="viewOrder(${order.id})" class="text-indigo-600 hover:text-indigo-900" title="عرض التفاصيل">
                            <i class="fas fa-eye"></i>
                        </button>
                        <button onclick="editOrder(${order.id})" class="text-blue-600 hover:text-blue-900" title="تعديل">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button onclick="printOrder(${order.id})" class="text-green-600 hover:text-green-900" title="طباعة">
                            <i class="fas fa-print"></i>
                        </button>
                        <button onclick="deleteOrder(${order.id})" class="text-red-600 hover:text-red-900" title="حذف">
                            <i class="fas fa-trash-alt"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `;
    });

    // تحديث الجدول
    ordersTableBody.innerHTML = tableRows;

    // تحديث معلومات الترقيم
    updatePagination();
}

/**
 * تحديث معلومات الترقيم
 */
function updatePagination() {
    const totalPages = Math.ceil(filteredOrders.length / ordersPerPage);
    const startItem = filteredOrders.length > 0 ? (currentPage - 1) * ordersPerPage + 1 : 0;
    const endItem = Math.min(startItem + ordersPerPage - 1, filteredOrders.length);

    // تحديث نص معلومات الترقيم
    document.querySelector('.text-gray-700').innerHTML = `
        عرض
        <span class="font-medium">${startItem}</span>
        إلى
        <span class="font-medium">${endItem}</span>
        من
        <span class="font-medium">${filteredOrders.length}</span>
        نتائج
    `;

    // تحديث أزرار الترقيم
    // (يمكن إضافة المزيد من الكود هنا لتحديث أزرار الترقيم)
}

/**
 * تطبيق فلتر التاريخ
 */
function applyDateFilter() {
    const dateFrom = document.getElementById('dateFrom').value;
    const dateTo = document.getElementById('dateTo').value;

    // تطبيق الفلتر فقط إذا تم تحديد تاريخ
    if (dateFrom || dateTo) {
        // تحويل التواريخ إلى كائنات Date
        const fromDate = dateFrom ? new Date(dateFrom) : new Date(0); // أقدم تاريخ ممكن
        const toDate = dateTo ? new Date(dateTo) : new Date(); // التاريخ الحالي

        // تعيين وقت نهاية اليوم لتاريخ النهاية
        toDate.setHours(23, 59, 59, 999);

        // تطبيق الفلتر على الطلبات
        filteredOrders = allOrders.filter(order => {
            // تحويل تاريخ الطلب إلى كائن Date
            const orderDate = new Date(order.order_upload_date || order.created_at);

            // التحقق من أن تاريخ الطلب يقع ضمن النطاق المحدد
            return orderDate >= fromDate && orderDate <= toDate;
        });

        // إعادة تعيين الصفحة الحالية
        currentPage = 1;

        // عرض الطلبات
        displayOrders();
    }
}

/**
 * عرض مودال تصدير الطلبات إلى PDF
 */
function showPdfExportModal() {
    // تعيين التواريخ الافتراضية (الشهر الحالي)
    const today = new Date();
    const firstDayOfMonth = new Date(today.getFullYear(), today.getMonth(), 1);
    const lastDayOfMonth = new Date(today.getFullYear(), today.getMonth() + 1, 0);

    // تنسيق التواريخ بتنسيق YYYY-MM-DD
    const formatDate = (date) => {
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const day = String(date.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    };

    // تعيين التواريخ في النموذج
    document.getElementById('pdf-date-from').value = formatDate(firstDayOfMonth);
    document.getElementById('pdf-date-to').value = formatDate(lastDayOfMonth);

    // إظهار المودال
    const modal = document.getElementById('pdfExportModal');
    modal.classList.remove('hidden');
}

/**
 * إخفاء مودال تصدير الطلبات إلى PDF
 */
function hidePdfExportModal() {
    const modal = document.getElementById('pdfExportModal');
    modal.classList.add('hidden');
}

/**
 * تعبئة قائمة الفروع في مودال تصدير PDF
 */
function populatePdfBranchSelect() {
    fetch('../api/get_branches.php')
        .then(response => {
            if (!response.ok) {
                throw new Error('فشل في جلب الفروع');
            }
            return response.json();
        })
        .then(data => {
            if (data.status === 'success') {
                // إضافة الفروع إلى قائمة الفلتر
                const branchSelect = document.getElementById('pdf-branch');

                // التأكد من وجود العنصر
                if (branchSelect) {
                    data.branches.forEach(branch => {
                        const option = document.createElement('option');
                        option.value = branch.id;
                        option.textContent = branch.name;
                        branchSelect.appendChild(option);
                    });
                }
            }
        })
        .catch(error => {
            console.error('خطأ في تحميل الفروع:', error);
        });
}

/**
 * معالجة تصدير الطلبات إلى PDF
 */
function handlePdfExport(event) {
    event.preventDefault();

    // الحصول على قيم النموذج
    const dateFrom = document.getElementById('pdf-date-from').value;
    const dateTo = document.getElementById('pdf-date-to').value;
    const branchId = document.getElementById('pdf-branch').value;
    const status = document.getElementById('pdf-status').value;

    // التحقق من صحة التواريخ
    if (dateFrom && dateTo) {
        const fromDate = new Date(dateFrom);
        const toDate = new Date(dateTo);

        if (fromDate > toDate) {
            alert('تاريخ البداية يجب أن يكون قبل تاريخ النهاية');
            return;
        }
    }

    // بناء عنوان URL للتصدير
    let exportUrl = '../api/export_orders_pdf.php?';

    if (dateFrom) {
        exportUrl += `date_from=${dateFrom}&`;
    }

    if (dateTo) {
        exportUrl += `date_to=${dateTo}&`;
    }

    if (branchId) {
        exportUrl += `branch_id=${branchId}&`;
    }

    if (status) {
        exportUrl += `status=${status}&`;
    }

    // إخفاء المودال
    hidePdfExportModal();

    // فتح نافذة جديدة لتنزيل الملف
    window.open(exportUrl, '_blank');
}

/**
 * تصدير الطلبات إلى ملف Excel
 */
function exportToExcel() {
    // إظهار رسالة للمستخدم
    alert('جاري تصدير الطلبات إلى ملف Excel...');

    // هنا يمكن إضافة كود لتصدير الطلبات إلى ملف Excel
    // يمكن استخدام مكتبة مثل SheetJS أو إرسال طلب إلى الخادم لإنشاء ملف Excel

    // مثال: إرسال طلب إلى الخادم لإنشاء ملف Excel
    // window.location.href = 'api/export_orders_excel.php';
}

/**
 * طباعة الطلبات
 */
function printOrders() {
    // إنشاء نافذة طباعة جديدة
    const printWindow = window.open('', '_blank');

    // إنشاء محتوى HTML للطباعة
    let printContent = `
        <!DOCTYPE html>
        <html lang="ar" dir="rtl">
        <head>
            <meta charset="UTF-8">
            <title>طباعة الطلبات</title>
            <style>
                body {
                    font-family: 'Arial', sans-serif;
                    margin: 0;
                    padding: 20px;
                }
                h1 {
                    text-align: center;
                    margin-bottom: 20px;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                }
                th, td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: right;
                }
                th {
                    background-color: #f2f2f2;
                }
                .print-header {
                    display: flex;
                    justify-content: space-between;
                    margin-bottom: 20px;
                }
                .print-date {
                    text-align: left;
                }
                @media print {
                    button {
                        display: none;
                    }
                }
            </style>
        </head>
        <body>
            <div class="print-header">
                <h1>نظام طلبات الشحن - قائمة الطلبات</h1>
                <div class="print-date">تاريخ الطباعة: ${new Date().toLocaleDateString('ar-SA')}</div>
            </div>
            <button onclick="window.print()">طباعة</button>
            <table>
                <thead>
                    <tr>
                        <th>#</th>
                        <th>رقم الطلب</th>
                        <th>اسم العميل</th>
                        <th>نوع الخدمة</th>
                        <th>البضاعة</th>
                        <th>منطقة الانطلاق</th>
                        <th>منطقة التسليم</th>
                        <th>تاريخ الطلب</th>
                        <th>حالة الطلب</th>
                    </tr>
                </thead>
                <tbody>
    `;

    // إضافة صفوف الطلبات
    filteredOrders.forEach((order, index) => {
        printContent += `
            <tr>
                <td>${index + 1}</td>
                <td>${order.order_number}</td>
                <td>${order.client_name}</td>
                <td>${order.service_type || '-'}</td>
                <td>${order.goods_name || '-'}</td>
                <td>${order.departure_area || '-'}</td>
                <td>${order.delivery_area || '-'}</td>
                <td>${order.order_upload_date || '-'}</td>
                <td>${order.status || 'قيد المعالجة'}</td>
            </tr>
        `;
    });

    // إغلاق محتوى HTML
    printContent += `
                </tbody>
            </table>
        </body>
        </html>
    `;

    // كتابة المحتوى في نافذة الطباعة
    printWindow.document.write(printContent);
    printWindow.document.close();
}

/**
 * إضافة طلبات تجريبية
 */
function addSampleOrders() {
    // عرض رسالة تأكيد
    if (!confirm('هل أنت متأكد من إضافة طلبات تجريبية؟')) {
        return;
    }

    // عرض مؤشر التحميل
    document.getElementById('ordersTableBody').innerHTML = `
        <tr>
            <td colspan="16" class="px-6 py-4 text-center">
                <div class="flex justify-center items-center">
                    <i class="fas fa-spinner fa-spin text-indigo-600 text-2xl"></i>
                    <span class="mr-2">جاري إضافة الطلبات التجريبية...</span>
                </div>
            </td>
        </tr>
    `;

    // إرسال طلب لإضافة طلبات تجريبية
    fetch('api/add_sample_orders.php')
        .then(response => {
            if (!response.ok) {
                throw new Error(`فشل في إضافة الطلبات التجريبية (${response.status})`);
            }
            return response.json();
        })
        .then(data => {
            if (data.status === 'success') {
                // عرض رسالة نجاح
                alert('تم إضافة الطلبات التجريبية بنجاح');

                // إعادة تحميل الطلبات
                loadOrders();
            } else {
                throw new Error(data.message || 'فشل في إضافة الطلبات التجريبية');
            }
        })
        .catch(error => {
            console.error('خطأ في إضافة الطلبات التجريبية:', error);

            // عرض رسالة الخطأ
            document.getElementById('ordersTableBody').innerHTML = `
                <tr>
                    <td colspan="16" class="px-6 py-4 text-center text-red-500">
                        <i class="fas fa-exclamation-triangle ml-2"></i>
                        ${error.message}
                    </td>
                </tr>
            `;
        });
}
