/**
 * ملف JavaScript لإدارة قائمة الطلبات
 * يتعامل مع جلب البيانات من قاعدة البيانات عبر API
 */

// متغيرات عامة
let allOrders = [];
let filteredOrders = [];
let currentPage = 1;
const pageSize = 10;

document.addEventListener('DOMContentLoaded', function() {
  // تهيئة الصفحة
  initOrdersPage();

  // إضافة مستمعي الأحداث
  setupEventListeners();
});

/**
 * تهيئة صفحة الطلبات
 */
function initOrdersPage() {
  console.log('تهيئة صفحة قائمة الطلبات...');

  // تحميل الطلبات
  loadOrders();

  // تهيئة عناصر التصفية
  initFilters();
}

/**
 * إضافة مستمعي الأحداث
 */
function setupEventListeners() {
  // زر تحديث الطلبات
  const refreshBtn = document.getElementById('refreshOrdersBtn');
  if (refreshBtn) {
    refreshBtn.addEventListener('click', loadOrders);
  }

  // عناصر التصفية
  const statusFilter = document.getElementById('statusFilter');
  const branchFilter = document.getElementById('branchFilter');
  const searchInput = document.getElementById('searchInput');

  if (statusFilter) {
    statusFilter.addEventListener('change', applyFilters);
  }

  if (branchFilter) {
    branchFilter.addEventListener('change', applyFilters);
  }

  if (searchInput) {
    searchInput.addEventListener('input', applyFilters);
  }

  // أزرار التنقل بين الصفحات
  const prevPageBtn = document.getElementById('prevPageBtn');
  const nextPageBtn = document.getElementById('nextPageBtn');

  if (prevPageBtn) {
    prevPageBtn.addEventListener('click', () => {
      if (currentPage > 1) {
        currentPage--;
        renderOrders();
      }
    });
  }

  if (nextPageBtn) {
    nextPageBtn.addEventListener('click', () => {
      const totalPages = Math.ceil(filteredOrders.length / pageSize);
      if (currentPage < totalPages) {
        currentPage++;
        renderOrders();
      }
    });
  }
}

/**
 * تهيئة عناصر التصفية
 */
function initFilters() {
  // تهيئة قائمة الفروع
  const branchFilter = document.getElementById('branchFilter');
  if (branchFilter) {
    fetch('api/branches.php')
      .then(response => response.json())
      .then(data => {
        if (data.status === 'success') {
          let options = '<option value="">الكل</option>';
          data.branches.forEach(branch => {
            options += `<option value="${branch.name}">${branch.name}</option>`;
          });
          branchFilter.innerHTML = options;
        }
      })
      .catch(error => {
        console.error('خطأ في تحميل الفروع:', error);
      });
  }
}

/**
 * تحميل الطلبات من الخادم
 */
function loadOrders() {
  const ordersTableBody = document.getElementById('ordersTableBody');
  if (!ordersTableBody) return;

  // عرض مؤشر التحميل
  ordersTableBody.innerHTML = `
    <tr>
      <td colspan="17" class="px-3 py-4 text-center text-gray-500">
        <i class="fas fa-spinner fa-spin ml-2"></i>
        جاري تحميل الطلبات...
      </td>
    </tr>
  `;

  // جلب الطلبات من الخادم
  fetch('api/get_orders.php')
    .then(response => response.json())
    .then(data => {
      if (data.status === 'success') {
        // تخزين الطلبات
        allOrders = data.orders;
        filteredOrders = [...allOrders];

        // تحديث عدد الطلبات
        updateOrdersCount(allOrders.length);

        // عرض الطلبات
        renderOrders();
      } else {
        // عرض رسالة الخطأ
        ordersTableBody.innerHTML = `
          <tr>
            <td colspan="17" class="px-3 py-4 text-center text-red-500">
              <i class="fas fa-exclamation-circle ml-2"></i>
              خطأ: ${data.message}
            </td>
          </tr>
        `;
      }
    })
    .catch(error => {
      console.error('خطأ في تحميل الطلبات:', error);

      // عرض رسالة الخطأ
      ordersTableBody.innerHTML = `
        <tr>
          <td colspan="17" class="px-3 py-4 text-center text-red-500">
            <i class="fas fa-exclamation-circle ml-2"></i>
            حدث خطأ أثناء تحميل الطلبات. يرجى المحاولة مرة أخرى.
          </td>
        </tr>
      `;
    });
}

/**
 * تحديث عدد الطلبات
 */
function updateOrdersCount(count) {
  const ordersCount = document.getElementById('ordersCount');
  if (ordersCount) {
    ordersCount.textContent = count;
  }
}

/**
 * تطبيق التصفية على الطلبات
 */
function applyFilters() {
  const statusFilter = document.getElementById('statusFilter');
  const branchFilter = document.getElementById('branchFilter');
  const searchInput = document.getElementById('searchInput');

  const statusValue = statusFilter ? statusFilter.value : '';
  const branchValue = branchFilter ? branchFilter.value : '';
  const searchTerm = searchInput ? searchInput.value.toLowerCase() : '';

  // تصفية الطلبات
  filteredOrders = allOrders.filter(order => {
    // تصفية حسب البحث
    const matchesSearch = !searchTerm ||
      (order.orderNumber && order.orderNumber.toLowerCase().includes(searchTerm)) ||
      (order.clientName && order.clientName.toLowerCase().includes(searchTerm)) ||
      (order.serviceType && order.serviceType.toLowerCase().includes(searchTerm)) ||
      (order.goodsName && order.goodsName.toLowerCase().includes(searchTerm)) ||
      (order.goodsType && order.goodsType.toLowerCase().includes(searchTerm)) ||
      (order.countryOfOrigin && order.countryOfOrigin.toLowerCase().includes(searchTerm)) ||
      (order.branch && order.branch.toLowerCase().includes(searchTerm)) ||
      (order.departureArea && order.departureArea.toLowerCase().includes(searchTerm)) ||
      (order.secondDestination && order.secondDestination.toLowerCase().includes(searchTerm)) ||
      (order.deliveryArea && order.deliveryArea.toLowerCase().includes(searchTerm)) ||
      (order.recipientName && order.recipientName.toLowerCase().includes(searchTerm)) ||
      (order.phoneNumber && order.phoneNumber.includes(searchTerm)) ||
      (order.notes && order.notes.toLowerCase().includes(searchTerm));

    // تصفية حسب الحالة
    const matchesStatus = !statusValue || order.status === statusValue;

    // تصفية حسب الفرع
    const matchesBranch = !branchValue || order.branch === branchValue;

    return matchesSearch && matchesStatus && matchesBranch;
  });

  // تحديث عدد الطلبات المصفاة
  updateOrdersCount(filteredOrders.length);

  // إعادة تعيين الصفحة الحالية
  currentPage = 1;

  // عرض الطلبات المصفاة
  renderOrders();
}

/**
 * عرض الطلبات في الجدول
 */
function renderOrders() {
  const ordersTableBody = document.getElementById('ordersTableBody');
  if (!ordersTableBody) return;

  // حساب نطاق الطلبات للصفحة الحالية
  const startIndex = (currentPage - 1) * pageSize;
  const endIndex = Math.min(startIndex + pageSize, filteredOrders.length);
  const pageOrders = filteredOrders.slice(startIndex, endIndex);

  // تحديث معلومات الصفحات
  updatePagination();

  // عرض الطلبات
  if (pageOrders.length === 0) {
    // عرض رسالة إذا لم تكن هناك طلبات
    ordersTableBody.innerHTML = `
      <tr>
        <td colspan="17" class="px-3 py-4 text-center text-gray-500">
          لا توجد طلبات متطابقة مع معايير البحث
        </td>
      </tr>
    `;
  } else {
    // عرض الطلبات
    ordersTableBody.innerHTML = '';

    pageOrders.forEach((order, index) => {
      // تحديد لون حالة الطلب
      const statusClass = getStatusClass(order.status);
      const paymentStatusClass = getPaymentStatusClass(order.paymentStatus);

      // إنشاء صف جديد
      const row = document.createElement('tr');
      row.className = 'hover:bg-gray-50';

      // إضافة محتوى الصف
      row.innerHTML = `
        <td class="px-3 py-2 whitespace-nowrap text-sm font-medium text-center text-gray-900">
          ${startIndex + index + 1}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm font-medium text-gray-900">
          ${order.orderNumber || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.clientName || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.serviceType || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.goodsName || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.goodsType || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.countryOfOrigin || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.weight || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.branch || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.departureArea || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.secondDestination || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.deliveryArea || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.serviceFees || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.currency || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.deliveryDuration || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.amountPaid || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.remainingAmount || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.uploadDate || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.deliveryDate || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.recipientName || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.phoneNumber || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.amountAgreedWithCustomer || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.amountAgreedWithAgent || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.otherExpenses || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.profit || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-gray-700">
          ${order.notes || ''}
        </td>
        <td class="px-3 py-2 whitespace-nowrap">
          <span class="px-2 py-1 text-xs font-semibold rounded-full ${paymentStatusClass}">
            ${order.paymentStatus || ''}
          </span>
        </td>
        <td class="px-3 py-2 whitespace-nowrap">
          <span class="px-2 py-1 text-xs font-semibold rounded-full ${statusClass}">
            ${order.status || ''}
          </span>
        </td>
        <td class="px-3 py-2 whitespace-nowrap text-sm text-right">
          <div class="flex items-center justify-end space-x-2 space-x-reverse">
            <button onclick="showOrderDetails(${order.id})" class="text-indigo-600 hover:text-indigo-900">
              <i class="fas fa-eye"></i>
            </button>
            <button onclick="editOrder(${order.id})" class="text-blue-600 hover:text-blue-900">
              <i class="fas fa-edit"></i>
            </button>
            <button onclick="printOrder(${order.id})" class="text-green-600 hover:text-green-900">
              <i class="fas fa-print"></i>
            </button>
          </div>
        </td>
      `;

      // إضافة الصف إلى الجدول
      ordersTableBody.appendChild(row);
    });
  }
}

/**
 * تحديث معلومات الصفحات
 */
function updatePagination() {
  const paginationInfo = document.getElementById('paginationInfo');
  const prevPageBtn = document.getElementById('prevPageBtn');
  const nextPageBtn = document.getElementById('nextPageBtn');

  if (!paginationInfo || !prevPageBtn || !nextPageBtn) return;

  const totalPages = Math.ceil(filteredOrders.length / pageSize);

  // تحديث معلومات الصفحات
  paginationInfo.textContent = `الصفحة ${currentPage} من ${totalPages || 1}`;

  // تحديث حالة أزرار التنقل
  prevPageBtn.disabled = currentPage <= 1;
  nextPageBtn.disabled = currentPage >= totalPages;

  // تحديث الفئات
  prevPageBtn.className = prevPageBtn.disabled
    ? 'btn btn-secondary px-3 py-1 text-sm opacity-50 cursor-not-allowed'
    : 'btn btn-secondary px-3 py-1 text-sm';

  nextPageBtn.className = nextPageBtn.disabled
    ? 'btn btn-secondary px-3 py-1 text-sm opacity-50 cursor-not-allowed'
    : 'btn btn-secondary px-3 py-1 text-sm';
}

/**
 * الحصول على فئة CSS لحالة الطلب
 */
function getStatusClass(status) {
  switch (status) {
    case 'قيد المعالجة':
      return 'bg-yellow-100 text-yellow-800';
    case 'في الطريق':
      return 'bg-blue-100 text-blue-800';
    case 'تم التسليم':
      return 'bg-green-100 text-green-800';
    case 'ملغى':
      return 'bg-red-100 text-red-800';
    default:
      return 'bg-gray-100 text-gray-800';
  }
}

/**
 * الحصول على فئة CSS لحالة الدفع
 */
function getPaymentStatusClass(status) {
  switch (status) {
    case 'مدفوع':
      return 'bg-green-100 text-green-800';
    case 'جزئي':
      return 'bg-yellow-100 text-yellow-800';
    case 'غير مدفوع':
      return 'bg-red-100 text-red-800';
    default:
      return 'bg-gray-100 text-gray-800';
  }
}

/**
 * عرض تفاصيل الطلب
 */
function showOrderDetails(orderId) {
  // البحث عن الطلب
  const order = allOrders.find(order => order.id == orderId);

  if (!order) {
    alert('لم يتم العثور على الطلب');
    return;
  }

  // عرض تفاصيل الطلب (يمكن تنفيذ هذا بطرق مختلفة مثل نافذة منبثقة أو صفحة منفصلة)
  alert(`تفاصيل الطلب: ${order.orderNumber} - ${order.clientName}`);

  // يمكن استبدال هذا بتنفيذ أكثر تفصيلاً لعرض التفاصيل
}

/**
 * تحرير الطلب
 */
function editOrder(orderId) {
  // الانتقال إلى صفحة تحرير الطلب
  window.location.href = `edit-order.html?id=${orderId}`;
}

/**
 * طباعة الطلب
 */
function printOrder(orderId) {
  // الانتقال إلى صفحة طباعة الطلب
  window.location.href = `print-order.html?id=${orderId}`;
}
