/**
 * سكريبت الخريطة التفاعلية
 * يقوم هذا السكريبت بإنشاء خريطة تفاعلية تعرض دول الشحن والمسارات بينها
 */

// انتظار تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
  // إنشاء الخريطة
  const map = L.map('map', {
    center: [24.5, 45], // مركز الخريطة (الشرق الأوسط)
    zoom: 3, // مستوى التكبير
    zoomControl: false, // إخفاء أزرار التكبير/التصغير
    attributionControl: false, // إخفاء حقوق الملكية
    dragging: false, // منع السحب
    scrollWheelZoom: false // منع التكبير/التصغير بعجلة الماوس
  });

  // إضافة طبقة الخريطة
  L.tileLayer('https://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}{r}.png', {
    maxZoom: 19
  }).addTo(map);

  // تعريف الدول الرئيسية
  const countries = [
    { name: 'الصين', code: 'CN', coords: [35.8617, 104.1954], flagPath: 'https://flagcdn.com/w160/cn.png' },
    { name: 'اليمن', code: 'YE', coords: [15.5527, 48.5164], flagPath: 'https://flagcdn.com/w160/ye.png' },
    { name: 'السعودية', code: 'SA', coords: [23.8859, 45.0792], flagPath: 'https://flagcdn.com/w160/sa.png' },
    { name: 'عمان', code: 'OM', coords: [21.4735, 55.9754], flagPath: 'https://flagcdn.com/w160/om.png' },
    { name: 'الأردن', code: 'JO', coords: [31.9454, 35.9284], flagPath: 'https://flagcdn.com/w160/jo.png' },
    { name: 'العراق', code: 'IQ', coords: [33.2232, 43.6793], flagPath: 'https://flagcdn.com/w160/iq.png' },
    { name: 'البحرين', code: 'BH', coords: [26.0667, 50.5577], flagPath: 'https://flagcdn.com/w160/bh.png' },
    { name: 'ماليزيا', code: 'MY', coords: [4.2105, 101.9758], flagPath: 'https://flagcdn.com/w160/my.png' }
  ];

  // إنشاء أيقونات مخصصة للدول
  const countryIcon = L.Icon.extend({
    options: {
      iconSize: [30, 20],
      iconAnchor: [15, 10],
      className: 'country-icon'
    }
  });

  // إضافة علامات الدول إلى الخريطة
  countries.forEach(country => {
    // التحقق من وجود الصورة قبل إنشاء الأيقونة
    const img = new Image();
    img.onload = function() {
      // إنشاء أيقونة الدولة بعد تحميل الصورة بنجاح
      const icon = new countryIcon({ iconUrl: country.flagPath });

      // إضافة علامة الدولة إلى الخريطة
      const marker = L.marker(country.coords, { icon: icon })
        .addTo(map)
        .bindTooltip(country.name, {
          permanent: true,
          direction: 'top',
          className: 'country-label',
          offset: [0, -15]
        });

      // إضافة تأثير نبض للعلامة
      if (marker._icon) {
        marker._icon.classList.add('pulse');
      }
    };

    img.onerror = function() {
      console.warn(`لم يتم العثور على علم الدولة: ${country.name} في المسار: ${country.flagPath}`);

      // إنشاء أيقونة بديلة للدولة
      const fallbackIcon = L.divIcon({
        html: `<div style="background-color: #3498db; color: white; font-size: 10px; text-align: center; width: 100%; height: 100%; display: flex; align-items: center; justify-content: center;">${country.code}</div>`,
        className: 'country-icon-fallback',
        iconSize: [30, 20],
        iconAnchor: [15, 10]
      });

      // إضافة علامة الدولة إلى الخريطة باستخدام الأيقونة البديلة
      const marker = L.marker(country.coords, { icon: fallbackIcon })
        .addTo(map)
        .bindTooltip(country.name, {
          permanent: true,
          direction: 'top',
          className: 'country-label',
          offset: [0, -15]
        });

      // إضافة تأثير نبض للعلامة
      if (marker._icon) {
        marker._icon.classList.add('pulse');
      }
    };

    // بدء تحميل الصورة
    img.src = country.flagPath;
  });

  // تعريف مسارات الشحن
  const routes = [
    { from: 'CN', to: 'YE', type: 'ship', color: '#3498db' },
    { from: 'CN', to: 'YE', type: 'plane', color: '#e74c3c' },
    { from: 'SA', to: 'YE', type: 'truck', color: '#2ecc71' },
    { from: 'OM', to: 'YE', type: 'truck', color: '#f39c12' },
    { from: 'JO', to: 'YE', type: 'plane', color: '#9b59b6' },
    { from: 'MY', to: 'YE', type: 'ship', color: '#1abc9c' },
    { from: 'BH', to: 'YE', type: 'ship', color: '#34495e' },
    { from: 'IQ', to: 'YE', type: 'truck', color: '#d35400' }
  ];

  // إنشاء أيقونات وسائل النقل
  const transportIcons = {
    ship: L.divIcon({
      html: '<i class="fas fa-ship" style="color: #3498db; font-size: 16px;"></i>',
      className: 'ship-icon',
      iconSize: [20, 20],
      iconAnchor: [10, 10]
    }),
    plane: L.divIcon({
      html: '<i class="fas fa-plane" style="color: #e74c3c; font-size: 16px;"></i>',
      className: 'plane-icon',
      iconSize: [20, 20],
      iconAnchor: [10, 10]
    }),
    truck: L.divIcon({
      html: '<i class="fas fa-truck" style="color: #2ecc71; font-size: 16px;"></i>',
      className: 'truck-icon',
      iconSize: [20, 20],
      iconAnchor: [10, 10]
    })
  };

  // إنشاء مسارات الشحن وتحريك وسائل النقل
  routes.forEach(route => {
    // الحصول على إحداثيات الدول
    const fromCountry = countries.find(c => c.code === route.from);
    const toCountry = countries.find(c => c.code === route.to);

    if (fromCountry && toCountry) {
      // إنشاء مسار متقطع
      const pathOptions = {
        color: route.color,
        weight: 2,
        opacity: 0.7,
        dashArray: '5, 10',
        className: 'animated-path'
      };

      // إضافة المسار إلى الخريطة
      const path = L.polyline([fromCountry.coords, toCountry.coords], pathOptions).addTo(map);

      // حساب نقطة في منتصف المسار
      const midPoint = [
        (fromCountry.coords[0] + toCountry.coords[0]) / 2,
        (fromCountry.coords[1] + toCountry.coords[1]) / 2
      ];

      // إضافة أيقونة وسيلة النقل
      const transportMarker = L.marker(midPoint, { icon: transportIcons[route.type] }).addTo(map);

      // إضافة تأثير حركي
      transportMarker._icon.classList.add('animate-bounce');

      // تحريك وسيلة النقل على طول المسار
      animateMarker(transportMarker, fromCountry.coords, toCountry.coords);
    }
  });

  // دالة لتحريك العلامة على طول المسار
  function animateMarker(marker, start, end) {
    let startTime = null;
    const duration = 5000; // مدة الحركة بالمللي ثانية

    function animate(currentTime) {
      if (!startTime) startTime = currentTime;

      // حساب نسبة الوقت المنقضي
      const elapsedTime = currentTime - startTime;
      const progress = Math.min(elapsedTime / duration, 1);

      // حساب الموقع الحالي
      const lat = start[0] + (end[0] - start[0]) * progress;
      const lng = start[1] + (end[1] - start[1]) * progress;

      // تحديث موقع العلامة
      marker.setLatLng([lat, lng]);

      // استمرار الحركة إذا لم تنته بعد
      if (progress < 1) {
        requestAnimationFrame(animate);
      } else {
        // إعادة الحركة من البداية
        startTime = null;
        setTimeout(() => {
          requestAnimationFrame(animate);
        }, 1000);
      }
    }

    // بدء الحركة
    requestAnimationFrame(animate);
  }

  // إضافة عنوان الخريطة
  const mapTitle = L.control({ position: 'topright' });
  mapTitle.onAdd = function() {
    const div = L.DomUtil.create('div', 'map-title');
    div.innerHTML = 'خريطة مسارات الشحن';
    return div;
  };
  mapTitle.addTo(map);

  // إضافة مفتاح الخريطة
  const mapLegend = L.control({ position: 'bottomright' });
  mapLegend.onAdd = function() {
    const div = L.DomUtil.create('div', 'map-legend');
    div.innerHTML = `
      <div class="legend-item">
        <div class="legend-color" style="background-color: #3498db;"></div>
        <span>شحن بحري</span>
      </div>
      <div class="legend-item">
        <div class="legend-color" style="background-color: #e74c3c;"></div>
        <span>شحن جوي</span>
      </div>
      <div class="legend-item">
        <div class="legend-color" style="background-color: #2ecc71;"></div>
        <span>شحن بري</span>
      </div>
    `;
    return div;
  };
  mapLegend.addTo(map);
});
