/**
 * ملف الرسوم المتحركة للخريطة
 * يقوم هذا الملف بإنشاء خريطة العالم وإضافة رسوم متحركة لتوضيح عملية الشحن
 */

// تهيئة الخريطة عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
  initMap();
});

// الدول المستهدفة مع إحداثياتها
const targetCountries = {
  china: {
    name: 'الصين',
    coords: [35.8617, 104.1954],
    color: '#e74c3c'
  },
  yemen: {
    name: 'اليمن',
    coords: [15.5527, 48.5164],
    color: '#3498db'
  },
  oman: {
    name: 'عمان',
    coords: [21.4735, 55.9754],
    color: '#2ecc71'
  },
  saudiArabia: {
    name: 'السعودية',
    coords: [23.8859, 45.0792],
    color: '#f39c12'
  }
};

// مسارات الشحن
const shippingRoutes = [
  {
    from: 'china',
    to: 'yemen',
    color: '#e74c3c',
    duration: 8000,
    delay: 0
  },
  {
    from: 'china',
    to: 'saudiArabia',
    color: '#e74c3c',
    duration: 7000,
    delay: 2000
  },
  {
    from: 'yemen',
    to: 'oman',
    color: '#3498db',
    duration: 3000,
    delay: 9000
  },
  {
    from: 'saudiArabia',
    to: 'yemen',
    color: '#f39c12',
    duration: 2500,
    delay: 10000
  }
];

// تهيئة الخريطة
function initMap() {
  const mapContainer = document.getElementById('worldMap');
  if (!mapContainer) return;

  // إنشاء الخريطة باستخدام مكتبة Leaflet
  const map = L.map('worldMap', {
    center: [25, 60],
    zoom: 3,
    zoomControl: false,
    attributionControl: false,
    dragging: false,
    scrollWheelZoom: false,
    doubleClickZoom: false
  });

  // إضافة طبقة الخريطة
  L.tileLayer('https://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}{r}.png', {
    maxZoom: 19
  }).addTo(map);

  // إضافة الدول المستهدفة
  Object.keys(targetCountries).forEach(countryKey => {
    const country = targetCountries[countryKey];
    
    // إضافة علامة للدولة
    const marker = L.circleMarker(country.coords, {
      radius: 8,
      fillColor: country.color,
      color: '#fff',
      weight: 2,
      opacity: 1,
      fillOpacity: 0.8
    }).addTo(map);
    
    // إضافة اسم الدولة
    L.tooltip({
      permanent: true,
      direction: 'top',
      className: 'country-label'
    })
    .setContent(country.name)
    .setLatLng(country.coords)
    .addTo(map);
    
    // تخزين المؤشر للاستخدام لاحقًا
    targetCountries[countryKey].marker = marker;
  });

  // بدء الرسوم المتحركة للشحن
  startShippingAnimation(map);
}

// بدء الرسوم المتحركة للشحن
function startShippingAnimation(map) {
  // إنشاء رمز الشاحنة
  const truckIcon = L.divIcon({
    html: '<i class="fas fa-truck text-yellow-300 text-xl animate-bounce"></i>',
    className: 'truck-icon',
    iconSize: [30, 30],
    iconAnchor: [15, 15]
  });

  // إنشاء رمز السفينة
  const shipIcon = L.divIcon({
    html: '<i class="fas fa-ship text-blue-300 text-xl animate-bounce"></i>',
    className: 'ship-icon',
    iconSize: [30, 30],
    iconAnchor: [15, 15]
  });

  // إنشاء رمز الطائرة
  const planeIcon = L.divIcon({
    html: '<i class="fas fa-plane text-white text-xl animate-bounce"></i>',
    className: 'plane-icon',
    iconSize: [30, 30],
    iconAnchor: [15, 15]
  });

  // تشغيل الرسوم المتحركة لكل مسار
  shippingRoutes.forEach(route => {
    setTimeout(() => {
      animateShipping(map, route, getTransportIcon(route));
    }, route.delay);
  });
  
  // تكرار الرسوم المتحركة كل 15 ثانية
  setInterval(() => {
    shippingRoutes.forEach(route => {
      setTimeout(() => {
        animateShipping(map, route, getTransportIcon(route));
      }, route.delay);
    });
  }, 15000);
}

// تحديد أيقونة النقل المناسبة بناءً على المسار
function getTransportIcon(route) {
  // استخدام الطائرة للمسافات الطويلة
  if (route.from === 'china') {
    return L.divIcon({
      html: '<i class="fas fa-plane text-white text-xl animate-bounce"></i>',
      className: 'plane-icon',
      iconSize: [30, 30],
      iconAnchor: [15, 15]
    });
  } 
  // استخدام السفينة للمسارات البحرية
  else if (route.from === 'yemen' && route.to === 'oman') {
    return L.divIcon({
      html: '<i class="fas fa-ship text-blue-300 text-xl animate-bounce"></i>',
      className: 'ship-icon',
      iconSize: [30, 30],
      iconAnchor: [15, 15]
    });
  } 
  // استخدام الشاحنة للمسارات البرية
  else {
    return L.divIcon({
      html: '<i class="fas fa-truck text-yellow-300 text-xl animate-bounce"></i>',
      className: 'truck-icon',
      iconSize: [30, 30],
      iconAnchor: [15, 15]
    });
  }
}

// رسم متحرك لعملية الشحن
function animateShipping(map, route, icon) {
  const fromCoords = targetCountries[route.from].coords;
  const toCoords = targetCountries[route.to].coords;
  
  // إنشاء مؤشر متحرك
  const movingMarker = L.marker(fromCoords, { icon: icon }).addTo(map);
  
  // إنشاء خط المسار
  const pathLine = L.polyline([fromCoords, fromCoords], {
    color: route.color,
    weight: 3,
    opacity: 0.7,
    dashArray: '5, 10'
  }).addTo(map);
  
  // وميض للدولة المصدر
  pulseMarker(targetCountries[route.from].marker);
  
  // تحريك المؤشر على طول المسار
  moveMarkerAlongPath(movingMarker, pathLine, fromCoords, toCoords, route.duration, () => {
    // وميض للدولة الوجهة عند الوصول
    pulseMarker(targetCountries[route.to].marker);
    
    // إزالة المؤشر والمسار بعد الانتهاء
    setTimeout(() => {
      map.removeLayer(movingMarker);
      map.removeLayer(pathLine);
    }, 1000);
  });
}

// تحريك المؤشر على طول المسار
function moveMarkerAlongPath(marker, pathLine, fromCoords, toCoords, duration, callback) {
  const startTime = Date.now();
  const totalDistance = calculateDistance(fromCoords, toCoords);
  
  function animate() {
    const elapsedTime = Date.now() - startTime;
    const progress = Math.min(elapsedTime / duration, 1);
    
    // حساب الموقع الحالي
    const currentLat = fromCoords[0] + (toCoords[0] - fromCoords[0]) * progress;
    const currentLng = fromCoords[1] + (toCoords[1] - fromCoords[1]) * progress;
    const currentPosition = [currentLat, currentLng];
    
    // تحديث موقع المؤشر
    marker.setLatLng(currentPosition);
    
    // تحديث خط المسار
    pathLine.setLatLngs([fromCoords, currentPosition]);
    
    if (progress < 1) {
      requestAnimationFrame(animate);
    } else {
      if (callback) callback();
    }
  }
  
  animate();
}

// حساب المسافة بين نقطتين
function calculateDistance(coord1, coord2) {
  const lat1 = coord1[0];
  const lon1 = coord1[1];
  const lat2 = coord2[0];
  const lon2 = coord2[1];
  
  const R = 6371; // نصف قطر الأرض بالكيلومتر
  const dLat = deg2rad(lat2 - lat1);
  const dLon = deg2rad(lon2 - lon1);
  
  const a = 
    Math.sin(dLat/2) * Math.sin(dLat/2) +
    Math.cos(deg2rad(lat1)) * Math.cos(deg2rad(lat2)) * 
    Math.sin(dLon/2) * Math.sin(dLon/2);
  
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
  const distance = R * c;
  
  return distance;
}

// تحويل الدرجات إلى راديان
function deg2rad(deg) {
  return deg * (Math.PI/180);
}

// تأثير وميض للمؤشر
function pulseMarker(marker) {
  const originalRadius = marker.options.radius;
  const originalOpacity = marker.options.fillOpacity;
  
  // زيادة حجم المؤشر وتقليل الشفافية
  marker.setRadius(originalRadius * 2);
  marker.setStyle({ fillOpacity: 1 });
  
  // العودة إلى الحجم الأصلي
  setTimeout(() => {
    marker.setRadius(originalRadius);
    marker.setStyle({ fillOpacity: originalOpacity });
  }, 1000);
}
