/**
 * ملف معالجة النماذج
 * يحتوي على دوال لمعالجة إرسال النماذج مع تحسين معالجة الأخطاء
 */

/**
 * دالة محسنة لإرسال النموذج
 * @param {HTMLFormElement} form - عنصر النموذج
 * @param {string} url - عنوان API لإرسال البيانات
 * @param {Function} onSuccess - دالة تنفذ عند نجاح العملية
 */
function submitFormWithErrorHandling(form, url, onSuccess) {
    // عرض مؤشر التحميل
    const submitButton = form.querySelector('button[type="submit"]');
    const originalButtonText = submitButton.innerHTML;
    submitButton.disabled = true;
    submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> جاري الحفظ...';
    
    // إزالة رسائل الخطأ السابقة
    const errorMessages = form.querySelectorAll('.error-message');
    errorMessages.forEach(el => el.remove());
    
    // جمع بيانات النموذج
    const formData = new FormData(form);
    
    // طباعة البيانات للتصحيح
    console.log('بيانات النموذج المرسلة:');
    for (let pair of formData.entries()) {
        console.log(pair[0] + ': ' + pair[1]);
    }
    
    // إرسال البيانات
    fetch(url, {
        method: 'POST',
        body: formData
    })
    .then(response => {
        // التحقق من حالة الاستجابة
        if (!response.ok) {
            console.error('خطأ في استجابة الخادم:', response.status, response.statusText);
            throw new Error(`خطأ في استجابة الخادم: ${response.status}`);
        }
        
        // محاولة تحليل الاستجابة كـ JSON
        return response.json().catch(err => {
            console.error('خطأ في تحليل استجابة JSON:', err);
            throw new Error('فشل في تحليل استجابة الخادم');
        });
    })
    .then(data => {
        console.log('استجابة الخادم:', data);
        
        // التحقق من حالة النجاح
        if (data.status === 'success') {
            // استدعاء دالة النجاح
            if (typeof onSuccess === 'function') {
                onSuccess(data);
            }
        } else {
            // عرض رسائل الخطأ من الخادم
            if (data.errors) {
                displayFormErrors(form, data.errors);
            } else {
                throw new Error(data.message || 'حدث خطأ أثناء معالجة البيانات');
            }
        }
    })
    .catch(error => {
        console.error('خطأ في إرسال البيانات:', error);
        
        // عرض رسالة الخطأ للمستخدم
        const errorDiv = document.createElement('div');
        errorDiv.className = 'bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-4 error-message';
        errorDiv.innerHTML = `
            <strong class="font-bold">خطأ!</strong>
            <span class="block sm:inline">${error.message || 'حدث خطأ أثناء إرسال البيانات. يرجى المحاولة مرة أخرى.'}</span>
        `;
        form.prepend(errorDiv);
        
        // التمرير إلى أعلى النموذج لعرض الخطأ
        errorDiv.scrollIntoView({ behavior: 'smooth', block: 'start' });
    })
    .finally(() => {
        // إعادة تفعيل زر الإرسال
        submitButton.disabled = false;
        submitButton.innerHTML = originalButtonText;
    });
}

/**
 * دالة لعرض أخطاء النموذج
 * @param {HTMLFormElement} form - عنصر النموذج
 * @param {Object} errors - كائن يحتوي على أخطاء الحقول
 */
function displayFormErrors(form, errors) {
    // إنشاء رسالة خطأ عامة
    const errorSummary = document.createElement('div');
    errorSummary.className = 'bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-4 error-message';
    errorSummary.innerHTML = '<strong class="font-bold">يرجى تصحيح الأخطاء التالية:</strong><ul class="mt-2 list-disc list-inside"></ul>';
    
    const errorList = errorSummary.querySelector('ul');
    
    // إضافة كل خطأ إلى القائمة وتمييز الحقل المقابل
    Object.keys(errors).forEach(fieldName => {
        const errorMessage = errors[fieldName];
        
        // إضافة الخطأ إلى قائمة الأخطاء
        const listItem = document.createElement('li');
        listItem.textContent = errorMessage;
        errorList.appendChild(listItem);
        
        // تمييز الحقل الذي يحتوي على خطأ
        const field = form.querySelector(`[name="${fieldName}"]`);
        if (field) {
            field.classList.add('border-red-500');
            
            // إضافة رسالة خطأ تحت الحقل
            const fieldContainer = field.closest('.form-group') || field.parentNode;
            const fieldError = document.createElement('p');
            fieldError.className = 'text-red-500 text-xs mt-1 error-message';
            fieldError.textContent = errorMessage;
            fieldContainer.appendChild(fieldError);
            
            // إضافة مستمع حدث لإزالة تمييز الخطأ عند تغيير القيمة
            field.addEventListener('input', function() {
                this.classList.remove('border-red-500');
                const errorMsg = fieldContainer.querySelector('.error-message');
                if (errorMsg) {
                    errorMsg.remove();
                }
            }, { once: true });
        }
    });
    
    // إضافة ملخص الأخطاء إلى أعلى النموذج
    form.prepend(errorSummary);
    
    // التمرير إلى أعلى النموذج لعرض الأخطاء
    errorSummary.scrollIntoView({ behavior: 'smooth', block: 'start' });
}

/**
 * دالة لتهيئة جميع النماذج في الصفحة
 */
function initForms() {
    // تهيئة نموذج إنشاء/تعديل الطلب
    const orderForm = document.getElementById('orderForm');
    if (orderForm) {
        orderForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            // استدعاء الدالة المحسنة لإرسال النموذج
            submitFormWithErrorHandling(this, 'api/save_order.php', function(data) {
                // دالة تنفذ عند نجاح العملية
                Swal.fire({
                    icon: 'success',
                    title: 'تم بنجاح!',
                    text: data.message,
                    confirmButtonText: 'حسناً'
                }).then(() => {
                    // التوجيه إلى صفحة إدارة الطلبات أو عرض الطلب
                    window.location.href = 'orders.php';
                });
            });
        });
    }
}

// تنفيذ الدالة عند اكتمال تحميل الصفحة
document.addEventListener('DOMContentLoaded', initForms);
