/**
 * إدارة جدول الدول
 * يقوم هذا الملف بإنشاء وإدارة جدول الدول مع إمكانية البحث والترتيب
 */

// تهيئة جدول الدول عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
  initCountriesTable();
});

// المتغيرات العامة
let countriesTableData = [];
let sortColumn = 'name';
let sortDirection = 'asc';
let searchQuery = '';

// تهيئة جدول الدول
function initCountriesTable() {
  const countriesTableContainer = document.getElementById('countriesTableContainer');
  if (!countriesTableContainer) return;
  
  // نسخ بيانات الدول
  countriesTableData = [...countriesData];
  
  // إنشاء عناصر الجدول
  createTableElements(countriesTableContainer);
  
  // عرض البيانات
  renderCountriesTable();
  
  // إخفاء شاشة التحميل بعد تحميل الجدول
  setTimeout(function() {
    const tableLoading = document.querySelector('.table-loading');
    if (tableLoading) {
      tableLoading.style.opacity = '0';
      setTimeout(function() {
        tableLoading.style.display = 'none';
      }, 500);
    }
  }, 1000);
}

// إنشاء عناصر الجدول
function createTableElements(container) {
  // إنشاء حاوية البحث
  const searchContainer = document.createElement('div');
  searchContainer.className = 'countries-search-container';
  
  // إنشاء حقل البحث
  const searchInput = document.createElement('input');
  searchInput.type = 'text';
  searchInput.className = 'countries-search-input';
  searchInput.placeholder = 'ابحث عن دولة...';
  searchInput.addEventListener('input', function(e) {
    searchQuery = e.target.value.trim();
    renderCountriesTable();
  });
  
  // إنشاء عداد النتائج
  const countElement = document.createElement('div');
  countElement.className = 'countries-count';
  countElement.id = 'countriesCount';
  
  // إضافة عناصر البحث إلى الحاوية
  searchContainer.appendChild(searchInput);
  searchContainer.appendChild(countElement);
  
  // إنشاء حاوية الجدول
  const tableContainer = document.createElement('div');
  tableContainer.className = 'countries-table-container';
  
  // إنشاء الجدول
  const table = document.createElement('table');
  table.className = 'countries-table';
  table.id = 'countriesTable';
  
  // إنشاء رأس الجدول
  const thead = document.createElement('thead');
  const headerRow = document.createElement('tr');
  
  // تعريف أعمدة الجدول
  const columns = [
    { id: 'name', label: 'الدولة' },
    { id: 'continent', label: 'القارة' },
    { id: 'capital', label: 'العاصمة' },
    { id: 'population', label: 'عدد السكان' },
    { id: 'area', label: 'المساحة (كم²)' },
    { id: 'currency', label: 'العملة' }
  ];
  
  // إنشاء خلايا رأس الجدول
  columns.forEach(column => {
    const th = document.createElement('th');
    th.textContent = column.label;
    th.dataset.column = column.id;
    
    // إضافة أيقونة الترتيب
    const sortIcon = document.createElement('i');
    sortIcon.className = 'fas fa-sort sort-icon';
    sortIcon.id = `sort-${column.id}`;
    th.appendChild(sortIcon);
    
    // إضافة حدث النقر للترتيب
    th.addEventListener('click', function() {
      handleSort(column.id);
    });
    
    headerRow.appendChild(th);
  });
  
  // إضافة صف الرأس إلى الجدول
  thead.appendChild(headerRow);
  table.appendChild(thead);
  
  // إنشاء جسم الجدول
  const tbody = document.createElement('tbody');
  tbody.id = 'countriesTableBody';
  table.appendChild(tbody);
  
  // إضافة الجدول إلى الحاوية
  tableContainer.appendChild(table);
  
  // إنشاء شاشة التحميل
  const loadingElement = document.createElement('div');
  loadingElement.className = 'table-loading';
  const spinner = document.createElement('div');
  spinner.className = 'table-loading-spinner';
  loadingElement.appendChild(spinner);
  tableContainer.appendChild(loadingElement);
  
  // إضافة العناصر إلى الحاوية الرئيسية
  container.appendChild(searchContainer);
  container.appendChild(tableContainer);
}

// عرض بيانات الجدول
function renderCountriesTable() {
  const tbody = document.getElementById('countriesTableBody');
  if (!tbody) return;
  
  // تفريغ الجدول
  tbody.innerHTML = '';
  
  // تطبيق البحث
  let filteredData = countriesTableData;
  if (searchQuery) {
    const query = searchQuery.toLowerCase();
    filteredData = countriesTableData.filter(country => 
      country.name.toLowerCase().includes(query) || 
      country.nameEn.toLowerCase().includes(query) || 
      country.continent.toLowerCase().includes(query) || 
      country.capital.toLowerCase().includes(query) || 
      country.currency.toLowerCase().includes(query)
    );
  }
  
  // تطبيق الترتيب
  filteredData.sort((a, b) => {
    let valueA = a[sortColumn];
    let valueB = b[sortColumn];
    
    // التعامل مع القيم العددية
    if (sortColumn === 'population' || sortColumn === 'area') {
      valueA = Number(valueA);
      valueB = Number(valueB);
    } else {
      // التعامل مع النصوص
      valueA = String(valueA).toLowerCase();
      valueB = String(valueB).toLowerCase();
    }
    
    // تطبيق اتجاه الترتيب
    if (sortDirection === 'asc') {
      return valueA > valueB ? 1 : valueA < valueB ? -1 : 0;
    } else {
      return valueA < valueB ? 1 : valueA > valueB ? -1 : 0;
    }
  });
  
  // تحديث أيقونات الترتيب
  updateSortIcons();
  
  // تحديث عداد النتائج
  const countElement = document.getElementById('countriesCount');
  if (countElement) {
    countElement.textContent = `عدد النتائج: ${filteredData.length}`;
  }
  
  // عرض رسالة في حالة عدم وجود نتائج
  if (filteredData.length === 0) {
    const emptyRow = document.createElement('tr');
    const emptyCell = document.createElement('td');
    emptyCell.colSpan = 6;
    emptyCell.className = 'no-results';
    emptyCell.textContent = 'لا توجد نتائج مطابقة للبحث';
    emptyRow.appendChild(emptyCell);
    tbody.appendChild(emptyRow);
    return;
  }
  
  // إنشاء صفوف الجدول
  filteredData.forEach(country => {
    const row = document.createElement('tr');
    
    // تمييز شركاء الشحن
    if (country.isShippingPartner) {
      row.className = 'shipping-partner';
    }
    
    // إنشاء خلايا البيانات
    const nameCell = document.createElement('td');
    nameCell.textContent = country.name;
    
    const continentCell = document.createElement('td');
    continentCell.textContent = country.continent;
    
    const capitalCell = document.createElement('td');
    capitalCell.textContent = country.capital;
    
    const populationCell = document.createElement('td');
    populationCell.textContent = formatNumber(country.population);
    
    const areaCell = document.createElement('td');
    areaCell.textContent = formatNumber(country.area);
    
    const currencyCell = document.createElement('td');
    currencyCell.textContent = country.currency;
    
    // إضافة الخلايا إلى الصف
    row.appendChild(nameCell);
    row.appendChild(continentCell);
    row.appendChild(capitalCell);
    row.appendChild(populationCell);
    row.appendChild(areaCell);
    row.appendChild(currencyCell);
    
    // إضافة الصف إلى الجدول
    tbody.appendChild(row);
  });
}

// معالجة الترتيب
function handleSort(column) {
  // تغيير اتجاه الترتيب إذا كان نفس العمود
  if (sortColumn === column) {
    sortDirection = sortDirection === 'asc' ? 'desc' : 'asc';
  } else {
    // تعيين العمود الجديد واتجاه الترتيب الافتراضي
    sortColumn = column;
    sortDirection = 'asc';
  }
  
  // إعادة عرض الجدول
  renderCountriesTable();
}

// تحديث أيقونات الترتيب
function updateSortIcons() {
  // إعادة تعيين جميع الأيقونات
  document.querySelectorAll('.sort-icon').forEach(icon => {
    icon.className = 'fas fa-sort sort-icon';
  });
  
  // تحديث أيقونة العمود المحدد
  const activeIcon = document.getElementById(`sort-${sortColumn}`);
  if (activeIcon) {
    activeIcon.className = `fas fa-sort-${sortDirection === 'asc' ? 'up' : 'down'} sort-icon sort-active`;
  }
}

// تنسيق الأرقام
function formatNumber(num) {
  return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}
