/**
 * تأثيرات متقدمة لصفحة تسجيل الدخول
 * يتضمن تأثيرات الجزيئات، التحريك، والتفاعلات
 */

document.addEventListener('DOMContentLoaded', function() {
    // ===== إنشاء تأثير الجزيئات =====
    createParticles();
    
    // ===== تأثيرات الحقول =====
    initializeInputEffects();
    
    // ===== تأثيرات ظهور العناصر =====
    animateElements();
    
    // ===== تأثير الزر =====
    initializeButtonEffects();
    
    // ===== تأثير الخلفية المتحركة =====
    initializeBackgroundEffect();
    
    // ===== تأثير التحقق من صحة النموذج =====
    initializeFormValidation();
});

/**
 * إنشاء تأثير الجزيئات في الخلفية
 */
function createParticles() {
    const particlesContainer = document.querySelector('.particles-container');
    if (!particlesContainer) return;
    
    // إنشاء عدد من الجزيئات
    const particleCount = 20;
    
    for (let i = 0; i < particleCount; i++) {
        const particle = document.createElement('div');
        particle.classList.add('particle');
        
        // تعيين حجم عشوائي
        const size = Math.random() * 15 + 5;
        particle.style.width = `${size}px`;
        particle.style.height = `${size}px`;
        
        // تعيين موقع عشوائي
        const posX = Math.random() * 100;
        const posY = Math.random() * 100;
        particle.style.left = `${posX}%`;
        particle.style.top = `${posY}%`;
        
        // تعيين تأخير عشوائي للحركة
        const delay = Math.random() * 5;
        particle.style.animationDelay = `${delay}s`;
        
        // تعيين مدة عشوائية للحركة
        const duration = Math.random() * 10 + 5;
        particle.style.animationDuration = `${duration}s`;
        
        // تعيين شفافية عشوائية
        const opacity = Math.random() * 0.5 + 0.1;
        particle.style.opacity = opacity;
        
        particlesContainer.appendChild(particle);
    }
}

/**
 * تهيئة تأثيرات الحقول
 */
function initializeInputEffects() {
    const inputs = document.querySelectorAll('.auth-input');
    
    inputs.forEach(input => {
        // تأثير التركيز
        input.addEventListener('focus', function() {
            this.parentElement.classList.add('input-focused');
            
            // تحريك الأيقونة
            const icon = this.nextElementSibling;
            if (icon) {
                icon.style.transform = 'translateY(-50%) scale(1.2)';
                icon.style.color = 'var(--primary-color)';
            }
        });
        
        // إزالة تأثير التركيز
        input.addEventListener('blur', function() {
            if (this.value === '') {
                this.parentElement.classList.remove('input-focused');
            }
            
            // إعادة الأيقونة
            const icon = this.nextElementSibling;
            if (icon) {
                icon.style.transform = 'translateY(-50%) scale(1)';
                icon.style.color = 'rgba(255, 255, 255, 0.7)';
            }
        });
        
        // تأثير الكتابة
        input.addEventListener('input', function() {
            if (this.value !== '') {
                this.parentElement.classList.add('has-text');
            } else {
                this.parentElement.classList.remove('has-text');
            }
        });
    });
}

/**
 * تحريك العناصر عند تحميل الصفحة
 */
function animateElements() {
    const authContainer = document.querySelector('.auth-container');
    const logo = document.querySelector('.site-logo');
    const title = document.querySelector('.auth-title');
    const inputGroups = document.querySelectorAll('.input-group');
    const rememberForgot = document.querySelector('.remember-forgot');
    const button = document.querySelector('.auth-button');
    const authSwitch = document.querySelector('.auth-switch');
    
    // تأثير ظهور الحاوية
    if (authContainer) {
        authContainer.style.opacity = '0';
        authContainer.style.transform = 'translateY(20px)';
        
        setTimeout(() => {
            authContainer.style.transition = 'opacity 0.8s ease, transform 0.8s ease';
            authContainer.style.opacity = '1';
            authContainer.style.transform = 'translateY(0)';
        }, 100);
    }
    
    // تأثير ظهور الشعار
    if (logo) {
        logo.style.opacity = '0';
        logo.style.transform = 'scale(0.8)';
        
        setTimeout(() => {
            logo.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
            logo.style.opacity = '1';
            logo.style.transform = 'scale(1)';
        }, 400);
    }
    
    // تأثير ظهور العنوان
    if (title) {
        title.style.opacity = '0';
        title.style.transform = 'translateY(-10px)';
        
        setTimeout(() => {
            title.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
            title.style.opacity = '1';
            title.style.transform = 'translateY(0)';
        }, 600);
    }
    
    // تأثير ظهور الحقول
    inputGroups.forEach((group, index) => {
        group.style.opacity = '0';
        group.style.transform = 'translateX(10px)';
        
        setTimeout(() => {
            group.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
            group.style.opacity = '1';
            group.style.transform = 'translateX(0)';
        }, 800 + (index * 150));
    });
    
    // تأثير ظهور التذكر ونسيان كلمة المرور
    if (rememberForgot) {
        rememberForgot.style.opacity = '0';
        
        setTimeout(() => {
            rememberForgot.style.transition = 'opacity 0.5s ease';
            rememberForgot.style.opacity = '1';
        }, 1200);
    }
    
    // تأثير ظهور الزر
    if (button) {
        button.style.opacity = '0';
        button.style.transform = 'translateY(10px)';
        
        setTimeout(() => {
            button.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
            button.style.opacity = '1';
            button.style.transform = 'translateY(0)';
            
            // تأثير نبض للزر
            setTimeout(() => {
                button.style.transform = 'scale(1.05)';
                button.style.boxShadow = '0 8px 25px rgba(0, 0, 0, 0.3)';
                
                setTimeout(() => {
                    button.style.transform = 'scale(1)';
                    button.style.boxShadow = '0 5px 15px rgba(0, 0, 0, 0.2)';
                }, 300);
            }, 500);
        }, 1400);
    }
    
    // تأثير ظهور رابط التبديل
    if (authSwitch) {
        authSwitch.style.opacity = '0';
        
        setTimeout(() => {
            authSwitch.style.transition = 'opacity 0.5s ease';
            authSwitch.style.opacity = '1';
        }, 1600);
    }
}

/**
 * تهيئة تأثيرات الزر
 */
function initializeButtonEffects() {
    const button = document.querySelector('.auth-button');
    if (!button) return;
    
    // تأثير تموج عند النقر
    button.addEventListener('click', function(e) {
        // إنشاء عنصر التموج
        const ripple = document.createElement('span');
        ripple.classList.add('ripple');
        this.appendChild(ripple);
        
        // تحديد موقع النقر
        const rect = button.getBoundingClientRect();
        const x = e.clientX - rect.left;
        const y = e.clientY - rect.top;
        
        // تعيين موقع وحجم التموج
        ripple.style.left = `${x}px`;
        ripple.style.top = `${y}px`;
        
        // إزالة عنصر التموج بعد انتهاء التأثير
        setTimeout(() => {
            ripple.remove();
        }, 600);
    });
    
    // تأثير التحويم
    button.addEventListener('mouseenter', function() {
        this.style.transform = 'translateY(-3px)';
        this.style.boxShadow = '0 7px 20px rgba(0, 0, 0, 0.3)';
    });
    
    button.addEventListener('mouseleave', function() {
        this.style.transform = 'translateY(0)';
        this.style.boxShadow = '0 5px 15px rgba(0, 0, 0, 0.2)';
    });
}

/**
 * تهيئة تأثير الخلفية المتحركة
 */
function initializeBackgroundEffect() {
    const container = document.querySelector('.auth-container');
    if (!container) return;
    
    // تأثير حركة الخلفية مع حركة الماوس
    document.addEventListener('mousemove', function(e) {
        const x = e.clientX / window.innerWidth;
        const y = e.clientY / window.innerHeight;
        
        container.style.boxShadow = `
            0 8px 32px rgba(0, 0, 0, 0.3),
            ${x * 20 - 10}px ${y * 20 - 10}px 30px rgba(71, 118, 230, 0.1)
        `;
    });
}

/**
 * تهيئة التحقق من صحة النموذج
 */
function initializeFormValidation() {
    const form = document.getElementById('loginForm');
    if (!form) return;
    
    form.addEventListener('submit', function(e) {
        const emailInput = form.querySelector('input[name="email"]');
        const passwordInput = form.querySelector('input[name="password"]');
        let isValid = true;
        
        // التحقق من البريد الإلكتروني
        if (emailInput && !validateEmail(emailInput.value)) {
            showInputError(emailInput, 'يرجى إدخال بريد إلكتروني صحيح');
            isValid = false;
        } else if (emailInput) {
            removeInputError(emailInput);
        }
        
        // التحقق من كلمة المرور
        if (passwordInput && passwordInput.value.length < 6) {
            showInputError(passwordInput, 'يجب أن تتكون كلمة المرور من 6 أحرف على الأقل');
            isValid = false;
        } else if (passwordInput) {
            removeInputError(passwordInput);
        }
        
        if (!isValid) {
            e.preventDefault();
        }
    });
}

/**
 * التحقق من صحة البريد الإلكتروني
 */
function validateEmail(email) {
    const re = /^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
    return re.test(String(email).toLowerCase());
}

/**
 * إظهار رسالة خطأ للحقل
 */
function showInputError(input, message) {
    const parent = input.parentElement;
    
    // إزالة رسالة خطأ سابقة إن وجدت
    removeInputError(input);
    
    // إضافة فئة الخطأ
    parent.classList.add('input-error');
    
    // إنشاء رسالة الخطأ
    const errorMessage = document.createElement('div');
    errorMessage.classList.add('input-error-message');
    errorMessage.textContent = message;
    
    // إضافة رسالة الخطأ
    parent.appendChild(errorMessage);
}

/**
 * إزالة رسالة خطأ من الحقل
 */
function removeInputError(input) {
    const parent = input.parentElement;
    
    // إزالة فئة الخطأ
    parent.classList.remove('input-error');
    
    // إزالة رسالة الخطأ إن وجدت
    const errorMessage = parent.querySelector('.input-error-message');
    if (errorMessage) {
        errorMessage.remove();
    }
}
