/**
 * إدارة الشريط المتحرك المتقدم
 * يقوم هذا الملف بإنشاء وإدارة الشريط المتحرك المتقدم للدول وأسعار الشحن
 * مع عرض دولة واحدة في كل مرة وتأثيرات انتقالية مميزة
 */

// تهيئة الشريط المتحرك المتقدم عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
  // تحميل بيانات الدول
  loadCountriesForAdvancedTicker();

  // توليد بيانات أسعار الشحن
  generateAdvancedShippingRates();

  // تهيئة الأشرطة المتحركة المتقدمة
  initAdvancedCountriesTicker();
  initAdvancedShippingRatesTicker();
});

// استخدام بيانات الدول من ملف countries-data.js
let advancedCountriesData = [];
let currentCountryIndex = 0;
let countryInterval;

// استخدام بيانات أسعار الشحن
let advancedRatesData = [];
let currentRateIndex = 0;
let rateInterval;

// تحميل بيانات الدول للشريط المتحرك المتقدم
function loadCountriesForAdvancedTicker() {
  // التأكد من وجود بيانات الدول
  if (typeof countriesData !== 'undefined' && Array.isArray(countriesData)) {
    // تحويل بيانات الدول إلى التنسيق المطلوب للشريط المتحرك المتقدم
    advancedCountriesData = countriesData.map(country => {
      return {
        code: country.code,
        name: country.name,
        nameEn: country.nameEn || '',
        continent: country.continent || '',
        capital: country.capital || '',
        currency: country.currency || '',
        flag: `https://flagcdn.com/w160/${country.code.toLowerCase()}.png`
      };
    });
  } else {
    // في حالة عدم وجود بيانات الدول، استخدام بيانات افتراضية
    advancedCountriesData = [
      {
        code: "CN",
        name: "الصين",
        nameEn: "China",
        continent: "آسيا",
        capital: "بكين",
        currency: "يوان صيني",
        flag: "https://flagcdn.com/w160/cn.png"
      },
      {
        code: "YE",
        name: "اليمن",
        nameEn: "Yemen",
        continent: "آسيا",
        capital: "صنعاء",
        currency: "ريال يمني",
        flag: "https://flagcdn.com/w160/ye.png"
      },
      {
        code: "OM",
        name: "عمان",
        nameEn: "Oman",
        continent: "آسيا",
        capital: "مسقط",
        currency: "ريال عماني",
        flag: "https://flagcdn.com/w160/om.png"
      },
      {
        code: "SA",
        name: "السعودية",
        nameEn: "Saudi Arabia",
        continent: "آسيا",
        capital: "الرياض",
        currency: "ريال سعودي",
        flag: "https://flagcdn.com/w160/sa.png"
      },
      {
        code: "AE",
        name: "الإمارات العربية المتحدة",
        nameEn: "United Arab Emirates",
        continent: "آسيا",
        capital: "أبوظبي",
        currency: "درهم إماراتي",
        flag: "https://flagcdn.com/w160/ae.png"
      }
    ];
  }
}

// توليد بيانات أسعار الشحن للشريط المتحرك المتقدم
function generateAdvancedShippingRates() {
  // إنشاء قائمة بالدول الرئيسية للشحن
  const mainShippingCountries = ["YE", "SA", "OM", "AE", "CN"];

  // توليد أسعار الشحن
  advancedRatesData = [];

  // إضافة المسارات الثابتة المهمة
  const fixedRoutes = [
    {
      route: "الصين - اليمن",
      from: "الصين",
      to: "اليمن",
      price: 25,
      unit: "دولار / كجم",
      icon: "fa-plane",
      duration: "5-7 أيام",
      distance: "7,800 كم"
    },
    {
      route: "الصين - السعودية",
      from: "الصين",
      to: "السعودية",
      price: 22,
      unit: "دولار / كجم",
      icon: "fa-plane",
      duration: "4-6 أيام",
      distance: "6,900 كم"
    },
    {
      route: "اليمن - عمان",
      from: "اليمن",
      to: "عمان",
      price: 15,
      unit: "دولار / كجم",
      icon: "fa-ship",
      duration: "2-3 أيام",
      distance: "1,200 كم"
    },
    {
      route: "السعودية - اليمن",
      from: "السعودية",
      to: "اليمن",
      price: 10,
      unit: "دولار / كجم",
      icon: "fa-truck",
      duration: "1-2 أيام",
      distance: "800 كم"
    },
    {
      route: "الإمارات - اليمن",
      from: "الإمارات",
      to: "اليمن",
      price: 18,
      unit: "دولار / كجم",
      icon: "fa-plane",
      duration: "3-4 أيام",
      distance: "1,500 كم"
    }
  ];

  // إضافة المسارات الثابتة
  advancedRatesData = [...fixedRoutes];

  // إضافة مسارات إضافية من الدول الرئيسية إلى دول أخرى
  if (advancedCountriesData.length > 0) {
    // اختيار بعض الدول العشوائية (بحد أقصى 20 دولة)
    const maxCountries = Math.min(20, advancedCountriesData.length);
    const randomCountries = [...advancedCountriesData]
      .sort(() => 0.5 - Math.random())
      .slice(0, maxCountries);

    randomCountries.forEach(destCountry => {
      // تجاهل الدول الرئيسية كوجهات (لتجنب التكرار)
      if (!mainShippingCountries.includes(destCountry.code)) {
        // اختيار دولة رئيسية عشوائية كمصدر
        const sourceIndex = Math.floor(Math.random() * mainShippingCountries.length);
        const sourceCode = mainShippingCountries[sourceIndex];
        const sourceCountry = advancedCountriesData.find(c => c.code === sourceCode);

        if (sourceCountry) {
          // توليد سعر عشوائي بين 10 و 35
          const price = Math.floor(Math.random() * 25) + 10;

          // اختيار وسيلة نقل عشوائية
          const transportIcons = ["fa-plane", "fa-ship", "fa-truck"];
          const iconIndex = Math.floor(Math.random() * transportIcons.length);

          // توليد مدة شحن عشوائية
          const minDays = Math.floor(Math.random() * 5) + 1;
          const maxDays = minDays + Math.floor(Math.random() * 3) + 1;
          const duration = `${minDays}-${maxDays} أيام`;

          // توليد مسافة عشوائية
          const distance = `${Math.floor(Math.random() * 8000) + 500} كم`;

          // إضافة مسار الشحن
          advancedRatesData.push({
            route: `${sourceCountry.name} - ${destCountry.name}`,
            from: sourceCountry.name,
            to: destCountry.name,
            price: price,
            unit: "دولار / كجم",
            icon: transportIcons[iconIndex],
            duration: duration,
            distance: distance
          });
        }
      }
    });
  }
}

// تهيئة شريط الدول المتحرك المتقدم
function initAdvancedCountriesTicker() {
  const tickerContainer = document.getElementById('advancedCountriesContainer');
  if (!tickerContainer) return;

  // تفريغ المحتوى الحالي
  tickerContainer.innerHTML = '';

  // إضافة عنوان الشريط
  const tickerTitle = document.createElement('div');
  tickerTitle.className = 'advanced-ticker-title';
  tickerTitle.textContent = 'الدول المتاحة للشحن';
  tickerTitle.style.fontSize = '14px'; // تصغير حجم الخط
  tickerContainer.appendChild(tickerTitle);

  // إنشاء حاوية العرض
  const displayContainer = document.createElement('div');
  displayContainer.className = 'advanced-ticker-display';

  // إضافة عناصر الدول
  advancedCountriesData.forEach((country, index) => {
    const countryDisplay = document.createElement('div');
    countryDisplay.className = `country-display ${index === 0 ? 'active' : ''}`;
    countryDisplay.dataset.index = index;

    // إضافة علم الدولة
    const flag = document.createElement('img');
    flag.className = 'country-flag-advanced';
    flag.src = country.flag;
    flag.alt = `علم ${country.name}`;

    // إضافة اسم الدولة
    const name = document.createElement('div');
    name.className = 'country-name-advanced';
    name.textContent = country.name;

    // إضافة معلومات إضافية
    const infoContainer = document.createElement('div');
    infoContainer.className = 'country-info';

    // إضافة العاصمة
    if (country.capital) {
      const capitalInfo = document.createElement('div');
      capitalInfo.className = 'info-item';

      const capitalLabel = document.createElement('div');
      capitalLabel.className = 'info-label';
      capitalLabel.textContent = 'العاصمة';

      const capitalValue = document.createElement('div');
      capitalValue.className = 'info-value';
      capitalValue.textContent = country.capital;

      capitalInfo.appendChild(capitalLabel);
      capitalInfo.appendChild(capitalValue);
      infoContainer.appendChild(capitalInfo);
    }

    // إضافة القارة
    if (country.continent) {
      const continentInfo = document.createElement('div');
      continentInfo.className = 'info-item';

      const continentLabel = document.createElement('div');
      continentLabel.className = 'info-label';
      continentLabel.textContent = 'القارة';

      const continentValue = document.createElement('div');
      continentValue.className = 'info-value';
      continentValue.textContent = country.continent;

      continentInfo.appendChild(continentLabel);
      continentInfo.appendChild(continentValue);
      infoContainer.appendChild(continentInfo);
    }

    // إضافة العملة
    if (country.currency) {
      const currencyInfo = document.createElement('div');
      currencyInfo.className = 'info-item';

      const currencyLabel = document.createElement('div');
      currencyLabel.className = 'info-label';
      currencyLabel.textContent = 'العملة';

      const currencyValue = document.createElement('div');
      currencyValue.className = 'info-value';
      currencyValue.textContent = country.currency;

      currencyInfo.appendChild(currencyLabel);
      currencyInfo.appendChild(currencyValue);
      infoContainer.appendChild(currencyInfo);
    }

    // إضافة العناصر إلى عنصر الدولة
    countryDisplay.appendChild(flag);
    countryDisplay.appendChild(name);
    countryDisplay.appendChild(infoContainer);

    // إضافة عنصر الدولة إلى حاوية العرض
    displayContainer.appendChild(countryDisplay);
  });

  // إضافة أزرار التنقل
  const prevButton = document.createElement('div');
  prevButton.className = 'ticker-arrow prev';
  prevButton.innerHTML = '<i class="fas fa-chevron-right"></i>';
  prevButton.addEventListener('click', () => showPrevCountry());

  const nextButton = document.createElement('div');
  nextButton.className = 'ticker-arrow next';
  nextButton.innerHTML = '<i class="fas fa-chevron-left"></i>';
  nextButton.addEventListener('click', () => showNextCountry());

  // إضافة نقاط التنقل
  const navDots = document.createElement('div');
  navDots.className = 'ticker-nav';

  advancedCountriesData.forEach((_, index) => {
    const dot = document.createElement('div');
    dot.className = `ticker-nav-dot ${index === 0 ? 'active' : ''}`;
    dot.dataset.index = index;
    dot.addEventListener('click', () => showCountry(index));
    navDots.appendChild(dot);
  });

  // إضافة العناصر إلى الحاوية
  tickerContainer.appendChild(displayContainer);
  tickerContainer.appendChild(prevButton);
  tickerContainer.appendChild(nextButton);
  tickerContainer.appendChild(navDots);

  // بدء التبديل التلقائي
  startCountryAutoRotation();
}

// عرض الدولة التالية
function showNextCountry() {
  stopCountryAutoRotation();
  currentCountryIndex = (currentCountryIndex + 1) % advancedCountriesData.length;
  updateCountryDisplay();
  startCountryAutoRotation();
}

// عرض الدولة السابقة
function showPrevCountry() {
  stopCountryAutoRotation();
  currentCountryIndex = (currentCountryIndex - 1 + advancedCountriesData.length) % advancedCountriesData.length;
  updateCountryDisplay();
  startCountryAutoRotation();
}

// عرض دولة محددة
function showCountry(index) {
  stopCountryAutoRotation();
  currentCountryIndex = index;
  updateCountryDisplay();
  startCountryAutoRotation();
}

// تحديث عرض الدولة
function updateCountryDisplay() {
  const container = document.getElementById('advancedCountriesContainer');
  if (!container) return;

  // تحديث عناصر الدول
  const countryElements = container.querySelectorAll('.country-display');
  countryElements.forEach((element, index) => {
    if (index === currentCountryIndex) {
      element.classList.add('active');
    } else {
      element.classList.remove('active');
    }
  });

  // تحديث نقاط التنقل
  const navDots = container.querySelectorAll('.ticker-nav-dot');
  navDots.forEach((dot, index) => {
    if (index === currentCountryIndex) {
      dot.classList.add('active');
    } else {
      dot.classList.remove('active');
    }
  });
}

// بدء التبديل التلقائي للدول
function startCountryAutoRotation() {
  stopCountryAutoRotation();
  countryInterval = setInterval(() => {
    currentCountryIndex = (currentCountryIndex + 1) % advancedCountriesData.length;
    updateCountryDisplay();
  }, 3000); // تسريع التمرير الأوتوماتيكي أكثر
}

// إيقاف التبديل التلقائي للدول
function stopCountryAutoRotation() {
  if (countryInterval) {
    clearInterval(countryInterval);
  }
}

// تهيئة شريط أسعار الشحن المتحرك المتقدم
function initAdvancedShippingRatesTicker() {
  const ratesContainer = document.getElementById('advancedRatesContainer');
  if (!ratesContainer) return;

  // تفريغ المحتوى الحالي
  ratesContainer.innerHTML = '';

  // إضافة عنوان الشريط
  const ratesTitle = document.createElement('div');
  ratesTitle.className = 'advanced-rates-title';
  ratesTitle.textContent = 'أسعار الشحن';
  ratesTitle.style.fontSize = '14px'; // تصغير حجم الخط
  ratesContainer.appendChild(ratesTitle);

  // إنشاء حاوية العرض
  const displayContainer = document.createElement('div');
  displayContainer.className = 'advanced-ticker-display';

  // إضافة عناصر أسعار الشحن
  advancedRatesData.forEach((rate, index) => {
    const rateDisplay = document.createElement('div');
    rateDisplay.className = `rate-display ${index === 0 ? 'active' : ''}`;
    rateDisplay.dataset.index = index;

    // إضافة أيقونة وسيلة النقل
    const icon = document.createElement('i');
    icon.className = `fas ${rate.icon} transport-icon-advanced`;

    // إضافة اسم المسار
    const route = document.createElement('div');
    route.className = 'route-name-advanced';
    route.textContent = rate.route;

    // إضافة سعر الشحن
    const price = document.createElement('div');
    price.className = 'shipping-price-advanced';
    price.textContent = rate.price;

    // إضافة وحدة السعر
    const unit = document.createElement('div');
    unit.className = 'price-unit-advanced';
    unit.textContent = rate.unit;

    // إضافة معلومات إضافية
    const infoContainer = document.createElement('div');
    infoContainer.className = 'country-info';

    // إضافة مدة الشحن
    if (rate.duration) {
      const durationInfo = document.createElement('div');
      durationInfo.className = 'info-item';

      const durationLabel = document.createElement('div');
      durationLabel.className = 'info-label';
      durationLabel.textContent = 'مدة الشحن';

      const durationValue = document.createElement('div');
      durationValue.className = 'info-value';
      durationValue.textContent = rate.duration;

      durationInfo.appendChild(durationLabel);
      durationInfo.appendChild(durationValue);
      infoContainer.appendChild(durationInfo);
    }

    // إضافة المسافة
    if (rate.distance) {
      const distanceInfo = document.createElement('div');
      distanceInfo.className = 'info-item';

      const distanceLabel = document.createElement('div');
      distanceLabel.className = 'info-label';
      distanceLabel.textContent = 'المسافة';

      const distanceValue = document.createElement('div');
      distanceValue.className = 'info-value';
      distanceValue.textContent = rate.distance;

      distanceInfo.appendChild(distanceLabel);
      distanceInfo.appendChild(distanceValue);
      infoContainer.appendChild(distanceInfo);
    }

    // إضافة العناصر إلى عنصر السعر
    rateDisplay.appendChild(icon);
    rateDisplay.appendChild(route);
    rateDisplay.appendChild(price);
    rateDisplay.appendChild(unit);
    rateDisplay.appendChild(infoContainer);

    // إضافة عنصر السعر إلى حاوية العرض
    displayContainer.appendChild(rateDisplay);
  });

  // إضافة أزرار التنقل
  const prevButton = document.createElement('div');
  prevButton.className = 'ticker-arrow prev';
  prevButton.innerHTML = '<i class="fas fa-chevron-right"></i>';
  prevButton.addEventListener('click', () => showPrevRate());

  const nextButton = document.createElement('div');
  nextButton.className = 'ticker-arrow next';
  nextButton.innerHTML = '<i class="fas fa-chevron-left"></i>';
  nextButton.addEventListener('click', () => showNextRate());

  // إضافة نقاط التنقل
  const navDots = document.createElement('div');
  navDots.className = 'ticker-nav';

  advancedRatesData.forEach((_, index) => {
    const dot = document.createElement('div');
    dot.className = `ticker-nav-dot ${index === 0 ? 'active' : ''}`;
    dot.dataset.index = index;
    dot.addEventListener('click', () => showRate(index));
    navDots.appendChild(dot);
  });

  // إضافة العناصر إلى الحاوية
  ratesContainer.appendChild(displayContainer);
  ratesContainer.appendChild(prevButton);
  ratesContainer.appendChild(nextButton);
  ratesContainer.appendChild(navDots);

  // بدء التبديل التلقائي
  startRateAutoRotation();
}

// عرض السعر التالي
function showNextRate() {
  stopRateAutoRotation();
  currentRateIndex = (currentRateIndex + 1) % advancedRatesData.length;
  updateRateDisplay();
  startRateAutoRotation();
}

// عرض السعر السابق
function showPrevRate() {
  stopRateAutoRotation();
  currentRateIndex = (currentRateIndex - 1 + advancedRatesData.length) % advancedRatesData.length;
  updateRateDisplay();
  startRateAutoRotation();
}

// عرض سعر محدد
function showRate(index) {
  stopRateAutoRotation();
  currentRateIndex = index;
  updateRateDisplay();
  startRateAutoRotation();
}

// تحديث عرض السعر
function updateRateDisplay() {
  const container = document.getElementById('advancedRatesContainer');
  if (!container) return;

  // تحديث عناصر الأسعار
  const rateElements = container.querySelectorAll('.rate-display');
  rateElements.forEach((element, index) => {
    if (index === currentRateIndex) {
      element.classList.add('active');
    } else {
      element.classList.remove('active');
    }
  });

  // تحديث نقاط التنقل
  const navDots = container.querySelectorAll('.ticker-nav-dot');
  navDots.forEach((dot, index) => {
    if (index === currentRateIndex) {
      dot.classList.add('active');
    } else {
      dot.classList.remove('active');
    }
  });
}

// بدء التبديل التلقائي للأسعار
function startRateAutoRotation() {
  stopRateAutoRotation();
  rateInterval = setInterval(() => {
    currentRateIndex = (currentRateIndex + 1) % advancedRatesData.length;
    updateRateDisplay();
  }, 3500); // تسريع التمرير الأوتوماتيكي أكثر
}

// إيقاف التبديل التلقائي للأسعار
function stopRateAutoRotation() {
  if (rateInterval) {
    clearInterval(rateInterval);
  }
}
