<?php
/**
 * API لتحديث حالة المستخدم (قبول أو رفض)
 */

// بدء الجلسة
session_start();

// التحقق من تسجيل دخول المستخدم
if (!isset($_SESSION['user_id'])) {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'يجب تسجيل الدخول للوصول إلى هذه البيانات'
    ]);
    exit;
}

// التحقق من صلاحيات المستخدم (مسموح فقط للمدير)
if ($_SESSION['role'] !== 'admin') {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'ليس لديك صلاحية الوصول إلى هذه البيانات'
    ]);
    exit;
}

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/db_connection.php';

// الحصول على البيانات المرسلة
$data = json_decode(file_get_contents('php://input'), true);

// التحقق من وجود البيانات المطلوبة
if (!isset($data['user_id']) || !isset($data['action'])) {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'البيانات المرسلة غير مكتملة'
    ]);
    exit;
}

// التحقق من صحة الإجراء
if ($data['action'] !== 'approve' && $data['action'] !== 'reject') {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'الإجراء غير صالح'
    ]);
    exit;
}

try {
    // بدء المعاملة
    $conn->beginTransaction();
    
    // التحقق من وجود المستخدم
    $stmt = $conn->prepare("
        SELECT id, username, email, full_name
        FROM users
        WHERE id = ? AND is_active = 0 AND is_deleted = 0
    ");
    $stmt->execute([$data['user_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        // إلغاء المعاملة
        $conn->rollBack();
        
        header('Content-Type: application/json');
        echo json_encode([
            'status' => 'error',
            'message' => 'المستخدم غير موجود أو تمت معالجة طلبه بالفعل'
        ]);
        exit;
    }
    
    if ($data['action'] === 'approve') {
        // الموافقة على المستخدم
        $stmt = $conn->prepare("
            UPDATE users
            SET is_active = 1, updated_at = NOW(), approved_by = ?
            WHERE id = ?
        ");
        $stmt->execute([$_SESSION['user_id'], $data['user_id']]);
        
        // إرسال بريد إلكتروني للمستخدم (يمكن إضافة هذه الوظيفة لاحقًا)
        // sendApprovalEmail($user['email'], $user['full_name']);
        
        $message = 'تمت الموافقة على المستخدم بنجاح';
    } else {
        // رفض المستخدم (حذف منطقي)
        $stmt = $conn->prepare("
            UPDATE users
            SET is_deleted = 1, updated_at = NOW(), deleted_by = ?
            WHERE id = ?
        ");
        $stmt->execute([$_SESSION['user_id'], $data['user_id']]);
        
        // إرسال بريد إلكتروني للمستخدم (يمكن إضافة هذه الوظيفة لاحقًا)
        // sendRejectionEmail($user['email'], $user['full_name']);
        
        $message = 'تم رفض المستخدم بنجاح';
    }
    
    // تسجيل الإجراء في سجل النشاط
    $action = $data['action'] === 'approve' ? 'approve_user' : 'reject_user';
    $stmt = $conn->prepare("
        INSERT INTO activity_log (user_id, action, target_id, target_type, details, created_at)
        VALUES (?, ?, ?, 'user', ?, NOW())
    ");
    $details = json_encode([
        'username' => $user['username'],
        'email' => $user['email'],
        'full_name' => $user['full_name']
    ]);
    $stmt->execute([$_SESSION['user_id'], $action, $data['user_id'], $details]);
    
    // تأكيد المعاملة
    $conn->commit();
    
    // إرجاع استجابة النجاح
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'message' => $message
    ]);
} catch (PDOException $e) {
    // التراجع عن المعاملة في حالة حدوث خطأ
    $conn->rollBack();
    
    // تسجيل الخطأ
    error_log('خطأ في قاعدة البيانات: ' . $e->getMessage());
    
    // إرجاع رسالة الخطأ
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'حدث خطأ أثناء تحديث حالة المستخدم'
    ]);
}
?>
