<?php
/**
 * API لتحديث حالة الطلب
 * يستقبل هذا الملف طلب POST يحتوي على معرف الطلب والحالة الجديدة
 * ويقوم بتحديث حالة الطلب في قاعدة البيانات
 *
 * @version 1.0
 * @author فريق التطوير
 */

// بدء الجلسة
session_start();

// التحقق من حالة تسجيل الدخول
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    // إرجاع رسالة خطأ
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'غير مصرح لك بالوصول']);
    exit;
}

// التحقق من طريقة الطلب
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    // إرجاع رسالة خطأ
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'طريقة الطلب غير صحيحة']);
    exit;
}

// الحصول على البيانات المرسلة
$data = json_decode(file_get_contents('php://input'), true);

// التحقق من وجود البيانات المطلوبة
if (!isset($data['order_id']) || !isset($data['status'])) {
    // إرجاع رسالة خطأ
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'البيانات المرسلة غير مكتملة']);
    exit;
}

// تنظيف البيانات
$orderId = filter_var($data['order_id'], FILTER_SANITIZE_NUMBER_INT);
$status = filter_var($data['status'], FILTER_SANITIZE_STRING);

// التحقق من صحة البيانات
if (empty($orderId) || empty($status)) {
    // إرجاع رسالة خطأ
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'البيانات المرسلة غير صحيحة']);
    exit;
}

// التحقق من صحة الحالة
$validStatuses = ['تم التسليم', 'قيد المعالجة', 'في الطريق', 'ملغى'];
if (!in_array($status, $validStatuses)) {
    // إرجاع رسالة خطأ
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'الحالة المرسلة غير صحيحة']);
    exit;
}

// استدعاء ملف الاتصال بقاعدة البيانات
require_once '../includes/db.php';

try {
    // التحقق من وجود الطلب
    $checkSql = "SELECT id FROM orders WHERE id = ?";
    $checkStmt = $conn->prepare($checkSql);
    $checkStmt->execute([$orderId]);
    
    if ($checkStmt->rowCount() === 0) {
        // إرجاع رسالة خطأ
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'message' => 'الطلب غير موجود']);
        exit;
    }
    
    // تحديث حالة الطلب
    $updateSql = "UPDATE orders SET status = ?, updated_at = NOW() WHERE id = ?";
    $updateStmt = $conn->prepare($updateSql);
    $updateStmt->execute([$status, $orderId]);
    
    // إضافة سجل في جدول سجل التغييرات
    $logSql = "INSERT INTO order_logs (order_id, user_id, action, details, created_at) VALUES (?, ?, ?, ?, NOW())";
    $logStmt = $conn->prepare($logSql);
    $logStmt->execute([
        $orderId,
        $_SESSION['user_id'],
        'تغيير الحالة',
        'تم تغيير حالة الطلب إلى: ' . $status
    ]);
    
    // إرجاع رسالة نجاح
    header('Content-Type: application/json');
    echo json_encode(['success' => true, 'message' => 'تم تحديث حالة الطلب بنجاح']);
} catch (PDOException $e) {
    // تسجيل الخطأ
    error_log("خطأ في تحديث حالة الطلب: " . $e->getMessage());
    
    // إرجاع رسالة خطأ
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'حدث خطأ أثناء تحديث حالة الطلب']);
}
