<?php
/**
 * ملف معالجة إنشاء/تعديل الطلبات
 */

// تفعيل عرض الأخطاء للتصحيح
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// تضمين ملف الاتصال بقاعدة البيانات
require_once '../config/db_connection.php';

// بدء الجلسة
session_start();

// التحقق من تسجيل دخول المستخدم
if (!isset($_SESSION['user_id'])) {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'يجب تسجيل الدخول لإنشاء طلب'
    ]);
    exit;
}

// التحقق من أن الطلب هو طلب POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'يجب استخدام طريقة POST'
    ]);
    exit;
}

// تسجيل الطلب للتصحيح
error_log("تم استلام طلب POST في save_order.php");

// تحديد نوع البيانات المستلمة (JSON أو FormData)
$contentType = isset($_SERVER['CONTENT_TYPE']) ? $_SERVER['CONTENT_TYPE'] : '';
$isFormData = strpos($contentType, 'multipart/form-data') !== false || strpos($contentType, 'application/x-www-form-urlencoded') !== false;

if ($isFormData) {
    // استخدام $_POST مباشرة
    $orderData = $_POST;
    error_log("تم استلام بيانات النموذج: " . print_r($orderData, true));
} else {
    // الحصول على بيانات الطلب من الطلب كـ JSON
    $postData = file_get_contents('php://input');
    error_log("البيانات المستلمة كـ JSON: " . $postData);

    $orderData = json_decode($postData, true);

    // التحقق من صحة البيانات
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("خطأ في تحليل JSON: " . json_last_error_msg());
        header('Content-Type: application/json');
        echo json_encode([
            'status' => 'error',
            'message' => 'بيانات JSON غير صالحة: ' . json_last_error_msg()
        ]);
        exit;
    }
}

error_log("تم تحليل البيانات بنجاح: " . print_r($orderData, true));

// دالة للتحقق من البيانات
function validateOrderData($data) {
    $errors = [];

    // التحقق من الحقول المطلوبة
    $requiredFields = [
        'client_name' => 'اسم العميل',
        'service_type' => 'نوع الخدمة',
        'goods_name' => 'اسم البضاعة',
        'goods_type' => 'نوع البضاعة',
        'departure_area' => 'منطقة المغادرة',
        'delivery_area' => 'منطقة التسليم',
        'recipient_name' => 'اسم المستلم',
        'phone_number' => 'رقم الهاتف',
        'employee_name' => 'اسم الموظف المسؤول'
    ];

    foreach ($requiredFields as $field => $label) {
        // التحقق من الحقول في كلا التنسيقين (camelCase و snake_case)
        $camelCaseField = lcfirst(str_replace('_', '', ucwords($field, '_')));

        if (empty($data[$field]) && empty($data[$camelCaseField])) {
            $errors[$field] = "حقل $label مطلوب";
        }
    }

    // التحقق من صحة القيم العددية
    $numericFields = [
        'weight' => 'الوزن',
        'service_fees' => 'رسوم الخدمة',
        'amount_paid' => 'المبلغ المدفوع',
        'amount_agreed_with_customer' => 'المبلغ المتفق مع العميل',
        'amount_agreed_with_agent' => 'المبلغ المتفق مع الوكيل'
    ];

    foreach ($numericFields as $field => $label) {
        $camelCaseField = lcfirst(str_replace('_', '', ucwords($field, '_')));
        $value = $data[$field] ?? $data[$camelCaseField] ?? null;

        if ($value !== null && $value !== '' && !is_numeric($value)) {
            $errors[$field] = "حقل $label يجب أن يكون قيمة رقمية";
        }
    }

    return $errors;
}

// التحقق من صحة البيانات
$errors = validateOrderData($orderData);

if (!empty($errors)) {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => 'يوجد أخطاء في البيانات المدخلة',
        'errors' => $errors
    ]);
    exit;
}

try {
    error_log("بدء محاولة حفظ الطلب في قاعدة البيانات");

    // بدء المعاملة
    $conn->beginTransaction();
    error_log("تم بدء المعاملة");

    // التحقق من نوع الطلب (إنشاء أو تعديل)
    $isUpdate = isset($orderData['order_id']) && !empty($orderData['order_id']) ||
                isset($orderData['orderId']) && !empty($orderData['orderId']);

    $orderId = $orderData['order_id'] ?? $orderData['orderId'] ?? null;

    // البحث عن معرف الفرع
    $branchId = null;
    if (!empty($orderData['branch_id'])) {
        $branchId = $orderData['branch_id'];
    } elseif (!empty($orderData['branchId'])) {
        $branchId = $orderData['branchId'];
    } elseif (!empty($orderData['branch'])) {
        $stmt = $conn->prepare("SELECT id FROM branches WHERE name = ?");
        $stmt->execute([$orderData['branch']]);
        $branch = $stmt->fetch();
        if ($branch) {
            $branchId = $branch['id'];
        }
    }

    // تحويل التواريخ إلى الصيغة المناسبة
    $uploadDate = null;
    if (!empty($orderData['order_upload_date'])) {
        $uploadDate = date('Y-m-d', strtotime($orderData['order_upload_date']));
    } elseif (!empty($orderData['orderUploadDate'])) {
        $uploadDate = date('Y-m-d', strtotime($orderData['orderUploadDate']));
    }

    $deliveryDate = null;
    if (!empty($orderData['scheduled_delivery_date'])) {
        $deliveryDate = date('Y-m-d', strtotime($orderData['scheduled_delivery_date']));
    } elseif (!empty($orderData['scheduledDeliveryDate'])) {
        $deliveryDate = date('Y-m-d', strtotime($orderData['scheduledDeliveryDate']));
    }

    // تحضير البيانات للإدراج أو التحديث
    $data = [
        'user_id' => $_SESSION['user_id'],
        'client_name' => $orderData['client_name'] ?? $orderData['clientName'] ?? null,
        'service_type' => $orderData['service_type'] ?? $orderData['serviceType'] ?? null,
        'branch_id' => $branchId,
        'departure_area' => $orderData['departure_area'] ?? $orderData['departureArea'] ?? null,
        'second_destination' => $orderData['second_destination'] ?? $orderData['secondDestination'] ?? null,
        'delivery_area' => $orderData['delivery_area'] ?? $orderData['deliveryArea'] ?? null,
        'goods_name' => $orderData['goods_name'] ?? $orderData['goodsName'] ?? null,
        'goods_type' => $orderData['goods_type'] ?? $orderData['goodsType'] ?? null,
        'country_of_origin' => $orderData['country_of_origin'] ?? $orderData['countryOfOrigin'] ?? null,
        'weight' => $orderData['weight'] ?? null,
        'quantity' => $orderData['quantity'] ?? null,
        'unit' => $orderData['unit'] ?? null,
        'service_fees' => $orderData['service_fees'] ?? $orderData['serviceFees'] ?? null,
        'currency' => $orderData['currency'] ?? null,
        'payment_type' => $orderData['payment_type'] ?? $orderData['paymentType'] ?? null,
        'delivery_duration' => $orderData['delivery_duration'] ?? $orderData['deliveryDuration'] ?? null,
        'amount_paid' => $orderData['amount_paid'] ?? $orderData['amountPaid'] ?? null,
        'remaining_amount' => $orderData['remaining_amount'] ?? $orderData['remainingAmount'] ?? null,
        'order_upload_date' => $uploadDate ?? date('Y-m-d'), // استخدام تاريخ اليوم إذا لم يتم تحديد تاريخ
        'scheduled_delivery_date' => $deliveryDate,
        'recipient_name' => $orderData['recipient_name'] ?? $orderData['recipientName'] ?? null,
        'phone_number' => $orderData['phone_number'] ?? $orderData['phoneNumber'] ?? null,
        'delivery_company' => $orderData['delivery_company'] ?? $orderData['deliveryCompany'] ?? null,
        'employee_name' => $orderData['employee_name'] ?? $orderData['employeeName'] ?? null,
        'notes' => $orderData['notes'] ?? null,
        'amount_agreed_with_customer' => $orderData['amount_agreed_with_customer'] ?? $orderData['amountAgreedWithCustomer'] ?? $orderData['serviceFees'] ?? $orderData['service_fees'] ?? null, // استخدام رسوم الخدمة كقيمة افتراضية
        'amount_agreed_with_agent' => $orderData['amount_agreed_with_agent'] ?? $orderData['amountAgreedWithAgent'] ?? null,
        'other_expenses' => $orderData['other_expenses'] ?? $orderData['otherExpenses'] ?? 0, // استخدام 0 كقيمة افتراضية
        'profit' => $orderData['profit'] ?? null,
        'payment_status' => $orderData['payment_status'] ?? $orderData['paymentStatus'] ?? 'غير مدفوع',
        'status' => $orderData['status'] ?? 'قيد المعالجة'
    ];

    // حساب الربح إذا لم يتم تحديده
    if (empty($data['profit']) && !empty($data['amount_agreed_with_customer']) && !empty($data['amount_agreed_with_agent'])) {
        $data['profit'] = floatval($data['amount_agreed_with_customer']) - floatval($data['amount_agreed_with_agent']) - floatval($data['other_expenses']);
    }

    // حساب المبلغ المتبقي إذا لم يتم تحديده
    if (empty($data['remaining_amount']) && !empty($data['amount_agreed_with_customer']) && !empty($data['amount_paid'])) {
        $data['remaining_amount'] = floatval($data['amount_agreed_with_customer']) - floatval($data['amount_paid']);
    }

    error_log("البيانات المعدة للإدراج/التحديث: " . print_r($data, true));

    if ($isUpdate) {
        // تحديث طلب موجود
        $updateFields = [];
        $params = [];

        foreach ($data as $field => $value) {
            if ($field !== 'user_id') { // لا نقوم بتحديث معرف المستخدم
                $updateFields[] = "$field = :$field";
                $params[":$field"] = $value;
            }
        }

        // إضافة معرف الطلب إلى المعلمات
        $params[':id'] = $orderId;

        $updateQuery = "UPDATE orders SET " . implode(', ', $updateFields) . ", updated_at = NOW() WHERE id = :id";

        $stmt = $conn->prepare($updateQuery);
        $stmt->execute($params);

        error_log("تم تنفيذ استعلام التحديث: " . $updateQuery);
        error_log("عدد الصفوف المتأثرة: " . $stmt->rowCount());

        // التحقق من نجاح التحديث
        if ($stmt->rowCount() === 0) {
            throw new Exception('لم يتم تحديث الطلب. قد يكون الطلب غير موجود أو لم يتم إجراء أي تغييرات.');
        }

        // تأكيد المعاملة
        $conn->commit();

        // إرجاع استجابة النجاح
        $response = [
            'status' => 'success',
            'message' => 'تم تحديث الطلب بنجاح',
            'orderId' => $orderId
        ];
    } else {
        // إنشاء طلب جديد
        // إنشاء رقم الطلب
        $orderNumber = $orderData['order_number'] ?? $orderData['orderNumber'] ?? null;

        if (empty($orderNumber)) {
            // إنشاء رقم طلب جديد
            if (!empty($branchId)) {
                // الحصول على رمز الفرع
                $stmt = $conn->prepare("SELECT code FROM branches WHERE id = ?");
                $stmt->execute([$branchId]);
                $branch = $stmt->fetch();

                if (!$branch) {
                    throw new Exception('الفرع غير موجود');
                }

                $branchCode = $branch['code'];
                $year = date('y');

                // الحصول على آخر رقم تسلسلي للفرع
                $stmt = $conn->prepare("SELECT last_sequence_number FROM order_sequences WHERE branch_code = ?");
                $stmt->execute([$branchCode]);
                $sequence = $stmt->fetch();

                $sequenceNumber = 1;
                if ($sequence) {
                    $sequenceNumber = $sequence['last_sequence_number'] + 1;

                    // تحديث رقم التسلسل
                    $stmt = $conn->prepare("UPDATE order_sequences SET last_sequence_number = ? WHERE branch_code = ?");
                    $stmt->execute([$sequenceNumber, $branchCode]);
                } else {
                    // إنشاء سجل جديد
                    $stmt = $conn->prepare("INSERT INTO order_sequences (branch_code, last_sequence_number) VALUES (?, ?)");
                    $stmt->execute([$branchCode, $sequenceNumber]);
                }

                // تنسيق رقم الطلب: رمز الفرع + السنة + رقم التسلسل (مثال: ADN23-01)
                $orderNumber = $branchCode . $year . '-' . str_pad($sequenceNumber, 2, '0', STR_PAD_LEFT);
            } else {
                // إذا لم يتم تحديد الفرع، استخدم رقم طلب افتراضي
                $orderNumber = 'ORD-' . date('ymd') . '-' . rand(1000, 9999);
            }
        }

        $data['order_number'] = $orderNumber;

        // إعداد استعلام الإدراج
        $fields = array_keys($data);
        $placeholders = array_map(function($field) { return ":$field"; }, $fields);

        $insertQuery = "INSERT INTO orders (" . implode(', ', $fields) . ") VALUES (" . implode(', ', $placeholders) . ")";

        $params = [];
        foreach ($data as $field => $value) {
            $params[":$field"] = $value;
        }

        $stmt = $conn->prepare($insertQuery);
        $stmt->execute($params);

        error_log("تم تنفيذ استعلام الإدراج: " . $insertQuery);

        // الحصول على معرف الطلب المدرج
        $newOrderId = $conn->lastInsertId();
        error_log("تم الحصول على معرف الطلب المدرج: " . $newOrderId);

        // تأكيد المعاملة
        $conn->commit();

        // إرجاع استجابة النجاح
        $response = [
            'status' => 'success',
            'message' => 'تم إنشاء الطلب بنجاح',
            'orderId' => $newOrderId,
            'orderNumber' => $orderNumber
        ];
    }

    error_log("إرسال استجابة نجاح: " . json_encode($response));

    header('Content-Type: application/json');
    echo json_encode($response);

} catch (PDOException $e) {
    // التراجع عن المعاملة في حالة حدوث خطأ
    $conn->rollBack();
    error_log("تم التراجع عن المعاملة بسبب خطأ: " . $e->getMessage());

    // إرجاع رسالة الخطأ
    $errorResponse = [
        'status' => 'error',
        'message' => 'فشل حفظ الطلب: ' . $e->getMessage(),
        'debug_info' => [
            'error_code' => $e->getCode(),
            'error_message' => $e->getMessage(),
            'error_trace' => $e->getTraceAsString()
        ]
    ];
    error_log("إرسال استجابة خطأ: " . json_encode($errorResponse));

    header('Content-Type: application/json');
    echo json_encode($errorResponse);
} catch (Exception $e) {
    // التراجع عن المعاملة في حالة حدوث خطأ
    $conn->rollBack();
    error_log("تم التراجع عن المعاملة بسبب خطأ: " . $e->getMessage());

    // إرجاع رسالة الخطأ
    $errorResponse = [
        'status' => 'error',
        'message' => $e->getMessage(),
        'debug_info' => [
            'error_code' => $e->getCode(),
            'error_message' => $e->getMessage(),
            'error_trace' => $e->getTraceAsString()
        ]
    ];
    error_log("إرسال استجابة خطأ: " . json_encode($errorResponse));

    header('Content-Type: application/json');
    echo json_encode($errorResponse);
}
?>
